package io.ultreia.java4all.bean.definition;

/*-
 * #%L
 * Java Beans extends by Ultreia.io
 * %%
 * Copyright (C) 2018 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import io.ultreia.java4all.bean.JavaBean;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Modifier;
import java.util.List;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public abstract class AbstractJavaBeanDefinition implements JavaBeanDefinition {

    private ImmutableSet<Class<?>> acceptedTypes;
    private ImmutableMap<String, JavaBeanPropertyDefinition<?, ?>> properties;
    private Class<?> mainType;

    @Override
    public final ImmutableSet<Class<?>> types() {
        return acceptedTypes == null ? acceptedTypes = Objects.requireNonNull(loadAcceptedTypes()) : acceptedTypes;
    }

    protected abstract ImmutableSet<Class<?>> loadAcceptedTypes();

    protected abstract ImmutableMap<String, Class<?>> loadTypes();

    protected abstract ImmutableMap<String, Function<?, ?>> loadGetters();

    protected abstract ImmutableMap<String, BiConsumer<?, ?>> loadSetters();

    @SuppressWarnings("unchecked")
    @Override
    public final ImmutableMap<String, JavaBeanPropertyDefinition<?, ?>> properties() {
        if (properties == null) {
            ImmutableMap<String, Function<?, ?>> getters = Objects.requireNonNull(loadGetters());
            ImmutableMap<String, BiConsumer<?, ?>> setters = Objects.requireNonNull(loadSetters());
            ImmutableMap<String, Class<?>> types = Objects.requireNonNull(loadTypes());
            List<String> allPropertyNames = Stream.concat(getters.keySet().stream(), setters.keySet().stream()).distinct().sorted(String::compareTo).collect(Collectors.toList());
            ImmutableMap.Builder<String, JavaBeanPropertyDefinition<?, ?>> propertiesBuilder = ImmutableMap.builder();
            for (String propertyName : allPropertyNames) {
                Class type = types.get(propertyName);
                Function getter = getters.get(propertyName);
                BiConsumer setter = setters.get(propertyName);
                propertiesBuilder.put(propertyName, new JavaBeanPropertyDefinition<>(propertyName, type, getter, setter));
            }
            this.properties = propertiesBuilder.build();
        }
        return properties;
    }

    @SuppressWarnings("unchecked")
    @Override
    public final <O extends JavaBean> O newInstance() {
        try {
            return (O) mainType().getDeclaredConstructor().newInstance();
        } catch (InstantiationException | IllegalAccessException | NoSuchMethodException | InvocationTargetException e) {
            throw new IllegalStateException("Can't instantiate with: " + mainType, e.getCause());
        }
    }

    private Class<?> mainType() {
        if (mainType == null) {
            for (Class<?> acceptedType : acceptedTypes) {
                if (acceptedType.isInterface() || Modifier.isAbstract(acceptedType.getModifiers())) {
                    continue;
                }
                mainType = acceptedType;
            }
            Objects.requireNonNull(mainType, "Can't find a type to instantiate in: " + acceptedTypes);
        }
        return mainType;
    }
}
