/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sshd.client;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.Writer;
import java.net.ConnectException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.channels.Channel;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import org.apache.sshd.client.SshClient;
import org.apache.sshd.client.future.ConnectFuture;
import org.apache.sshd.client.keyverifier.ServerKeyVerifier;
import org.apache.sshd.client.session.ClientSession;
import org.apache.sshd.common.NamedFactory;
import org.apache.sshd.common.cipher.ECCurves;
import org.apache.sshd.common.config.keys.KeyUtils;
import org.apache.sshd.common.config.keys.PublicKeyEntry;
import org.apache.sshd.common.io.IoSession;
import org.apache.sshd.common.kex.KexProposalOption;
import org.apache.sshd.common.session.Session;
import org.apache.sshd.common.session.SessionListener;
import org.apache.sshd.common.signature.BuiltinSignatures;
import org.apache.sshd.common.signature.Signature;
import org.apache.sshd.common.util.GenericUtils;
import org.apache.sshd.common.util.SecurityUtils;
import org.apache.sshd.common.util.ValidateUtils;
import org.apache.sshd.common.util.io.IoUtils;
import org.apache.sshd.common.util.io.NoCloseInputStream;
import org.apache.sshd.common.util.logging.AbstractSimplifiedLog;
import org.apache.sshd.common.util.logging.LoggingUtils;
import org.apache.sshd.common.util.net.SshdSocketAddress;

public class SshKeyScan
extends AbstractSimplifiedLog
implements Channel,
Callable<Void>,
ServerKeyVerifier,
SessionListener {
    public static final List<String> DEFAULT_KEY_TYPES = Collections.unmodifiableList(Arrays.asList("RSA", "ECDSA"));
    public static final long DEFAULT_TIMEOUT = TimeUnit.SECONDS.toMillis(5L);
    public static final Level DEFAULT_LEVEL = Level.INFO;
    private final AtomicBoolean open = new AtomicBoolean(true);
    private SshClient client;
    private int port;
    private long timeout;
    private List<String> keyTypes;
    private InputStream input;
    private Level level;
    private final Map<String, String> currentHostFingerprints = new TreeMap<String, String>(String.CASE_INSENSITIVE_ORDER);

    public int getPort() {
        return this.port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    public InputStream getInputStream() {
        return this.input;
    }

    public void setInputStream(InputStream input) {
        this.input = input;
    }

    public List<String> getKeyTypes() {
        return this.keyTypes;
    }

    public void setKeyTypes(List<String> keyTypes) {
        this.keyTypes = keyTypes;
    }

    public long getTimeout() {
        return this.timeout;
    }

    public void setTimeout(long timeout) {
        this.timeout = timeout;
    }

    public Level getLogLevel() {
        return this.level;
    }

    public void setLogLevel(Level level) {
        this.level = level;
    }

    @Override
    public void log(Level level, Object message, Throwable t) {
        if (this.isEnabled(level)) {
            PrintStream ps = System.out;
            if (t != null || Level.SEVERE.equals(level) || Level.WARNING.equals(level)) {
                ps = System.err;
            }
            ps.append('\t').println(message);
            if (t != null) {
                ps.append("\t\t").append(t.getClass().getSimpleName()).append(": ").println(t.getMessage());
            }
        }
    }

    @Override
    public boolean isEnabled(Level level) {
        return LoggingUtils.isLoggable(level, this.getLogLevel());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Void call() throws Exception {
        ValidateUtils.checkTrue(this.isOpen(), "Scanner is closed");
        List<String> typeNames = this.getKeyTypes();
        Map<String, List<KeyPair>> pairsMap = this.createKeyPairs(typeNames);
        TreeMap<String, List<NamedFactory<Signature>>> sigFactories = new TreeMap<String, List<NamedFactory<Signature>>>(String.CASE_INSENSITIVE_ORDER);
        for (String kt : new TreeSet<String>(pairsMap.keySet())) {
            List<NamedFactory<Signature>> factories = this.resolveSignatureFactories(kt);
            if (GenericUtils.isEmpty(factories)) {
                if (this.isEnabled(Level.FINEST)) {
                    this.log(Level.FINEST, "Skip empty signature factories for " + kt);
                }
                pairsMap.remove(kt);
                continue;
            }
            sigFactories.put(kt, factories);
        }
        ValidateUtils.checkTrue(!GenericUtils.isEmpty(pairsMap), "No client key pairs");
        ValidateUtils.checkTrue(!GenericUtils.isEmpty(sigFactories), "No signature factories");
        Exception err = null;
        try {
            ValidateUtils.checkTrue(this.client == null, "Client still active");
            this.client = SshClient.setUpDefaultClient();
            this.client.setServerKeyVerifier(this);
            try (BufferedReader rdr = new BufferedReader(new InputStreamReader(this.getInputStream(), StandardCharsets.UTF_8));){
                this.client.start();
                String line = rdr.readLine();
                while (line != null) {
                    String[] hosts = GenericUtils.split(GenericUtils.trimToEmpty(line), ',');
                    if (!GenericUtils.isEmpty(hosts)) {
                        for (String h : hosts) {
                            if (!this.isOpen()) {
                                throw new InterruptedIOException("Closed while preparing to contact host=" + h);
                            }
                            try {
                                this.resolveServerKeys(this.client, h, pairsMap, sigFactories);
                            }
                            catch (Exception e) {
                                if (e instanceof InterruptedIOException) {
                                    throw e;
                                }
                                if (this.isEnabled(Level.FINE)) {
                                    this.log(Level.FINE, "Failed to retrieve keys from " + h, e);
                                }
                                err = GenericUtils.accumulateException(err, e);
                            }
                            finally {
                                this.currentHostFingerprints.clear();
                            }
                        }
                    }
                    line = rdr.readLine();
                }
            }
        }
        finally {
            try {
                this.close();
            }
            catch (IOException e) {
                err = GenericUtils.accumulateException(err, e);
            }
        }
        if (err != null) {
            throw err;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void resolveServerKeys(SshClient client, String host, Map<String, List<KeyPair>> pairsMap, Map<String, List<NamedFactory<Signature>>> sigFactories) throws IOException {
        for (Map.Entry<String, List<KeyPair>> pe : pairsMap.entrySet()) {
            String kt = pe.getKey();
            if (!this.isOpen()) {
                throw new InterruptedIOException("Closed while attempting to retrieve key type=" + kt + " from " + host);
            }
            List<NamedFactory<Signature>> current = client.getSignatureFactories();
            try {
                List<NamedFactory<Signature>> forced = sigFactories.get(kt);
                client.setSignatureFactories(forced);
                this.resolveServerKeys(client, host, kt, pe.getValue());
            }
            catch (Exception e) {
                if (this.isEnabled(Level.FINE)) {
                    this.log(Level.FINE, "Failed to resolve key=" + kt + " for " + host);
                }
                if (!(e instanceof ConnectException)) continue;
                return;
            }
            finally {
                client.setSignatureFactories(current);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void resolveServerKeys(SshClient client, String host, String kt, List<KeyPair> ids) throws Exception {
        long waitTime;
        ConnectFuture future;
        int connectPort = this.getPort();
        if (this.isEnabled(Level.FINE)) {
            this.log(Level.FINE, "Connecting to " + host + ":" + connectPort + " to retrieve key type=" + kt);
        }
        if (!(future = client.connect(UUID.randomUUID().toString(), host, connectPort)).await(waitTime = this.getTimeout())) {
            throw new ConnectException("Failed to connect to " + host + ":" + connectPort + " within " + waitTime + " msec." + " to retrieve key type=" + kt);
        }
        try (ClientSession session = future.getSession();){
            IoSession ioSession = session.getIoSession();
            SocketAddress remoteAddress = ioSession.getRemoteAddress();
            String remoteLocation = SshKeyScan.toString(remoteAddress);
            if (this.isEnabled(Level.FINE)) {
                this.log(Level.FINE, "Connected to " + remoteLocation + " to retrieve key type=" + kt);
            }
            try {
                session.addSessionListener(this);
                if (this.isEnabled(Level.FINER)) {
                    this.log(Level.FINER, "Authenticating with key type=" + kt + " to " + remoteLocation);
                }
                for (KeyPair kp : ids) {
                    session.addPublicKeyIdentity(kp);
                }
                try {
                    session.auth().verify(waitTime);
                    this.log(Level.WARNING, "Unexpected authentication success using key type=" + kt + " with " + remoteLocation);
                }
                catch (Exception e) {
                    if (this.isEnabled(Level.FINER)) {
                        this.log(Level.FINER, "Failed to authenticate using key type=" + kt + " with " + remoteLocation);
                    }
                }
                finally {
                    for (KeyPair kp : ids) {
                        session.removePublicKeyIdentity(kp);
                    }
                }
            }
            finally {
                session.removeSessionListener(this);
            }
        }
    }

    @Override
    public void sessionCreated(Session session) {
        this.logSessionEvent(session, "Created");
    }

    @Override
    public void sessionEvent(Session session, SessionListener.Event event) {
        this.logSessionEvent(session, (Object)event);
        if (this.isEnabled(Level.FINEST) && SessionListener.Event.KexCompleted.equals((Object)event)) {
            IoSession ioSession = session.getIoSession();
            SocketAddress remoteAddress = ioSession.getRemoteAddress();
            String remoteLocation = SshKeyScan.toString(remoteAddress);
            for (KexProposalOption paramType : KexProposalOption.VALUES) {
                String paramValue = session.getNegotiatedKexParameter(paramType);
                this.log(Level.FINEST, remoteLocation + "[" + paramType.getDescription() + "]: " + paramValue);
            }
        }
    }

    @Override
    public void sessionException(Session session, Throwable t) {
        this.logSessionEvent(session, t);
    }

    @Override
    public void sessionClosed(Session session) {
        this.logSessionEvent(session, "Closed");
    }

    protected void logSessionEvent(Session session, Object event) {
        if (this.isEnabled(Level.FINEST)) {
            IoSession ioSession = session.getIoSession();
            SocketAddress remoteAddress = ioSession.getRemoteAddress();
            this.log(Level.FINEST, "Session " + SshKeyScan.toString(remoteAddress) + " event: " + event);
        }
    }

    @Override
    public boolean verifyServerKey(ClientSession sshClientSession, SocketAddress remoteAddress, PublicKey serverKey) {
        String remoteLocation = SshKeyScan.toString(remoteAddress);
        String extra = KeyUtils.getFingerPrint(serverKey);
        try {
            String current;
            String keyType = KeyUtils.getKeyType(serverKey);
            String string = current = GenericUtils.isEmpty(keyType) ? null : this.currentHostFingerprints.get(keyType);
            if (Objects.equals(current, extra)) {
                if (this.isEnabled(Level.FINER)) {
                    this.log(Level.FINER, "verifyServerKey(" + remoteLocation + ")[" + keyType + "] skip existing key: " + extra);
                }
            } else {
                if (this.isEnabled(Level.FINE)) {
                    this.log(Level.FINE, "verifyServerKey(" + remoteLocation + ")[" + keyType + "] found new key: " + extra);
                }
                this.writeServerKey(remoteLocation, keyType, serverKey);
                if (!GenericUtils.isEmpty(keyType)) {
                    this.currentHostFingerprints.put(keyType, extra);
                }
            }
        }
        catch (Exception e) {
            this.log(Level.SEVERE, "Failed to output the public key " + extra + " from " + remoteLocation, e);
        }
        return true;
    }

    protected void writeServerKey(String remoteLocation, String keyType, PublicKey serverKey) throws Exception {
        StringBuilder sb = new StringBuilder(256).append(remoteLocation).append(' ');
        PublicKeyEntry.appendPublicKeyEntry(sb, serverKey);
        this.log(Level.INFO, sb);
    }

    private static String toString(SocketAddress addr) {
        if (addr == null) {
            return null;
        }
        if (addr instanceof InetSocketAddress) {
            return ((InetSocketAddress)addr).getHostString();
        }
        if (addr instanceof SshdSocketAddress) {
            return ((SshdSocketAddress)addr).getHostName();
        }
        return addr.toString();
    }

    protected List<NamedFactory<Signature>> resolveSignatureFactories(String keyType) throws GeneralSecurityException {
        if (this.isEnabled(Level.FINE)) {
            this.log(Level.FINE, "Resolve signature factories for " + keyType);
        }
        if ("RSA".equalsIgnoreCase(keyType)) {
            return Collections.singletonList(BuiltinSignatures.rsa);
        }
        if ("DSA".equalsIgnoreCase(keyType)) {
            return Collections.singletonList(BuiltinSignatures.dsa);
        }
        if ("ECDSA".equalsIgnoreCase(keyType)) {
            ArrayList<NamedFactory<Signature>> factories = new ArrayList<NamedFactory<Signature>>(ECCurves.NAMES.size());
            for (String n : ECCurves.NAMES) {
                if (this.isEnabled(Level.FINER)) {
                    this.log(Level.FINER, "Resolve signature factory for curve=" + n);
                }
                NamedFactory f = ValidateUtils.checkNotNull(BuiltinSignatures.fromString(n), "Unknown curve signature: %s", (Object)n);
                factories.add(f);
            }
            return factories;
        }
        throw new InvalidKeySpecException("Unknown key type: " + keyType);
    }

    protected Map<String, List<KeyPair>> createKeyPairs(Collection<String> typeNames) throws GeneralSecurityException {
        if (GenericUtils.isEmpty(typeNames)) {
            return Collections.emptyMap();
        }
        TreeMap<String, List<KeyPair>> pairsMap = new TreeMap<String, List<KeyPair>>(String.CASE_INSENSITIVE_ORDER);
        for (String kt : typeNames) {
            if (pairsMap.containsKey(kt)) {
                this.log(Level.WARNING, "Key type " + kt + " re-specified");
                continue;
            }
            List<KeyPair> kps = this.createKeyPairs(kt);
            if (GenericUtils.isEmpty(kps)) {
                this.log(Level.WARNING, "No key-pairs generated for key type " + kt);
                continue;
            }
            pairsMap.put(kt, kps);
        }
        return pairsMap;
    }

    protected List<KeyPair> createKeyPairs(String keyType) throws GeneralSecurityException {
        if (this.isEnabled(Level.FINE)) {
            this.log(Level.FINE, "Generate key pairs for " + keyType);
        }
        if ("RSA".equalsIgnoreCase(keyType)) {
            return Collections.singletonList(KeyUtils.generateKeyPair("ssh-rsa", 1024));
        }
        if ("DSA".equalsIgnoreCase(keyType)) {
            return Collections.singletonList(KeyUtils.generateKeyPair("ssh-dss", 512));
        }
        if ("ECDSA".equalsIgnoreCase(keyType)) {
            if (!SecurityUtils.hasEcc()) {
                throw new InvalidKeySpecException("ECC not supported");
            }
            ArrayList<KeyPair> kps = new ArrayList<KeyPair>(ECCurves.NAMES.size());
            for (ECCurves curve : ECCurves.VALUES) {
                String curveName = curve.getName();
                if (this.isEnabled(Level.FINER)) {
                    this.log(Level.FINER, "Generate key pair for curve=" + curveName);
                }
                kps.add(KeyUtils.generateKeyPair(curve.getKeyType(), curve.getKeySize()));
            }
            return kps;
        }
        throw new InvalidKeySpecException("Unknown key type: " + keyType);
    }

    @Override
    public boolean isOpen() {
        return this.open.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        if (!this.open.getAndSet(false)) {
            return;
        }
        IOException err = null;
        if (this.input != null) {
            try {
                this.input.close();
            }
            catch (IOException e) {
                err = GenericUtils.accumulateException(err, e);
            }
            finally {
                this.input = null;
            }
        }
        if (this.client != null) {
            try {
                this.client.close();
            }
            catch (IOException e) {
                err = GenericUtils.accumulateException(err, e);
            }
            finally {
                try {
                    this.client.stop();
                }
                finally {
                    this.client = null;
                }
            }
        }
        if (err != null) {
            throw err;
        }
    }

    public static List<String> parseCommandLineArguments(SshKeyScan scanner, String ... args) throws IOException {
        int numArgs = GenericUtils.length(args);
        for (int index = 0; index < numArgs; ++index) {
            String optName = args[index];
            if ("-f".equals(optName)) {
                ValidateUtils.checkTrue(++index < numArgs, "Missing %s option argument", (Object)optName);
                ValidateUtils.checkTrue(scanner.getInputStream() == null, "%s option re-specified", (Object)optName);
                String filePath = args[index];
                if ("-".equals(filePath)) {
                    scanner.setInputStream(new NoCloseInputStream(System.in));
                    continue;
                }
                scanner.setInputStream(new FileInputStream(filePath));
                continue;
            }
            if ("-t".equals(optName)) {
                ValidateUtils.checkTrue(++index < numArgs, "Missing %s option argument", (Object)optName);
                ValidateUtils.checkTrue(GenericUtils.isEmpty(scanner.getKeyTypes()), "%s option re-specified", (Object)optName);
                String typeList = args[index];
                String[] types = GenericUtils.split(typeList, ',');
                ValidateUtils.checkTrue(GenericUtils.length(types) > 0, "No types specified for %s", (Object)optName);
                scanner.setKeyTypes(Arrays.asList(types));
                continue;
            }
            if ("-p".equals(optName)) {
                ValidateUtils.checkTrue(++index < numArgs, "Missing %s option argument", (Object)optName);
                ValidateUtils.checkTrue(scanner.getPort() <= 0, "%s option re-specified", (Object)optName);
                String portValue = args[index];
                int port = Integer.parseInt(portValue);
                ValidateUtils.checkTrue(port > 0 && port <= 65535, "Bad port: %s", (Object)portValue);
                scanner.setPort(port);
                continue;
            }
            if ("-T".equals(optName)) {
                ValidateUtils.checkTrue(++index < numArgs, "Missing %s option argument", (Object)optName);
                ValidateUtils.checkTrue(scanner.getTimeout() <= 0L, "%s option re-specified", (Object)optName);
                String timeoutValue = args[index];
                long timeout = Long.parseLong(timeoutValue);
                ValidateUtils.checkTrue(timeout > 0L, "Bad timeout: %s", (Object)timeoutValue);
                scanner.setTimeout(timeout);
                continue;
            }
            if ("-v".equals(optName)) {
                ValidateUtils.checkTrue(scanner.getLogLevel() == null, "%s option re-specified", (Object)optName);
                scanner.setLogLevel(Level.FINEST);
                continue;
            }
            ValidateUtils.checkTrue(optName.charAt(0) != '-', "Unknown option: %s", (Object)optName);
            int remaining = numArgs - index;
            if (remaining == 1) {
                return Collections.singletonList(optName);
            }
            ArrayList<String> hosts = new ArrayList<String>(remaining);
            while (index < numArgs) {
                hosts.add(args[index]);
                ++index;
            }
            return hosts;
        }
        return Collections.emptyList();
    }

    public static <S extends SshKeyScan> S setInputStream(S scanner, Collection<String> hosts) throws IOException {
        if (GenericUtils.isEmpty(hosts)) {
            ValidateUtils.checkNotNull(scanner.getInputStream(), "No hosts or file specified");
        } else {
            ValidateUtils.checkTrue(scanner.getInputStream() == null, "Both hosts and file specified");
            try (ByteArrayOutputStream baos = new ByteArrayOutputStream(hosts.size() * 32);){
                try (OutputStreamWriter w = new OutputStreamWriter((OutputStream)baos, StandardCharsets.UTF_8);){
                    for (String h : hosts) {
                        ((Writer)w).append(h).append(IoUtils.EOL);
                    }
                }
                byte[] data = baos.toByteArray();
                scanner.setInputStream(new ByteArrayInputStream(data));
            }
        }
        return scanner;
    }

    public static <S extends SshKeyScan> S initializeScanner(S scanner, Collection<String> hosts) throws IOException {
        SshKeyScan.setInputStream(scanner, hosts);
        if (scanner.getPort() <= 0) {
            scanner.setPort(22);
        }
        if (scanner.getTimeout() <= 0L) {
            scanner.setTimeout(DEFAULT_TIMEOUT);
        }
        if (GenericUtils.isEmpty(scanner.getKeyTypes())) {
            scanner.setKeyTypes(DEFAULT_KEY_TYPES);
        }
        if (scanner.getLogLevel() == null) {
            scanner.setLogLevel(DEFAULT_LEVEL);
        }
        return scanner;
    }

    public static void main(String[] args) throws Exception {
        try (SshKeyScan scanner = new SshKeyScan();){
            List<String> hosts = SshKeyScan.parseCommandLineArguments(scanner, args);
            SshKeyScan.initializeScanner(scanner, hosts);
            scanner.call();
        }
    }
}

