/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.servicediscovery.kubernetes;

import io.vertx.core.AsyncResult;
import io.vertx.core.CompositeFuture;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import io.vertx.core.parsetools.JsonParser;
import io.vertx.core.streams.WriteStream;
import io.vertx.ext.web.client.HttpResponse;
import io.vertx.ext.web.client.WebClient;
import io.vertx.ext.web.client.WebClientOptions;
import io.vertx.ext.web.codec.BodyCodec;
import io.vertx.servicediscovery.Record;
import io.vertx.servicediscovery.spi.ServiceImporter;
import io.vertx.servicediscovery.spi.ServicePublisher;
import io.vertx.servicediscovery.types.HttpLocation;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

public class KubernetesServiceImporter
implements ServiceImporter {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)KubernetesServiceImporter.class.getName());
    public static final String KUBERNETES_UUID = "kubernetes.uuid";
    private ServicePublisher publisher;
    private String namespace;
    private List<Record> records = new CopyOnWriteArrayList<Record>();
    private WebClient client;
    private static final String OPENSHIFT_KUBERNETES_TOKEN_FILE = "/var/run/secrets/kubernetes.io/serviceaccount/token";

    public void start(Vertx vertx, ServicePublisher publisher, JsonObject configuration, Future<Void> completion) {
        String p;
        this.publisher = publisher;
        JsonObject conf = configuration == null ? new JsonObject() : configuration;
        int port = conf.getInteger("port", Integer.valueOf(0));
        if (port == 0) {
            port = conf.getBoolean("ssl", Boolean.valueOf(true)) != false ? 443 : 80;
        }
        if ((p = System.getenv("KUBERNETES_SERVICE_PORT")) != null) {
            port = Integer.valueOf(p);
        }
        String host = conf.getString("host");
        String h = System.getenv("KUBERNETES_SERVICE_HOST");
        if (h != null) {
            host = h;
        }
        this.client = WebClient.create((Vertx)vertx, (WebClientOptions)new WebClientOptions().setTrustAll(true).setSsl(conf.getBoolean("ssl", Boolean.valueOf(true)).booleanValue()).setDefaultHost(host).setDefaultPort(port).setFollowRedirects(true));
        Future<String> retrieveTokenFuture = this.getToken(vertx, conf);
        this.namespace = conf.getString("namespace", this.getNamespaceOrDefault());
        LOGGER.info((Object)("Kubernetes discovery configured for namespace: " + this.namespace));
        LOGGER.info((Object)("Kubernetes master url: http" + (conf.getBoolean("ssl", Boolean.valueOf(true)) != false ? "s" : "") + "//" + host + ":" + port));
        retrieveTokenFuture.map(t -> {
            LOGGER.info((Object)("Kubernetes discovery: Bearer Token { " + t + " }"));
            return t;
        }).compose(this::retrieveServices).map(list -> {
            LOGGER.info((Object)("Kubernetes initial import of " + list.size() + " services"));
            ArrayList publications = new ArrayList();
            list.forEach(s -> {
                JsonObject svc = (JsonObject)s;
                Record record = KubernetesServiceImporter.createRecord(svc);
                if (this.addRecordIfNotContained(record)) {
                    Future fut = Future.future();
                    this.publishRecord(record, (Handler<AsyncResult<Record>>)fut.completer());
                    publications.add(fut);
                }
            });
            return publications;
        }).compose(CompositeFuture::all).setHandler(ar -> {
            if (ar.succeeded()) {
                LOGGER.info((Object)("Kubernetes importer instantiated with " + this.records.size() + " services imported"));
                completion.complete();
            } else {
                LOGGER.error((Object)"Error while interacting with kubernetes", ar.cause());
                completion.fail(ar.cause());
            }
        });
    }

    private Future<JsonArray> retrieveServices(String token) {
        Future future = Future.future();
        String path = "/api/v1/namespaces/" + this.namespace + "/services";
        this.client.get(path).putHeader("Authorization", "Bearer " + token).send(resp -> {
            if (resp.failed()) {
                future.fail(resp.cause());
            } else if (((HttpResponse)resp.result()).statusCode() != 200) {
                future.fail("Unable to retrieve services from namespace " + this.namespace + ", status code: " + ((HttpResponse)resp.result()).statusCode() + ", content: " + ((HttpResponse)resp.result()).bodyAsString());
            } else {
                HttpResponse response = (HttpResponse)resp.result();
                JsonArray items = response.bodyAsJsonObject().getJsonArray("items");
                if (items == null) {
                    future.fail("Unable to retrieve services from namespace " + this.namespace + " - no items");
                } else {
                    future.complete((Object)items);
                    this.watch(this.client, token, response.bodyAsJsonObject().getJsonObject("metadata").getString("resourceVersion"));
                }
            }
        });
        return future;
    }

    private void watch(WebClient client, String token, String resourceVersion) {
        String path = "/api/v1/namespaces/" + this.namespace + "/services";
        final JsonParser parser = JsonParser.newParser().objectValueMode().handler(event -> this.onChunk(event.objectValue()));
        client.get(path).addQueryParam("watch", "true").addQueryParam("resourceVersion", resourceVersion).as(BodyCodec.pipe((WriteStream)new WriteStream<Buffer>(){

            public WriteStream<Buffer> exceptionHandler(Handler<Throwable> handler) {
                return this;
            }

            public WriteStream<Buffer> write(Buffer data) {
                parser.write(data);
                return this;
            }

            public void end() {
                parser.end();
            }

            public WriteStream<Buffer> setWriteQueueMaxSize(int maxSize) {
                return this;
            }

            public boolean writeQueueFull() {
                return false;
            }

            public WriteStream<Buffer> drainHandler(Handler<Void> handler) {
                return this;
            }
        })).putHeader("Authorization", "Bearer " + token).send(ar -> {
            if (ar.failed()) {
                LOGGER.error((Object)"Unable to setup the watcher on the service list", ar.cause());
            } else {
                LOGGER.info((Object)("Watching services from namespace " + this.namespace));
            }
        });
    }

    private void onChunk(JsonObject json) {
        String type = json.getString("type");
        if (type == null) {
            return;
        }
        JsonObject service = json.getJsonObject("object");
        switch (type) {
            case "ADDED": {
                Record record = KubernetesServiceImporter.createRecord(service);
                if (!this.addRecordIfNotContained(record)) break;
                LOGGER.info((Object)("Adding service " + record.getName()));
                this.publishRecord(record, null);
                break;
            }
            case "DELETED": 
            case "ERROR": {
                Record record = KubernetesServiceImporter.createRecord(service);
                LOGGER.info((Object)("Removing service " + record.getName()));
                Record storedRecord = this.removeRecordIfContained(record);
                if (storedRecord == null) break;
                this.unpublishRecord(storedRecord, null);
                break;
            }
            case "MODIFIED": {
                Record record = KubernetesServiceImporter.createRecord(service);
                LOGGER.info((Object)("Modifying service " + record.getName()));
                Record storedRecord = this.replaceRecordIfContained(record);
                if (storedRecord == null) break;
                this.unpublishRecord(storedRecord, (Handler<Void>)((Handler)x -> this.publishRecord(record, null)));
            }
        }
    }

    private Future<String> getToken(Vertx vertx, JsonObject conf) {
        Future result = Future.future();
        String token = conf.getString("token");
        if (token != null && !token.trim().isEmpty()) {
            result.complete((Object)token);
            return result;
        }
        vertx.fileSystem().readFile(OPENSHIFT_KUBERNETES_TOKEN_FILE, ar -> {
            if (ar.failed()) {
                result.tryFail(ar.cause());
            } else {
                result.tryComplete((Object)((Buffer)ar.result()).toString());
            }
        });
        return result;
    }

    private void publishRecord(Record record, Handler<AsyncResult<Record>> completionHandler) {
        this.publisher.publish(record, ar -> {
            if (completionHandler != null) {
                completionHandler.handle(ar);
            }
            if (ar.succeeded()) {
                LOGGER.info((Object)("Kubernetes service published in the vert.x service registry: " + record.toJson()));
            } else {
                LOGGER.error((Object)"Kubernetes service not published in the vert.x service registry", ar.cause());
            }
        });
    }

    private synchronized boolean addRecordIfNotContained(Record record) {
        for (Record rec : this.records) {
            if (!this.areTheSameService(rec, record)) continue;
            return false;
        }
        return this.records.add(record);
    }

    private String getNamespaceOrDefault() {
        String ns = System.getenv("KUBERNETES_NAMESPACE");
        if (ns == null && (ns = System.getenv("OPENSHIFT_BUILD_NAMESPACE")) == null) {
            ns = "default";
        }
        return ns;
    }

    private boolean areTheSameService(Record record1, Record record2) {
        String uuid = record1.getMetadata().getString(KUBERNETES_UUID, "");
        String uuid2 = record2.getMetadata().getString(KUBERNETES_UUID, "");
        String endpoint = record1.getLocation().getString("endpoint", "");
        String endpoint2 = record2.getLocation().getString("endpoint", "");
        return uuid.equals(uuid2) && endpoint.equals(endpoint2);
    }

    static Record createRecord(JsonObject service) {
        JsonObject metadata = service.getJsonObject("metadata");
        Record record = new Record().setName(metadata.getString("name"));
        JsonObject labels = metadata.getJsonObject("labels");
        if (labels != null) {
            labels.forEach(entry -> record.getMetadata().put((String)entry.getKey(), entry.getValue().toString()));
        }
        record.getMetadata().put("kubernetes.namespace", metadata.getString("namespace"));
        record.getMetadata().put("kubernetes.name", metadata.getString("name"));
        record.getMetadata().put(KUBERNETES_UUID, metadata.getString("uid"));
        String type = record.getMetadata().getString("service-type");
        if (type == null) {
            type = KubernetesServiceImporter.discoveryType(service, record);
        }
        switch (type) {
            case "http-endpoint": {
                KubernetesServiceImporter.manageHttpService(record, service);
                break;
            }
            default: {
                KubernetesServiceImporter.manageUnknownService(record, service, type);
            }
        }
        return record;
    }

    static String discoveryType(JsonObject service, Record record) {
        JsonObject port;
        int p;
        JsonObject spec = service.getJsonObject("spec");
        JsonArray ports = spec.getJsonArray("ports");
        if (ports == null || ports.isEmpty()) {
            return "unknown";
        }
        if (ports.size() > 1) {
            LOGGER.warn((Object)("More than one ports has been found for " + record.getName() + " - taking the first one to build the record location"));
        }
        if ((p = (port = ports.getJsonObject(0)).getInteger("port").intValue()) == 80 || p == 443 || p >= 8080 && p <= 9000) {
            return "http-endpoint";
        }
        if (p == 5432 || p == 5433) {
            return "jdbc";
        }
        if (p == 3306 || p == 13306) {
            return "jdbc";
        }
        if (p == 6379) {
            return "redis";
        }
        if (p == 27017 || p == 27018 || p == 27019) {
            return "mongo";
        }
        return "unknown";
    }

    private static void manageUnknownService(Record record, JsonObject service, String type) {
        JsonObject location;
        JsonObject spec = service.getJsonObject("spec");
        JsonArray ports = spec.getJsonArray("ports");
        if (ports != null && !ports.isEmpty()) {
            if (ports.size() > 1) {
                LOGGER.warn((Object)("More than one ports has been found for " + record.getName() + " - taking the first one to build the record location"));
            }
            JsonObject port = ports.getJsonObject(0);
            location = port.copy();
            if (KubernetesServiceImporter.isExternalService(service)) {
                location.put("host", spec.getString("externalName"));
            } else {
                Object targetPort = port.getValue("targetPort");
                if (targetPort instanceof Integer) {
                    location.put("internal-port", (Integer)targetPort);
                }
                location.put("host", spec.getString("clusterIP"));
            }
        } else {
            throw new IllegalStateException("Cannot extract the location from the service " + record + " - no port");
        }
        record.setLocation(location).setType(type);
    }

    private static void manageHttpService(Record record, JsonObject service) {
        HttpLocation location;
        JsonObject spec = service.getJsonObject("spec");
        JsonArray ports = spec.getJsonArray("ports");
        if (ports != null && !ports.isEmpty()) {
            if (ports.size() > 1) {
                LOGGER.warn((Object)("More than one port has been found for " + record.getName() + " - taking the first one to extract the HTTP endpoint location"));
            }
            JsonObject port = ports.getJsonObject(0);
            Integer p = port.getInteger("port");
            record.setType("http-endpoint");
            location = new HttpLocation(port.copy());
            if (KubernetesServiceImporter.isExternalService(service)) {
                location.setHost(spec.getString("externalName"));
            } else {
                location.setHost(spec.getString("clusterIP"));
            }
            if (KubernetesServiceImporter.isTrue(record.getMetadata().getString("ssl")) || p != null && p == 443) {
                location.setSsl(true);
            }
        } else {
            throw new IllegalStateException("Cannot extract the HTTP URL from the service " + record + " - no port");
        }
        record.setLocation(location.toJson());
    }

    private static boolean isExternalService(JsonObject service) {
        return service.containsKey("spec") && service.getJsonObject("spec").containsKey("type") && service.getJsonObject("spec").getString("type").equals("ExternalName");
    }

    private static boolean isTrue(String ssl) {
        return "true".equalsIgnoreCase(ssl);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close(Handler<Void> completionHandler) {
        KubernetesServiceImporter kubernetesServiceImporter = this;
        synchronized (kubernetesServiceImporter) {
            if (this.client != null) {
                this.client.close();
                this.client = null;
            }
        }
        if (completionHandler != null) {
            completionHandler.handle(null);
        }
    }

    private void unpublishRecord(Record record, Handler<Void> completionHandler) {
        this.publisher.unpublish(record.getRegistration(), ar -> {
            if (ar.failed()) {
                LOGGER.error((Object)"Cannot unregister kubernetes service", ar.cause());
            } else {
                LOGGER.info((Object)("Kubernetes service unregistered from the vert.x registry: " + record.toJson()));
                if (completionHandler != null) {
                    completionHandler.handle(null);
                }
            }
        });
    }

    private Record removeRecordIfContained(Record record) {
        for (Record rec : this.records) {
            if (!this.areTheSameService(rec, record)) continue;
            this.records.remove(rec);
            return rec;
        }
        return null;
    }

    private Record replaceRecordIfContained(Record record) {
        for (Record rec : this.records) {
            if (!this.areTheSameService(rec, record)) continue;
            this.records.remove(rec);
            this.records.add(record);
            return rec;
        }
        return null;
    }
}

