/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.stack;

import com.jayway.awaitility.Awaitility;
import io.vertx.stack.model.Dependency;
import io.vertx.stack.model.DependencyConflictException;
import io.vertx.stack.model.Stack;
import io.vertx.stack.model.StackResolution;
import io.vertx.stack.model.StackResolutionOptions;
import io.vertx.stack.utils.FileUtils;
import io.vertx.stack.utils.LocalArtifact;
import io.vertx.stack.utils.LocalDependency;
import io.vertx.stack.utils.LocalRepoBuilder;
import java.io.File;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.maven.model.Exclusion;
import org.assertj.core.api.Assertions;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public class StackResolutionTest {
    private File root = new File("target/stack");
    private static final StackResolutionOptions STRICT = new StackResolutionOptions().setFailOnConflicts(true);

    @Before
    public void setUp() {
        FileUtils.delete(this.root);
        Awaitility.await().atMost(10L, TimeUnit.SECONDS).until(() -> !this.root.exists());
    }

    @After
    public void tearDown() {
        System.clearProperty("vertx.version");
    }

    @Test
    public void testTheResolutionOfAVerySmallStack() {
        Stack stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0"));
        StackResolution resolution = new StackResolution(stack, this.root, STRICT);
        Map map = resolution.resolve();
        Assertions.assertThat((Map)map).containsKey((Object)"io.vertx:vertx-core:jar:3.1.0");
    }

    @Test
    public void testTheResolutionOfAVerySmallStackWithFiltering() {
        Stack stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "${vertx.version}")).addVariable("vertx.version", "3.1.0");
        StackResolution resolution = new StackResolution(stack, this.root, STRICT);
        Map map = resolution.resolve();
        Assertions.assertThat((Map)map).containsKey((Object)"io.vertx:vertx-core:jar:3.1.0");
    }

    @Test
    public void testTheResolutionOfVertxCoreWithoutTransitive() {
        Stack stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0").setTransitive(false));
        StackResolution resolution = new StackResolution(stack, this.root, STRICT);
        Map map = resolution.resolve();
        Assertions.assertThat((Map)map).containsKeys((Object[])new String[]{"io.vertx:vertx-core:jar:3.1.0"}).hasSize(1);
    }

    @Test
    public void testNoConflictWhenADependencyIsDeclaredTwice() {
        Dependency dependency = new Dependency("io.vertx", "vertx-core", "3.1.0");
        Stack stack = new Stack().addDependency(dependency).addDependency(new Dependency("com.fasterxml.jackson.core", "jackson-databind", "2.6.1"));
        StackResolution resolution = new StackResolution(stack, this.root, STRICT);
        Map map = resolution.resolve();
        Assertions.assertThat((Map)map).containsKey((Object)"io.vertx:vertx-core:jar:3.1.0");
        Assertions.assertThat((Map)map).containsKey((Object)"com.fasterxml.jackson.core:jackson-databind:jar:2.6.1");
    }

    @Test(expected=DependencyConflictException.class)
    public void testConflictOnDependencyVersionMismatch() {
        Dependency dependency = new Dependency("io.vertx", "vertx-core", "3.1.0");
        Stack stack = new Stack().addDependency(dependency).addDependency(new Dependency("com.fasterxml.jackson.core", "jackson-databind", "2.4.1.3"));
        StackResolution resolution = new StackResolution(stack, this.root, STRICT);
        resolution.resolve();
    }

    @Test
    public void testConflictManagementUsingExclusions() {
        Dependency dependency = new Dependency("io.vertx", "vertx-core", "3.1.0");
        Exclusion exclusion1 = new Exclusion();
        exclusion1.setGroupId("com.fasterxml.jackson.core");
        exclusion1.setArtifactId("jackson-databind");
        Exclusion exclusion2 = new Exclusion();
        exclusion2.setGroupId("com.fasterxml.jackson.core");
        exclusion2.setArtifactId("jackson-core");
        dependency.addExclusion(exclusion1);
        dependency.addExclusion(exclusion2);
        Stack stack = new Stack().addDependency(dependency).addDependency(new Dependency("com.fasterxml.jackson.core", "jackson-databind", "2.4.1.3"));
        StackResolution resolution = new StackResolution(stack, this.root, STRICT);
        resolution.resolve();
        Map map = resolution.resolve();
        Assertions.assertThat((Map)map).containsKey((Object)"io.vertx:vertx-core:jar:3.1.0");
        Assertions.assertThat((Map)map).containsKey((Object)"com.fasterxml.jackson.core:jackson-databind:jar:2.4.1.3");
    }

    @Test
    public void testModificationOfStack() {
        Stack stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0")).addDependency(new Dependency("io.vertx", "vertx-stomp", "3.1.0").setIncluded(false));
        StackResolution resolution = new StackResolution(stack, this.root, STRICT);
        Map resolved = resolution.resolve();
        Assertions.assertThat((Map)resolved).doesNotContainKeys((Object[])new String[]{"io.vertx:vertx-stomp:jar:3.1.0"});
        int numberOfArtifacts = this.root.listFiles().length;
        stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0")).addDependency(new Dependency("io.vertx", "vertx-stomp", "3.1.0").setIncluded(true));
        resolution = new StackResolution(stack, this.root, STRICT);
        resolved = resolution.resolve();
        Assertions.assertThat((Map)resolved).containsKey((Object)"io.vertx:vertx-stomp:jar:3.1.0");
        stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0")).addDependency(new Dependency("io.vertx", "vertx-stomp", "3.1.0").setIncluded(false));
        resolution = new StackResolution(stack, this.root, STRICT);
        resolved = resolution.resolve();
        Assertions.assertThat((Map)resolved).doesNotContainKeys((Object[])new String[]{"io.vertx:vertx-stomp:jar:3.1.0"});
        int numberOfArtifacts2 = this.root.listFiles().length;
        Assertions.assertThat((int)numberOfArtifacts).isEqualTo(numberOfArtifacts2);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testTheResolutionWhenAnArtifactIsMissing() {
        Stack stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0")).addDependency(new Dependency("io.vertx", "vertx-missing", "3.1.0"));
        StackResolution resolution = new StackResolution(stack, this.root, STRICT);
        resolution.resolve();
    }

    @Test(expected=IllegalArgumentException.class)
    public void testTheResolutionWhenATransitiveDependencyIsMissing() {
        File local = new File("target/test-repos/incomplete");
        new LocalRepoBuilder(local).addArtifact(new LocalArtifact("org.acme", "acme", "1.0").generateMainArtifact().addDependency(new LocalDependency("org.acme", "acme-missing", "1.0"))).build();
        Stack stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0")).addDependency(new Dependency("org.acme", "acme", "1.0", "txt"));
        StackResolutionOptions options = new StackResolutionOptions().setFailOnConflicts(true).setLocalRepository(local.getAbsolutePath()).setCacheDisabled(true);
        StackResolution resolution = new StackResolution(stack, this.root, options);
        resolution.resolve();
    }

    @Test
    public void testTheResolutionWhenATransitiveDependencyIsMissingButExcluded() {
        File local = new File("target/test-repos/incomplete");
        new LocalRepoBuilder(local).addArtifact(new LocalArtifact("org.acme", "acme", "1.0").generateMainArtifact().addDependency(new LocalDependency("org.acme", "acme-missing", "1.0"))).build();
        Exclusion exclusion = new Exclusion();
        exclusion.setArtifactId("acme-missing");
        exclusion.setGroupId("org.acme");
        Dependency dependency = new Dependency("org.acme", "acme", "1.0", "txt");
        dependency.addExclusion(exclusion);
        Stack stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0")).addDependency(dependency);
        StackResolutionOptions options = new StackResolutionOptions().setFailOnConflicts(true).setLocalRepository(local.getAbsolutePath());
        StackResolution resolution = new StackResolution(stack, this.root, options);
        Map map = resolution.resolve();
        Assertions.assertThat((Map)map).containsKey((Object)"io.vertx:vertx-core:jar:3.1.0");
        Assertions.assertThat((Map)map).containsKey((Object)"org.acme:acme:txt:1.0");
    }

    @Test
    public void testTheResolutionWhenATransitiveDependencyIsMissingButOptional() {
        File local = new File("target/test-repos/incomplete");
        new LocalRepoBuilder(local).addArtifact(new LocalArtifact("org.acme", "acme", "1.0").generateMainArtifact().addDependency(new LocalDependency("org.acme", "acme-missing", "1.0").optional(true))).build();
        Exclusion exclusion = new Exclusion();
        exclusion.setArtifactId("acme-missing");
        exclusion.setGroupId("org.acme");
        Dependency dependency = new Dependency("org.acme", "acme", "1.0", "txt");
        Stack stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0")).addDependency(dependency);
        StackResolutionOptions options = new StackResolutionOptions().setFailOnConflicts(true).setLocalRepository(local.getAbsolutePath());
        StackResolution resolution = new StackResolution(stack, this.root, options);
        Map map = resolution.resolve();
        Assertions.assertThat((Map)map).containsKey((Object)"io.vertx:vertx-core:jar:3.1.0");
        Assertions.assertThat((Map)map).containsKey((Object)"org.acme:acme:txt:1.0");
    }

    @Test
    public void testModificationOfStackIntroducingConflict() {
        Stack stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0")).addDependency(new Dependency("io.vertx", "vertx-stomp", "3.1.0").setIncluded(false)).addDependency(new Dependency("io.vertx", "vertx-core", "3.0.0").setIncluded(false));
        StackResolution resolution = new StackResolution(stack, this.root, STRICT);
        Map resolved = resolution.resolve();
        Assertions.assertThat((Map)resolved).doesNotContainKeys((Object[])new String[]{"io.vertx:vertx-core:jar:3.0.0"});
        stack = new Stack().addDependency(new Dependency("io.vertx", "vertx-core", "3.1.0")).addDependency(new Dependency("io.vertx", "vertx-stomp", "3.1.0").setIncluded(true)).addDependency(new Dependency("io.vertx", "vertx-core", "3.0.0").setIncluded(true));
        resolution = new StackResolution(stack, this.root, STRICT);
        try {
            resolved = resolution.resolve();
            Assertions.fail((String)"Conflict expected");
        }
        catch (DependencyConflictException dependencyConflictException) {
            // empty catch block
        }
        Assertions.assertThat((Map)resolved).containsKey((Object)"io.vertx:vertx-core:jar:3.1.0");
        Assertions.assertThat((Map)resolved).doesNotContainKeys((Object[])new String[]{"io.vertx:vertx-core:jar:3.0.0"});
        Assertions.assertThat((Map)resolved).doesNotContainKeys((Object[])new String[]{"io.vertx:vertx-stomp:jar:3.1.0"});
    }
}

