/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.servicediscovery.kubernetes;

import io.vertx.core.AsyncResult;
import io.vertx.core.CompositeFuture;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpClientOptions;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.impl.ContextInternal;
import io.vertx.core.impl.future.PromiseInternal;
import io.vertx.core.impl.logging.Logger;
import io.vertx.core.impl.logging.LoggerFactory;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.core.parsetools.JsonParser;
import io.vertx.servicediscovery.Record;
import io.vertx.servicediscovery.spi.ServiceImporter;
import io.vertx.servicediscovery.spi.ServicePublisher;
import io.vertx.servicediscovery.types.HttpLocation;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class KubernetesServiceImporter
implements ServiceImporter {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)KubernetesServiceImporter.class.getName());
    private static final Set<String> SUPPORTED_EVENT_TYPES = Stream.of("BOOKMARK", "ADDED", "DELETED", "ERROR", "MODIFIED").collect(Collectors.toSet());
    public static final String KUBERNETES_UUID = "kubernetes.uuid";
    private final Map<RecordKey, Record> records = new HashMap<RecordKey, Record>();
    private ContextInternal context;
    private ServicePublisher publisher;
    private String token;
    private String namespace;
    private HttpClient client;
    private String lastResourceVersion;
    private BatchOfUpdates batchOfUpdates;
    private volatile boolean stop;
    private static final String OPENSHIFT_KUBERNETES_TOKEN_FILE = "/var/run/secrets/kubernetes.io/serviceaccount/token";

    public void start(Vertx vertx, ServicePublisher publisher, JsonObject configuration, Promise<Void> completion) {
        this.context = (ContextInternal)vertx.getOrCreateContext();
        this.context.runOnContext(v -> this.init(publisher, configuration, completion));
    }

    private void init(ServicePublisher publisher, JsonObject configuration, Promise<Void> completion) {
        String p;
        this.publisher = publisher;
        JsonObject conf = configuration == null ? new JsonObject() : configuration;
        int port = conf.getInteger("port", Integer.valueOf(0));
        if (port == 0) {
            port = conf.getBoolean("ssl", Boolean.valueOf(true)) != false ? 443 : 80;
        }
        if ((p = System.getenv("KUBERNETES_SERVICE_PORT")) != null) {
            port = Integer.parseInt(p);
        }
        String host = conf.getString("host");
        String h = System.getenv("KUBERNETES_SERVICE_HOST");
        if (h != null) {
            host = h;
        }
        this.client = this.context.owner().createHttpClient(new HttpClientOptions().setTrustAll(true).setSsl(conf.getBoolean("ssl", Boolean.valueOf(true)).booleanValue()).setDefaultHost(host).setDefaultPort(port));
        Future<Void> retrieveTokenFuture = this.retrieveToken(conf);
        this.namespace = conf.getString("namespace", this.getNamespaceOrDefault());
        LOGGER.info((Object)("Kubernetes discovery configured for namespace: " + this.namespace));
        LOGGER.info((Object)("Kubernetes master url: http" + (conf.getBoolean("ssl", Boolean.valueOf(true)) != false ? "s" : "") + "//" + host + ":" + port));
        retrieveTokenFuture.compose(v -> this.retrieveServices()).onSuccess(items -> LOGGER.info((Object)("Kubernetes initial import of " + items.size() + " services"))).compose(this::publishRecords).onComplete(ar -> {
            if (ar.succeeded()) {
                LOGGER.info((Object)("Kubernetes importer instantiated with " + this.records.size() + " services imported"));
                completion.complete();
                this.watch();
            } else {
                LOGGER.error((Object)"Error while interacting with kubernetes", ar.cause());
                completion.fail(ar.cause());
            }
        });
    }

    private Future<JsonArray> retrieveServices() {
        String path = "/api/v1/namespaces/" + this.namespace + "/services";
        return this.client.request(HttpMethod.GET, path).compose(request -> {
            request.setFollowRedirects(true);
            request.putHeader("Authorization", "Bearer " + this.token);
            return request.send();
        }).compose(response -> response.body().compose(body -> {
            if (response.statusCode() != 200) {
                return this.context.failedFuture("Unable to retrieve services from namespace " + this.namespace + ", status code: " + response.statusCode() + ", content: " + body.toString());
            }
            JsonObject serviceList = body.toJsonObject();
            this.lastResourceVersion = serviceList.getJsonObject("metadata").getString("resourceVersion");
            JsonArray items = serviceList.getJsonArray("items");
            if (!serviceList.containsKey("items")) {
                return this.context.failedFuture("Unable to retrieve services from namespace " + this.namespace + " - no items");
            }
            return this.context.succeededFuture((Object)items);
        }));
    }

    private CompositeFuture publishRecords(JsonArray items) {
        ArrayList publications = new ArrayList();
        items.forEach(s -> {
            JsonObject svc = (JsonObject)s;
            Record record = KubernetesServiceImporter.createRecord(svc);
            if (this.addRecordIfNotContained(record)) {
                PromiseInternal promise = this.context.promise();
                this.publishRecord(record, (Handler<AsyncResult<Record>>)promise);
                publications.add(promise.future());
            }
        });
        return CompositeFuture.all(publications);
    }

    private void watch() {
        if (this.stop) {
            return;
        }
        String path = "/api/v1/namespaces/" + this.namespace + "/services?watch=true&allowWatchBookmarks=true&resourceVersion=" + this.lastResourceVersion;
        JsonParser parser = JsonParser.newParser().objectValueMode().handler(event -> this.addToBatch(event.objectValue()));
        this.client.request(HttpMethod.GET, path).compose(request -> {
            request.setFollowRedirects(true);
            request.putHeader("Authorization", "Bearer " + this.token);
            return request.send();
        }).compose(response -> {
            Promise promise = Promise.promise();
            if (response.statusCode() == 200) {
                LOGGER.info((Object)("Watching services from namespace " + this.namespace));
                response.exceptionHandler(t -> promise.tryComplete()).endHandler(v -> promise.tryComplete()).handler((Handler)parser);
            } else {
                promise.fail("");
            }
            return promise.future();
        }).onComplete(res -> {
            if (res.succeeded()) {
                this.watch();
            } else {
                LOGGER.error((Object)"Failure while watching service list", res.cause());
                this.fetchAndWatch();
            }
        });
    }

    private void fetchAndWatch() {
        if (!this.stop) {
            this.context.setTimer(2000L, l -> this.retrieveServices().compose(this::publishRecords).onComplete(res -> {
                if (res.succeeded()) {
                    this.watch();
                } else {
                    this.fetchAndWatch();
                }
            }));
        }
    }

    private void addToBatch(JsonObject json) {
        if (this.batchOfUpdates == null) {
            long timerId = this.context.setTimer(500L, l -> this.processBatch());
            this.batchOfUpdates = new BatchOfUpdates((Vertx)this.context.owner(), timerId);
        }
        this.batchOfUpdates.objects.add(json);
    }

    private void processBatch() {
        Map<Object, JsonObject> objects = this.compact(this.batchOfUpdates.objects);
        this.batchOfUpdates = null;
        for (JsonObject json : objects.values()) {
            this.onChunk(json);
        }
    }

    private Map<Object, JsonObject> compact(List<JsonObject> source) {
        HashMap<Object, JsonObject> res = new HashMap<Object, JsonObject>();
        for (JsonObject json : source) {
            String type = json.getString("type");
            if (type == null || !SUPPORTED_EVENT_TYPES.contains(type)) continue;
            JsonObject object = json.getJsonObject("object");
            if ("BOOKMARK".equals(type)) {
                res.merge("BOOKMARK", json, (oldVal, newVal) -> newVal);
                continue;
            }
            RecordKey key = new RecordKey(KubernetesServiceImporter.createRecord(object));
            if ("DELETED".equals(type) || "ERROR".equals(type)) {
                res.put(key, json);
                continue;
            }
            JsonObject oldVal2 = (JsonObject)res.get(key);
            if (oldVal2 == null) {
                res.put(key, json);
                continue;
            }
            oldVal2.put("object", (Object)object);
        }
        return res;
    }

    private void onChunk(JsonObject json) {
        String type = json.getString("type");
        JsonObject object = json.getJsonObject("object");
        switch (type) {
            case "BOOKMARK": {
                this.lastResourceVersion = object.getJsonObject("metadata").getString("resourceVersion");
                break;
            }
            case "ADDED": {
                Record record = KubernetesServiceImporter.createRecord(object);
                if (!this.addRecordIfNotContained(record)) break;
                LOGGER.info((Object)("Adding service " + record.getName()));
                this.publishRecord(record, null);
                break;
            }
            case "DELETED": 
            case "ERROR": {
                Record record = KubernetesServiceImporter.createRecord(object);
                LOGGER.info((Object)("Removing service " + record.getName()));
                Record storedRecord = this.removeRecordIfContained(record);
                if (storedRecord == null) break;
                this.unpublishRecord(storedRecord, null);
                break;
            }
            case "MODIFIED": {
                Record record = KubernetesServiceImporter.createRecord(object);
                LOGGER.info((Object)("Modifying service " + record.getName()));
                Record storedRecord = this.replaceRecordIfContained(record);
                if (storedRecord == null) break;
                this.unpublishRecord(storedRecord, (Handler<Void>)((Handler)x -> this.publishRecord(record, null)));
            }
        }
    }

    private Future<Void> retrieveToken(JsonObject conf) {
        String token = conf.getString("token");
        Future result = token != null && !token.trim().isEmpty() ? this.context.succeededFuture((Object)token) : this.context.owner().fileSystem().readFile(OPENSHIFT_KUBERNETES_TOKEN_FILE).map(Buffer::toString);
        return result.onSuccess(tk -> {
            this.token = tk;
        }).mapEmpty();
    }

    private void publishRecord(Record record, Handler<AsyncResult<Record>> completionHandler) {
        this.publisher.publish(record, ar -> {
            if (completionHandler != null) {
                completionHandler.handle(ar);
            }
            if (ar.succeeded()) {
                LOGGER.info((Object)("Kubernetes service published in the vert.x service registry: " + record.toJson()));
            } else {
                LOGGER.error((Object)"Kubernetes service not published in the vert.x service registry", ar.cause());
            }
        });
    }

    private boolean addRecordIfNotContained(Record record) {
        return this.records.putIfAbsent(new RecordKey(record), record) == null;
    }

    private String getNamespaceOrDefault() {
        String ns = System.getenv("KUBERNETES_NAMESPACE");
        if (ns == null && (ns = System.getenv("OPENSHIFT_BUILD_NAMESPACE")) == null) {
            ns = "default";
        }
        return ns;
    }

    static Record createRecord(JsonObject service) {
        JsonObject metadata = service.getJsonObject("metadata");
        Record record = new Record().setName(metadata.getString("name"));
        JsonObject labels = metadata.getJsonObject("labels");
        if (labels != null) {
            labels.forEach(entry -> record.getMetadata().put((String)entry.getKey(), (Object)entry.getValue().toString()));
        }
        record.getMetadata().put("kubernetes.namespace", (Object)metadata.getString("namespace"));
        record.getMetadata().put("kubernetes.name", (Object)metadata.getString("name"));
        record.getMetadata().put(KUBERNETES_UUID, (Object)metadata.getString("uid"));
        String type = record.getMetadata().getString("service-type");
        if (type == null) {
            type = KubernetesServiceImporter.discoveryType(service, record);
        }
        switch (type) {
            case "http-endpoint": {
                KubernetesServiceImporter.manageHttpService(record, service);
                break;
            }
            default: {
                KubernetesServiceImporter.manageUnknownService(record, service, type);
            }
        }
        return record;
    }

    static String discoveryType(JsonObject service, Record record) {
        JsonObject port;
        int p;
        JsonObject spec = service.getJsonObject("spec");
        JsonArray ports = spec.getJsonArray("ports");
        if (ports == null || ports.isEmpty()) {
            return "unknown";
        }
        if (ports.size() > 1) {
            LOGGER.warn((Object)("More than one ports has been found for " + record.getName() + " - taking the first one to build the record location"));
        }
        if ((p = (port = ports.getJsonObject(0)).getInteger("port").intValue()) == 80 || p == 443 || p >= 8080 && p <= 9000) {
            return "http-endpoint";
        }
        if (p == 5432 || p == 5433) {
            return "jdbc";
        }
        if (p == 3306 || p == 13306) {
            return "jdbc";
        }
        if (p == 6379) {
            return "redis";
        }
        if (p == 27017 || p == 27018 || p == 27019) {
            return "mongo";
        }
        return "unknown";
    }

    private static void manageUnknownService(Record record, JsonObject service, String type) {
        JsonObject location;
        JsonObject spec = service.getJsonObject("spec");
        JsonArray ports = spec.getJsonArray("ports");
        if (ports != null && !ports.isEmpty()) {
            if (ports.size() > 1) {
                LOGGER.warn((Object)("More than one ports has been found for " + record.getName() + " - taking the first one to build the record location"));
            }
            JsonObject port = ports.getJsonObject(0);
            location = port.copy();
            if (KubernetesServiceImporter.isExternalService(service)) {
                location.put("host", (Object)spec.getString("externalName"));
            } else {
                Object targetPort = port.getValue("targetPort");
                if (targetPort instanceof Integer) {
                    location.put("internal-port", targetPort);
                }
                location.put("host", (Object)spec.getString("clusterIP"));
            }
        } else {
            throw new IllegalStateException("Cannot extract the location from the service " + record + " - no port");
        }
        record.setLocation(location).setType(type);
    }

    private static void manageHttpService(Record record, JsonObject service) {
        HttpLocation location;
        JsonObject spec = service.getJsonObject("spec");
        JsonArray ports = spec.getJsonArray("ports");
        if (ports != null && !ports.isEmpty()) {
            if (ports.size() > 1) {
                LOGGER.warn((Object)("More than one port has been found for " + record.getName() + " - taking the first one to extract the HTTP endpoint location"));
            }
            JsonObject port = ports.getJsonObject(0);
            Integer p = port.getInteger("port");
            record.setType("http-endpoint");
            location = new HttpLocation(port.copy());
            if (KubernetesServiceImporter.isExternalService(service)) {
                location.setHost(spec.getString("externalName"));
            } else {
                location.setHost(spec.getString("clusterIP"));
            }
            if (Boolean.parseBoolean(record.getMetadata().getString("ssl")) || p != null && p == 443) {
                location.setSsl(true);
            }
        } else {
            throw new IllegalStateException("Cannot extract the HTTP URL from the service " + record + " - no port");
        }
        record.setLocation(location.toJson());
    }

    private static boolean isExternalService(JsonObject service) {
        return service.containsKey("spec") && service.getJsonObject("spec").containsKey("type") && service.getJsonObject("spec").getString("type").equals("ExternalName");
    }

    public void close(Handler<Void> completionHandler) {
        this.stop = true;
        if (this.context != null) {
            this.context.runOnContext(v -> {
                if (this.batchOfUpdates != null) {
                    this.batchOfUpdates.cancel();
                }
                this.client.close();
                this.client = null;
                if (completionHandler != null) {
                    completionHandler.handle(null);
                }
            });
        } else if (completionHandler != null) {
            completionHandler.handle(null);
        }
    }

    private void unpublishRecord(Record record, Handler<Void> completionHandler) {
        this.publisher.unpublish(record.getRegistration(), ar -> {
            if (ar.failed()) {
                LOGGER.error((Object)"Cannot unregister kubernetes service", ar.cause());
            } else {
                LOGGER.info((Object)("Kubernetes service unregistered from the vert.x registry: " + record.toJson()));
                if (completionHandler != null) {
                    completionHandler.handle(null);
                }
            }
        });
    }

    private Record removeRecordIfContained(Record record) {
        return this.records.remove(new RecordKey(record));
    }

    private Record replaceRecordIfContained(Record record) {
        RecordKey key = new RecordKey(record);
        Record old = this.records.remove(key);
        if (old != null) {
            this.records.put(key, record);
        }
        return old;
    }

    private static class BatchOfUpdates {
        final Vertx vertx;
        final long timerId;
        final List<JsonObject> objects = new ArrayList<JsonObject>();

        public BatchOfUpdates(Vertx vertx, long timerId) {
            this.vertx = vertx;
            this.timerId = timerId;
        }

        public void cancel() {
            this.vertx.cancelTimer(this.timerId);
        }
    }

    private static class RecordKey {
        final String uuid;
        final String endpoint;

        RecordKey(Record record) {
            this.uuid = Objects.requireNonNull(record.getMetadata().getString(KubernetesServiceImporter.KUBERNETES_UUID));
            this.endpoint = record.getLocation().getString("endpoint", "");
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            RecordKey recordKey = (RecordKey)o;
            return this.uuid.equals(recordKey.uuid) && this.endpoint.equals(recordKey.endpoint);
        }

        public int hashCode() {
            int result = this.uuid.hashCode();
            result = 31 * result + this.endpoint.hashCode();
            return result;
        }

        public String toString() {
            return "RecordKey{uuid='" + this.uuid + '\'' + ", endpoint='" + this.endpoint + '\'' + '}';
        }
    }
}

