/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.stack.model;

import io.vertx.core.impl.logging.Logger;
import io.vertx.core.impl.logging.LoggerFactory;
import io.vertx.stack.model.Dependency;
import io.vertx.stack.model.DependencyConflictException;
import io.vertx.stack.model.Stack;
import io.vertx.stack.model.StackResolutionOptions;
import io.vertx.stack.resolver.Resolver;
import io.vertx.stack.utils.Actions;
import io.vertx.stack.utils.Cache;
import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import shaded.org.eclipse.aether.artifact.Artifact;

public class StackResolution {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)"Stack Resolution");
    private final File directory;
    private final Stack stack;
    private final Map<String, String> selectedVersions = new LinkedHashMap<String, String>();
    private final Map<String, ResolvedArtifact> selectedArtifacts = new LinkedHashMap<String, ResolvedArtifact>();
    private final Map<String, List<String>> traces = new LinkedHashMap<String, List<String>>();
    private final StackResolutionOptions options;
    private Resolver resolver;
    private Cache cache;

    public StackResolution(Stack stack, File directory, StackResolutionOptions options) {
        Objects.requireNonNull(stack);
        Objects.requireNonNull(options);
        Objects.requireNonNull(directory);
        this.stack = stack;
        this.options = options;
        this.directory = directory;
        this.cache = new Cache(options.isCacheDisabled(), options.isCacheDisabledForSnapshots(), options.getCacheFile());
    }

    public Map<String, File> resolve() {
        this.traces.clear();
        this.selectedVersions.clear();
        this.init();
        this.stack.getDependencies().forEach(this::resolve);
        List<Actions.Action> chain = this.computeChainOfActions();
        chain.forEach(Actions.Action::execute);
        LinkedHashMap<String, File> resolved = new LinkedHashMap<String, File>();
        for (ResolvedArtifact artifact : this.selectedArtifacts.values()) {
            Path source = artifact.artifact.getFile().toPath();
            Path output = this.directory.toPath().resolve(source.getFileName());
            resolved.put(artifact.artifact.toString(), output.toFile());
        }
        return resolved;
    }

    public Map<String, File> resolve(Predicate<String> validator) {
        this.traces.clear();
        this.selectedVersions.clear();
        this.init();
        this.stack.getDependencies().forEach(this::resolve);
        List<Actions.Action> chain = this.computeChainOfActions();
        chain.forEach(Actions.Action::execute);
        LinkedHashMap<String, File> resolved = new LinkedHashMap<String, File>();
        for (ResolvedArtifact artifact : this.selectedArtifacts.values()) {
            String gav = artifact.artifact.toString();
            if (!validator.test(gav)) {
                throw new IllegalStateException("Invalid artifact " + gav + ", used by " + artifact.getUsages());
            }
            Path source = artifact.artifact.getFile().toPath();
            Path output = this.directory.toPath().resolve(source.getFileName());
            resolved.put(gav, output.toFile());
        }
        return resolved;
    }

    private void init() {
        if (!this.directory.isDirectory()) {
            LOGGER.info((Object)("Creating directory " + this.directory.getAbsolutePath()));
            boolean mkdirs = this.directory.mkdirs();
            LOGGER.debug((Object)("Directory created: " + mkdirs));
        }
        this.stack.applyFiltering();
        this.stack.getDependencies().filter(Dependency::isIncluded).forEach(dependency -> this.selectedVersions.put(dependency.getManagementKey(), dependency.getVersion()));
        this.resolver = Resolver.create(this.options);
    }

    private List<Actions.Action> computeChainOfActions() {
        File[] files = this.directory.listFiles((dir, name) -> name.endsWith(".jar"));
        if (files == null) {
            throw new IllegalStateException("Unable to read from the file system");
        }
        HashMap<String, Boolean> marks = new HashMap<String, Boolean>();
        for (File file : files) {
            marks.put(file.getName(), false);
        }
        ArrayList<Actions.Action> chain = new ArrayList<Actions.Action>();
        this.selectedArtifacts.forEach((key, artifact) -> {
            String fileName = artifact.getArtifact().getFile().getName();
            if (marks.containsKey(fileName)) {
                marks.put(fileName, true);
                chain.add(Actions.skip(artifact.getArtifact()));
            } else {
                chain.add(Actions.copy(artifact.getArtifact(), this.directory));
            }
        });
        marks.forEach((fileName, mark) -> {
            if (!mark.booleanValue() && !fileName.startsWith("vertx-stack-manager-")) {
                chain.add(Actions.remove(new File(this.directory, (String)fileName)));
            }
        });
        return chain;
    }

    private void resolve(Dependency dependency) {
        List<Artifact> list;
        if (dependency.isIncluded()) {
            list = this.cache.get(dependency.getGACV(), dependency.getResolutionOptions());
            if (list == null || list.isEmpty()) {
                list = this.resolver.resolve(dependency.getGACV(), dependency.getResolutionOptions());
                this.cache.put(dependency.getGACV(), dependency.getResolutionOptions(), list);
                this.cache.writeCacheOnFile();
            } else {
                LOGGER.info((Object)("Dependency " + dependency + " loaded from cache"));
            }
        } else {
            return;
        }
        if (list == null || list.isEmpty()) {
            throw new IllegalArgumentException("Cannot resolve " + dependency.toString());
        }
        LOGGER.debug((Object)("Artifacts resolved for " + dependency.getGACV() + " : " + list.stream().map(Object::toString).collect(Collectors.toList())));
        list.forEach(artifact -> {
            String gaec = this.getManagementKey((Artifact)artifact);
            String version = this.selectedVersions.get(gaec);
            if (version == null || version.equalsIgnoreCase(artifact.getBaseVersion())) {
                this.selectedVersions.put(gaec, artifact.getBaseVersion());
                this.keepATrace(dependency, (Artifact)artifact);
            } else {
                List<String> trace = this.traces.get(gaec + ":" + version);
                if (this.options.isFailOnConflicts()) {
                    throw new DependencyConflictException(gaec, version, trace, dependency.getGACV(), artifact.getBaseVersion());
                }
            }
            this.addSelectedArtifact(dependency, (Artifact)artifact, version);
        });
    }

    private void keepATrace(Dependency dependency, Artifact artifact) {
        String traceId = this.getManagementKey(artifact) + ":" + artifact.getBaseVersion();
        List deps = this.traces.computeIfAbsent(traceId, k -> new ArrayList());
        deps.add(dependency.getGACV());
    }

    private void addSelectedArtifact(Dependency dependency, Artifact artifact, String version) {
        String key = this.getManagementKey(artifact);
        ResolvedArtifact resolved = this.selectedArtifacts.get(key);
        if (resolved != null) {
            resolved.addUsage(this.getManagementKey(dependency));
        } else {
            this.selectedArtifacts.put(key, new ResolvedArtifact().addUsage(this.getManagementKey(dependency)).setSelectedVersion(version).setArtifact(artifact));
        }
    }

    private String getManagementKey(Artifact artifact) {
        return artifact.getGroupId() + ":" + artifact.getArtifactId() + ":" + artifact.getExtension() + (artifact.getClassifier() != null && artifact.getClassifier().length() > 0 ? ":" + artifact.getClassifier() : "");
    }

    private String getManagementKey(Dependency dependency) {
        return dependency.getManagementKey();
    }

    public static class ResolvedArtifact {
        private Artifact artifact;
        private String selectedVersion;
        private Set<String> usages = new LinkedHashSet<String>();

        public Artifact getArtifact() {
            return this.artifact;
        }

        public ResolvedArtifact setArtifact(Artifact artifact) {
            this.artifact = artifact;
            return this;
        }

        public String getSelectedVersion() {
            return this.selectedVersion;
        }

        public ResolvedArtifact setSelectedVersion(String selectedVersion) {
            this.selectedVersion = selectedVersion;
            return this;
        }

        public ResolvedArtifact addUsage(String key) {
            this.usages.add(key);
            return this;
        }

        public Set<String> getUsages() {
            return this.usages;
        }
    }
}

