/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.ext.web.handler;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.Verticle;
import io.vertx.core.Vertx;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpClientOptions;
import io.vertx.core.http.HttpClientRequest;
import io.vertx.core.http.HttpHeaders;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServer;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.core.http.HttpVersion;
import io.vertx.core.json.JsonArray;
import io.vertx.core.net.PemKeyCertOptions;
import io.vertx.core.net.PemTrustOptions;
import io.vertx.ext.web.Http2PushMapping;
import io.vertx.ext.web.Router;
import io.vertx.ext.web.WebTestBase;
import io.vertx.ext.web.handler.StaticHandler;
import io.vertx.ext.web.impl.Utils;
import java.io.File;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import org.junit.Test;

public class StaticHandlerTest
extends WebTestBase {
    protected StaticHandler stat;

    @Override
    public void setUp() throws Exception {
        super.setUp();
        this.stat = StaticHandler.create();
        this.router.route().handler((Handler)this.stat);
    }

    @Test
    public void testGetDefaultIndex() throws Exception {
        this.testRequest(HttpMethod.GET, "/", 200, "OK", "<html><body>Index page</body></html>");
    }

    @Test
    public void testGetSubdirectoryWithoutSlashDefaultIndex() throws Exception {
        this.testRequest(HttpMethod.GET, "/somedir", null, res -> {
            String location = res.headers().get("location");
            this.assertEquals("/somedir/", location);
        }, 301, "Moved Permanently", null);
    }

    @Test
    public void testGetSubdirectorySlashDefaultIndex() throws Exception {
        this.testRequest(HttpMethod.GET, "/somedir/", 200, "OK", "<html><body>Subdirectory index page</body></html>");
    }

    @Test
    public void testGetOtherIndex() throws Exception {
        this.stat.setIndexPage("otherpage.html");
        this.testRequest(HttpMethod.GET, "/", 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testGetSubdirectoryOtherIndex() throws Exception {
        this.stat.setIndexPage("otherpage.html");
        this.testRequest(HttpMethod.GET, "/somedir/", 200, "OK", "<html><body>Subdirectory other page</body></html>");
    }

    @Test
    public void testGetSubdirectorySlashOtherIndex() throws Exception {
        this.stat.setIndexPage("otherpage.html");
        this.testRequest(HttpMethod.GET, "/somedir/", 200, "OK", "<html><body>Subdirectory other page</body></html>");
    }

    @Test
    public void testGetFileWithSpaces() throws Exception {
        this.testRequest(HttpMethod.GET, "/file%20with%20spaces.html", 200, "OK", "<html><body>File with spaces</body></html>");
    }

    @Test
    public void testGetOtherPage() throws Exception {
        this.testRequest(HttpMethod.GET, "/otherpage.html", 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testGetPageFromSubdir() throws Exception {
        this.testRequest(HttpMethod.GET, "/somedir/something.html", 200, "OK", "<html><body>Blah page</body></html>");
    }

    @Test
    public void testBadPathNoLeadingSlash() throws Exception {
        this.testRequest(HttpMethod.GET, "otherpage.html", 404, "Not Found");
    }

    @Test
    public void testGetHiddenPage() throws Exception {
        this.testRequest(HttpMethod.GET, "/.hidden.html", 200, "OK", "<html><body>Hidden page</body></html>");
    }

    @Test
    public void testCantGetHiddenPage() throws Exception {
        this.stat.setIncludeHidden(false);
        this.testRequest(HttpMethod.GET, "/.hidden.html", 404, "Not Found");
    }

    @Test
    public void testGetHiddenPageSubdir() throws Exception {
        this.testRequest(HttpMethod.GET, "/somedir/.hidden.html", 200, "OK", "<html><body>Hidden page</body></html>");
    }

    @Test
    public void testCantGetHiddenPageSubdir() throws Exception {
        this.stat.setIncludeHidden(false);
        this.testRequest(HttpMethod.GET, "/somedir/.hidden.html", 404, "Not Found");
    }

    @Test
    public void testCantGetNoSuchPage() throws Exception {
        this.testRequest(HttpMethod.GET, "/notexists.html", 404, "Not Found");
    }

    @Test
    public void testCantGetNoSuchPageInSubDir() throws Exception {
        this.testRequest(HttpMethod.GET, "/somedir/notexists.html", 404, "Not Found");
    }

    @Test
    public void testDateHeaderSet() throws Exception {
        this.testRequest(HttpMethod.GET, "/otherpage.html", null, res -> {
            String dateHeader = res.headers().get("date");
            this.assertNotNull(dateHeader);
            long diff = System.currentTimeMillis() - this.toDateTime(dateHeader);
            this.assertTrue(diff > 0L && diff < 2000L);
        }, 200, "OK", null);
    }

    @Test
    public void testContentHeadersSet() throws Exception {
        this.stat.setDefaultContentEncoding("UTF-8");
        this.testRequest(HttpMethod.GET, "/otherpage.html", null, res -> {
            String contentType = res.headers().get("content-type");
            String contentLength = res.headers().get("content-length");
            this.assertEquals("text/html;charset=UTF-8", contentType);
            this.assertEquals(this.fileSize("src/test/resources/webroot/otherpage.html"), Integer.valueOf(contentLength).intValue());
        }, 200, "OK", null);
        this.testRequest(HttpMethod.GET, "/foo.json", null, res -> {
            String contentType = res.headers().get("content-type");
            String contentLength = res.headers().get("content-length");
            this.assertEquals("application/json", contentType);
            this.assertEquals(this.fileSize("src/test/resources/webroot/foo.json"), Integer.valueOf(contentLength).intValue());
        }, 200, "OK", null);
    }

    @Test
    public void testNoLinkPreload() throws Exception {
        this.stat.setWebRoot("webroot/somedir3");
        this.testRequest(HttpMethod.GET, "/testLinkPreload.html", null, res -> {
            List linkHeaders = res.headers().getAll("Link");
            this.assertTrue(linkHeaders.isEmpty());
        }, 200, "OK", null);
    }

    @Test
    public void testLinkPreload() throws Exception {
        ArrayList<Http2PushMapping> mappings = new ArrayList<Http2PushMapping>();
        mappings.add(new Http2PushMapping("style.css", "style", false));
        mappings.add(new Http2PushMapping("coin.png", "image", false));
        this.stat.setHttp2PushMapping(mappings).setWebRoot("webroot/somedir3");
        this.testRequest(HttpMethod.GET, "/testLinkPreload.html", null, res -> {
            List linkHeaders = res.headers().getAll("Link");
            this.assertTrue(linkHeaders.contains("<style.css>; rel=preload; as=style"));
            this.assertTrue(linkHeaders.contains("<coin.png>; rel=preload; as=image"));
        }, 200, "OK", null);
    }

    @Test
    public void testNoHttp2Push() throws Exception {
        this.stat.setWebRoot("webroot/somedir3");
        this.router.route().handler((Handler)this.stat);
        HttpServer http2Server = this.vertx.createHttpServer(new HttpServerOptions().setUseAlpn(true).setSsl(true).setPemKeyCertOptions(new PemKeyCertOptions().setKeyPath("tls/server-key.pem").setCertPath("tls/server-cert.pem")));
        http2Server.requestHandler((Handler)this.router).listen(8443);
        HttpClientOptions options = new HttpClientOptions().setSsl(true).setUseAlpn(true).setProtocolVersion(HttpVersion.HTTP_2).setPemTrustOptions(new PemTrustOptions().addCertPath("tls/server-cert.pem"));
        HttpClient client = this.vertx.createHttpClient(options);
        client.request(HttpMethod.GET, 8443, "localhost", "/testLinkPreload.html").onComplete(this.onSuccess(req -> {
            req.pushHandler(pushedReq -> pushedReq.response(pushedResp -> this.fail()));
            req.send(this.onSuccess(resp -> {
                this.assertEquals(200L, resp.statusCode());
                this.assertEquals(HttpVersion.HTTP_2, resp.version());
                resp.bodyHandler(arg_0 -> ((StaticHandlerTest)this).assertNotNull(arg_0));
                this.testComplete();
            }));
        }));
        this.await();
    }

    @Test
    public void testHttp2Push() throws Exception {
        ArrayList<Http2PushMapping> mappings = new ArrayList<Http2PushMapping>();
        mappings.add(new Http2PushMapping("style.css", "style", false));
        mappings.add(new Http2PushMapping("coin.png", "image", false));
        this.stat.setHttp2PushMapping(mappings).setWebRoot("webroot/somedir3");
        this.router.route().handler((Handler)this.stat);
        HttpServer http2Server = this.vertx.createHttpServer(new HttpServerOptions().setUseAlpn(true).setSsl(true).setPemKeyCertOptions(new PemKeyCertOptions().setKeyPath("tls/server-key.pem").setCertPath("tls/server-cert.pem")));
        http2Server.requestHandler((Handler)this.router).listen(8443);
        HttpClientOptions options = new HttpClientOptions().setSsl(true).setUseAlpn(true).setProtocolVersion(HttpVersion.HTTP_2).setPemTrustOptions(new PemTrustOptions().addCertPath("tls/server-cert.pem"));
        HttpClient client = this.vertx.createHttpClient(options);
        CountDownLatch latch = new CountDownLatch(2);
        client.request(HttpMethod.GET, 8443, "localhost", "/testLinkPreload.html").onComplete(this.onSuccess(req -> req.pushHandler(pushedReq -> pushedReq.response(this.onSuccess(pushedResp -> {
            this.assertNotNull(pushedResp);
            pushedResp.bodyHandler(arg_0 -> ((StaticHandlerTest)this).assertNotNull(arg_0));
            latch.countDown();
        }))).send(this.onSuccess(resp -> {
            this.assertEquals(200L, resp.statusCode());
            this.assertEquals(HttpVersion.HTTP_2, resp.version());
            resp.bodyHandler(arg_0 -> ((StaticHandlerTest)this).assertNotNull(arg_0));
        }))));
        latch.await();
    }

    @Test
    public void testSkipCompressionForMediaTypes() throws Exception {
        StaticHandler staticHandler = StaticHandler.create().skipCompressionForMediaTypes(Collections.singleton("image/jpeg"));
        List<String> uris = Arrays.asList("/testCompressionSuffix.html", "/somedir/range.jpg", "/somedir/range.jpeg", "/somedir3/coin.png");
        List<String> expectedContentEncodings = Arrays.asList("gzip", HttpHeaders.IDENTITY.toString(), HttpHeaders.IDENTITY.toString(), "gzip");
        this.testSkipCompression(staticHandler, uris, expectedContentEncodings);
    }

    @Test
    public void testSkipCompressionForSuffixes() throws Exception {
        StaticHandler staticHandler = StaticHandler.create().skipCompressionForSuffixes(Collections.singleton("jpg"));
        List<String> uris = Arrays.asList("/testCompressionSuffix.html", "/somedir/range.jpg", "/somedir/range.jpeg", "/somedir3/coin.png");
        List<String> expectedContentEncodings = Arrays.asList("gzip", HttpHeaders.IDENTITY.toString(), "gzip", "gzip");
        this.testSkipCompression(staticHandler, uris, expectedContentEncodings);
    }

    private void testSkipCompression(StaticHandler staticHandler, List<String> uris, List<String> expectedContentEncodings) throws Exception {
        this.server.close();
        this.server = this.vertx.createHttpServer(this.getHttpServerOptions().setPort(0).setCompressionSupported(true));
        this.router = Router.router((Vertx)this.vertx);
        this.router.route().handler((Handler)staticHandler);
        CountDownLatch serverReady = new CountDownLatch(1);
        this.server.requestHandler((Handler)this.router).listen(this.onSuccess(s -> serverReady.countDown()));
        this.awaitLatch(serverReady);
        List contentEncodings = Collections.synchronizedList(new ArrayList());
        for (String uri : uris) {
            CountDownLatch responseReceived = new CountDownLatch(1);
            this.client.request(HttpMethod.GET, this.server.actualPort(), this.getHttpClientOptions().getDefaultHost(), uri, this.onSuccess(req -> req.putHeader(HttpHeaders.ACCEPT_ENCODING, (CharSequence)String.join((CharSequence)", ", "gzip", "jpg", "jpeg", "png")).send(this.onSuccess(resp -> {
                this.assertEquals(200L, resp.statusCode());
                contentEncodings.add(resp.getHeader(HttpHeaders.CONTENT_ENCODING));
                responseReceived.countDown();
            }))));
            this.awaitLatch(responseReceived);
        }
        this.assertEquals(expectedContentEncodings, contentEncodings);
    }

    @Test
    public void testHead() throws Exception {
        CountDownLatch latch = new CountDownLatch(1);
        this.testRequest(HttpMethod.HEAD, "/otherpage.html", null, res -> {
            res.bodyHandler(buff -> this.assertEquals(0L, buff.length()));
            res.endHandler(v -> latch.countDown());
        }, 200, "OK", null);
        this.awaitLatch(latch);
    }

    @Test
    public void testCacheReturnFromCache() throws Exception {
        this.testCacheReturnFromCache((lastModified, req) -> req.putHeader("if-modified-since", lastModified), 304, "Not Modified", null);
    }

    @Test
    public void testCacheGetNew() throws Exception {
        this.testCacheReturnFromCache((lastModified, req) -> req.putHeader("if-modified-since", Utils.formatRFC1123DateTime((long)(this.toDateTime((String)lastModified) - 1L))), 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testCacheReturnFromCacheWhenNoHeader() throws Exception {
        this.testCacheReturnFromCache((lastModified, req) -> {}, 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testCacheReturnFromCacheWhenInvalidHeader() throws Exception {
        this.testCacheReturnFromCache((lastModified, req) -> req.putHeader("if-modified-since", "whatever"), 200, "OK", "<html><body>Other page</body></html>");
    }

    private void testCacheReturnFromCache(BiConsumer<String, HttpClientRequest> handler, int expectedStatusCode, String expectedStatusMessage, String expectedStatusBody) throws Exception {
        AtomicReference lastModifiedRef = new AtomicReference();
        this.testRequest(HttpMethod.GET, "/otherpage.html", null, res -> {
            String cacheControl = res.headers().get("cache-control");
            String lastModified = res.headers().get("last-modified");
            lastModifiedRef.set(lastModified);
            this.assertNotNull(cacheControl);
            this.assertNotNull(lastModified);
            this.assertEquals("public, immutable, max-age=86400", cacheControl);
        }, 200, "OK", "<html><body>Other page</body></html>");
        this.testRequest(HttpMethod.GET, "/otherpage.html", req -> handler.accept((String)lastModifiedRef.get(), (HttpClientRequest)req), null, expectedStatusCode, expectedStatusMessage, expectedStatusBody);
    }

    @Test
    public void testCacheIndexPageReturnFromCache() throws Exception {
        AtomicReference lastModifiedRef = new AtomicReference();
        this.testRequest(HttpMethod.GET, "/somedir/", null, res -> {
            String cacheControl = res.headers().get("cache-control");
            String lastModified = res.headers().get("last-modified");
            lastModifiedRef.set(lastModified);
            this.assertNotNull(cacheControl);
            this.assertNotNull(lastModified);
            this.assertEquals("public, immutable, max-age=86400", cacheControl);
        }, 200, "OK", "<html><body>Subdirectory index page</body></html>");
        this.testRequest(HttpMethod.GET, "/somedir/", req -> req.putHeader("if-modified-since", (String)lastModifiedRef.get()), null, 304, "Not Modified", null);
    }

    @Test
    public void testCachingDisabled() throws Exception {
        this.stat.setCachingEnabled(false);
        this.testRequest(HttpMethod.GET, "/otherpage.html", null, res -> {
            String cacheControl = res.headers().get("cache-control");
            String lastModified = res.headers().get("last-modified");
            this.assertNull(cacheControl);
            this.assertNull(lastModified);
        }, 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testCacheNoCacheAsNoIfModifiedSinceHeader() throws Exception {
        this.testRequest(HttpMethod.GET, "/otherpage.html", 200, "OK", "<html><body>Other page</body></html>");
        this.testRequest(HttpMethod.GET, "/otherpage.html", 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testCacheNotOverwritingCacheControlHeaderValues() throws Exception {
        this.router.clear();
        this.router.route().order(0).handler(context -> {
            context.response().putHeader("cache-control", "test1");
            context.response().putHeader("last-modified", "test2");
            context.response().putHeader("vary", "test3");
            context.next();
        });
        this.router.route().order(2).handler((Handler)this.stat);
        this.testRequest(HttpMethod.GET, "/otherpage.html", req -> req.putHeader("accept-encoding", "gzip"), res -> {
            String cacheControl = res.headers().get("cache-control");
            String lastModified = res.headers().get("last-modified");
            String vary = res.headers().get("vary");
            this.assertEquals("test1", cacheControl);
            this.assertEquals("test2", lastModified);
            this.assertEquals("test3", vary);
        }, 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testSendVaryAcceptEncodingHeader() throws Exception {
        this.testRequest(HttpMethod.GET, "/otherpage.html", req -> req.putHeader("accept-encoding", "gzip"), res -> {
            String vary = res.headers().get("vary");
            this.assertNotNull(vary);
            this.assertEquals("accept-encoding", vary);
        }, 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testNoSendingOfVaryAcceptEncodingHeader() throws Exception {
        this.testRequest(HttpMethod.GET, "/otherpage.html", null, res -> {
            String vary = res.headers().get("vary");
            this.assertNull(vary);
        }, 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testSetMaxAge() throws Exception {
        long maxAge = 3600L;
        this.stat.setMaxAgeSeconds(maxAge);
        this.testRequest(HttpMethod.GET, "/otherpage.html", null, res -> {
            String cacheControl = res.headers().get("cache-control");
            this.assertEquals("public, immutable, max-age=" + maxAge, cacheControl);
        }, 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testGetOtherPageTwice() throws Exception {
        this.testRequest(HttpMethod.GET, "/otherpage.html", 200, "OK", "<html><body>Other page</body></html>");
        this.testRequest(HttpMethod.GET, "/otherpage.html", 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testServeFilesFromFilesystem() throws Exception {
        this.stat.setWebRoot("src/test/filesystemwebroot");
        this.testRequest(HttpMethod.GET, "/fspage.html", 200, "OK", "<html><body>File system page</body></html>");
    }

    @Test
    public void testServeFilesFromFilesystemWebRootConstructor() throws Exception {
        this.stat = StaticHandler.create((String)"src/test/filesystemwebroot");
        this.router.clear();
        this.router.route().handler((Handler)this.stat);
        this.testRequest(HttpMethod.GET, "/fspage.html", 200, "OK", "<html><body>File system page</body></html>");
    }

    @Test
    public void testCacheFilesNotReadOnly() throws Exception {
        this.stat.setFilesReadOnly(false);
        this.stat.setWebRoot("src/test/filesystemwebroot");
        long modified = Utils.secondsFactor((long)new File("src/test/filesystemwebroot", "fspage.html").lastModified());
        this.testRequest(HttpMethod.GET, "/fspage.html", null, res -> {
            String lastModified = res.headers().get("last-modified");
            this.assertEquals(modified, this.toDateTime(lastModified));
        }, 200, "OK", "<html><body>File system page</body></html>");
        this.testRequest(HttpMethod.GET, "/fspage.html", req -> req.putHeader("if-modified-since", Utils.formatRFC1123DateTime((long)modified)), null, 304, "Not Modified", null);
    }

    @Test
    public void testCacheFilesEntryCached() throws Exception {
        this.stat.setFilesReadOnly(false);
        this.stat.setWebRoot("src/test/filesystemwebroot");
        File resource = new File("src/test/filesystemwebroot", "fspage.html");
        long modified = resource.lastModified();
        this.testRequest(HttpMethod.GET, "/fspage.html", null, res -> {
            String lastModified = res.headers().get("last-modified");
            this.assertEquals(modified, this.toDateTime(lastModified));
            resource.setLastModified(modified + 1000L);
        }, 200, "OK", "<html><body>File system page</body></html>");
        this.testRequest(HttpMethod.GET, "/fspage.html", req -> req.putHeader("if-modified-since", Utils.formatRFC1123DateTime((long)modified)), null, 304, "Not Modified", null);
    }

    @Test
    public void testCacheFilesEntryOld() throws Exception {
        String webroot = "src/test/filesystemwebroot";
        String page = "/fspage.html";
        File resource = new File(webroot + page);
        String html = new String(Files.readAllBytes(resource.toPath()));
        int cacheEntryTimeout = 100;
        this.stat.setFilesReadOnly(false);
        this.stat.setWebRoot(webroot);
        this.stat.setCacheEntryTimeout((long)cacheEntryTimeout);
        long modified = Utils.secondsFactor((long)resource.lastModified());
        this.testRequest(HttpMethod.GET, page, null, res -> {
            String lastModified = res.headers().get("last-modified");
            this.assertEquals(modified, this.toDateTime(lastModified));
            resource.setLastModified(modified + 1000L);
        }, 200, "OK", html);
        Thread.sleep(cacheEntryTimeout + 1);
        this.testRequest(HttpMethod.GET, page, req -> req.putHeader("if-modified-since", Utils.formatRFC1123DateTime((long)modified)), res -> {
            String lastModified = res.headers().get("last-modified");
            this.assertEquals(modified + 1000L, this.toDateTime(lastModified));
        }, 200, "OK", html);
        Thread.sleep(cacheEntryTimeout + 1);
        this.testRequest(HttpMethod.GET, page, (HttpClientRequest req) -> req.putHeader("if-modified-since", Utils.formatRFC1123DateTime((long)(modified + 1000L))), 304, "Not Modified", null);
    }

    @Test
    public void testCacheFilesFileDeleted() throws Exception {
        File webroot = new File("target/.vertx/webroot");
        File pageFile = new File(webroot, "deleted.html");
        if (!pageFile.exists()) {
            webroot.mkdirs();
            pageFile.createNewFile();
        }
        String page = '/' + pageFile.getName();
        this.stat.setFilesReadOnly(false);
        this.stat.setWebRoot(webroot.getPath());
        this.stat.setCacheEntryTimeout(3600000L);
        long modified = Utils.secondsFactor((long)pageFile.lastModified());
        this.testRequest(HttpMethod.GET, page, req -> req.putHeader("if-modified-since", Utils.formatRFC1123DateTime((long)modified)), null, 304, "Not Modified", null);
        pageFile.delete();
        this.testRequest(HttpMethod.GET, page, 404, "Not Found");
        this.testRequest(HttpMethod.GET, page, req -> req.putHeader("if-modified-since", Utils.formatRFC1123DateTime((long)modified)), null, 404, "Not Found", null);
    }

    @Test
    public void testDirectoryListingText() throws Exception {
        this.stat.setDirectoryListing(true);
        HashSet<String> expected = new HashSet<String>(Arrays.asList(".hidden.html", "a", "foo.json", "index.html", "otherpage.html", "somedir", "somedir2", "somedir3", "testCompressionSuffix.html", "file with spaces.html", "sockjs"));
        this.testRequest(HttpMethod.GET, "/", null, resp -> resp.bodyHandler(buff -> {
            String sBuff = buff.toString();
            String[] elems = sBuff.split("\n");
            this.assertEquals(expected.size(), elems.length);
            for (String elem : elems) {
                this.assertTrue(expected.contains(elem));
            }
        }), 200, "OK", null);
    }

    @Test
    public void testDirectoryListingTextNoHidden() throws Exception {
        this.stat.setDirectoryListing(true);
        this.stat.setIncludeHidden(false);
        HashSet<String> expected = new HashSet<String>(Arrays.asList("foo.json", "a", "index.html", "otherpage.html", "somedir", "somedir2", "somedir3", "testCompressionSuffix.html", "file with spaces.html", "sockjs"));
        this.testRequest(HttpMethod.GET, "/", null, resp -> resp.bodyHandler(buff -> {
            this.assertEquals("text/plain", resp.headers().get("content-type"));
            String sBuff = buff.toString();
            String[] elems = sBuff.split("\n");
            this.assertEquals(expected.size(), elems.length);
            for (String elem : elems) {
                this.assertTrue(expected.contains(elem));
            }
        }), 200, "OK", null);
    }

    @Test
    public void testDirectoryListingJson() throws Exception {
        this.stat.setDirectoryListing(true);
        HashSet<String> expected = new HashSet<String>(Arrays.asList(".hidden.html", "foo.json", "index.html", "otherpage.html", "a", "somedir", "somedir2", "somedir3", "testCompressionSuffix.html", "file with spaces.html", "sockjs"));
        this.testRequest(HttpMethod.GET, "/", req -> req.putHeader("accept", "application/json"), resp -> resp.bodyHandler(buff -> {
            this.assertEquals("application/json", resp.headers().get("content-type"));
            String sBuff = buff.toString();
            JsonArray arr = new JsonArray(sBuff);
            this.assertEquals(expected.size(), arr.size());
            for (Object elem : arr) {
                this.assertTrue(expected.contains(elem));
            }
            this.testComplete();
        }), 200, "OK", null);
        this.await();
    }

    @Test
    public void testDirectoryListingJsonNoHidden() throws Exception {
        this.stat.setDirectoryListing(true);
        this.stat.setIncludeHidden(false);
        HashSet<String> expected = new HashSet<String>(Arrays.asList("foo.json", "a", "index.html", "otherpage.html", "somedir", "somedir2", "somedir3", "testCompressionSuffix.html", "file with spaces.html", "sockjs"));
        this.testRequest(HttpMethod.GET, "/", req -> req.putHeader("accept", "application/json"), resp -> resp.bodyHandler(buff -> {
            this.assertEquals("application/json", resp.headers().get("content-type"));
            String sBuff = buff.toString();
            JsonArray arr = new JsonArray(sBuff);
            this.assertEquals(expected.size(), arr.size());
            for (Object elem : arr) {
                this.assertTrue(expected.contains(elem));
            }
            this.testComplete();
        }), 200, "OK", null);
        this.await();
    }

    @Test
    public void testDirectoryListingHtml() throws Exception {
        this.stat.setDirectoryListing(true);
        this.testDirectoryListingHtmlCustomTemplate("META-INF/vertx/web/vertx-web-directory.html");
    }

    @Test
    public void testCustomDirectoryListingHtml() throws Exception {
        this.stat.setDirectoryListing(true);
        String dirTemplate = "custom_dir_template.html";
        this.stat.setDirectoryTemplate(dirTemplate);
        this.testDirectoryListingHtmlCustomTemplate(dirTemplate);
    }

    private void testDirectoryListingHtmlCustomTemplate(String dirTemplateFile) throws Exception {
        this.stat.setDirectoryListing(true);
        String directoryTemplate = this.vertx.fileSystem().readFileBlocking(dirTemplateFile).toString();
        String parentLink = "<a href=\"/\">..</a>";
        String files = "<ul id=\"files\"><li><a href=\"/somedir2/foo2.json\" title=\"foo2.json\">foo2.json</a></li><li><a href=\"/somedir2/somepage.html\" title=\"somepage.html\">somepage.html</a></li><li><a href=\"/somedir2/somepage2.html\" title=\"somepage2.html\">somepage2.html</a></li></ul>";
        String expected = directoryTemplate.replace("{directory}", "/somedir2/").replace("{parent}", parentLink).replace("{files}", files);
        this.testRequest(HttpMethod.GET, "/somedir2/", req -> req.putHeader("accept", "text/html"), resp -> resp.bodyHandler(buff -> {
            this.assertEquals("text/html", resp.headers().get("content-type"));
            String sBuff = buff.toString();
            this.assertEquals(expected, sBuff);
            this.testComplete();
        }), 200, "OK", null);
        this.await();
    }

    @Test
    public void testFSBlockingTuning() throws Exception {
        this.stat.setCachingEnabled(false);
        this.stat.setMaxAvgServeTimeNs(10000L);
        for (int i = 0; i < 2000; ++i) {
            this.testRequest(HttpMethod.GET, "/otherpage.html", null, res -> {
                String cacheControl = res.headers().get("cache-control");
                String lastModified = res.headers().get("last-modified");
                this.assertNull(cacheControl);
                this.assertNull(lastModified);
            }, 200, "OK", "<html><body>Other page</body></html>");
        }
    }

    @Test
    public void testServerRelativeToPath() throws Exception {
        this.router.clear();
        this.router.route("/somedir/*").handler((Handler)this.stat);
        this.testRequest(HttpMethod.GET, "/somedir/otherpage.html", 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testServerRelativeToPathAndMountPoint() throws Exception {
        this.router.clear();
        Router subRouter = Router.router((Vertx)this.vertx);
        subRouter.route("/somedir/*").handler((Handler)this.stat);
        this.router.mountSubRouter("/mymount/", subRouter);
        this.testRequest(HttpMethod.GET, "/mymount/somedir/otherpage.html", 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testRangeAwareRequestHeaders() throws Exception {
        this.stat.setEnableRangeSupport(true);
        this.testRequest(HttpMethod.HEAD, "/somedir/range.jpg", null, res -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("15783", res.headers().get("Content-Length"));
        }, 200, "OK", null);
        this.testRequest(HttpMethod.GET, "/somedir/range.jpg", req -> req.headers().set("Range", "bytes=0-999"), res -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("1000", res.headers().get("Content-Length"));
            this.assertEquals("bytes 0-999/15783", res.headers().get("Content-Range"));
        }, 206, "Partial Content", null);
        this.testRequest(HttpMethod.GET, "/somedir/range.jpg", req -> req.headers().set("Range", "bytes=1000-"), res -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("14783", res.headers().get("Content-Length"));
            this.assertEquals("bytes 1000-15782/15783", res.headers().get("Content-Range"));
        }, 206, "Partial Content", null);
        this.testRequest(HttpMethod.GET, "/somedir/range.jpg", req -> req.headers().set("Range", "bytes=1000-5000000"), res -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("14783", res.headers().get("Content-Length"));
            this.assertEquals("bytes 1000-15782/15783", res.headers().get("Content-Range"));
        }, 206, "Partial Content", null);
    }

    @Test
    public void testRangeAwareRequestBody() throws Exception {
        this.stat.setEnableRangeSupport(true);
        this.testRequest(HttpMethod.GET, "/somedir/range.jpg", req -> req.headers().set("Range", "bytes=0-999"), res -> res.bodyHandler(buff -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("1000", res.headers().get("Content-Length"));
            this.assertEquals("bytes 0-999/15783", res.headers().get("Content-Range"));
            this.assertEquals(1000L, buff.length());
            this.testComplete();
        }), 206, "Partial Content", null);
        this.await();
    }

    @Test
    public void testRangeAwareRequestSegment() throws Exception {
        this.stat.setEnableRangeSupport(true);
        this.testRequest(HttpMethod.GET, "/somedir/range.bin", req -> req.headers().set("Range", "bytes=0-1023"), res -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("1024", res.headers().get("Content-Length"));
            res.bodyHandler(body -> this.assertEquals(1024L, body.length()));
        }, 206, "Partial Content", null);
        this.testRequest(HttpMethod.GET, "/somedir/range.bin", req -> req.headers().set("Range", "bytes=1024-2047"), res -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("1024", res.headers().get("Content-Length"));
            res.bodyHandler(body -> this.assertEquals(1024L, body.length()));
        }, 206, "Partial Content", null);
        this.testRequest(HttpMethod.GET, "/somedir/range.bin", req -> req.headers().set("Range", "bytes=2048-3071"), res -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("1024", res.headers().get("Content-Length"));
            res.bodyHandler(body -> this.assertEquals(1024L, body.length()));
        }, 206, "Partial Content", null);
        this.testRequest(HttpMethod.GET, "/somedir/range.bin", req -> req.headers().set("Range", "bytes=3072-4095"), res -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("1024", res.headers().get("Content-Length"));
            res.bodyHandler(body -> this.assertEquals(1024L, body.length()));
        }, 206, "Partial Content", null);
        this.testRequest(HttpMethod.GET, "/somedir/range.bin", req -> req.headers().set("Range", "bytes=4096-5119"), res -> {
            this.assertEquals("bytes", res.headers().get("Accept-Ranges"));
            this.assertEquals("1024", res.headers().get("Content-Length"));
            res.bodyHandler(body -> this.assertEquals(1024L, body.length()));
        }, 206, "Partial Content", null);
    }

    @Test
    public void testRangeAwareRequestBodyForDisabledRangeSupport() throws Exception {
        this.stat.setEnableRangeSupport(false);
        this.testRequest(HttpMethod.GET, "/somedir/range.jpg", req -> req.headers().set("Range", "bytes=0-999"), res -> res.bodyHandler(buff -> {
            this.assertNull(res.headers().get("Accept-Ranges"));
            this.assertNotSame("1000", res.headers().get("Content-Length"));
            this.assertNotSame(1000, buff.length());
            this.testComplete();
        }), 200, "OK", null);
        this.await();
    }

    @Test
    public void testOutOfRangeRequestBody() throws Exception {
        this.stat.setEnableRangeSupport(true);
        this.testRequest(HttpMethod.GET, "/somedir/range.jpg", req -> req.headers().set("Range", "bytes=15783-"), res -> res.bodyHandler(buff -> {
            this.assertEquals("bytes */15783", res.headers().get("Content-Range"));
            this.testComplete();
        }), 416, "Requested Range Not Satisfiable", null);
        this.await();
    }

    @Test
    public void testContentTypeSupport() throws Exception {
        this.testRequest(HttpMethod.GET, "/somedir/range.jpg", req -> {}, res -> {
            this.assertNotNull(res.getHeader("Content-Type"));
            this.assertEquals("image/jpeg", res.getHeader("Content-Type"));
            this.testComplete();
        }, 200, "OK", null);
        this.await();
    }

    @Test
    public void testAsyncExceptionIssue231() throws Exception {
        this.stat.setAlwaysAsyncFS(true);
        this.testRequest(HttpMethod.GET, "/non_existing.html", 404, "Not Found");
    }

    @Test
    public void testServerFileSystemPath() throws Exception {
        this.router.clear();
        File file = File.createTempFile("vertx", "tmp");
        file.deleteOnExit();
        try {
            this.stat = StaticHandler.create((String)file.getParent());
            this.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        this.stat = StaticHandler.create().setAllowRootFileSystemAccess(true).setWebRoot(file.getParent());
        this.router.route().handler((Handler)this.stat);
        this.testRequest(HttpMethod.GET, "/" + file.getName(), 200, "OK", "");
    }

    @Test(expected=IllegalArgumentException.class)
    public void testAccessToRootPath() throws Exception {
        this.router.clear();
        File file = File.createTempFile("vertx", "tmp");
        file.deleteOnExit();
        this.stat = StaticHandler.create().setWebRoot(file.getParent());
    }

    @Test
    public void testLastModifiedInGMT() throws Exception {
        this.testRequest(HttpMethod.GET, "/otherpage.html", null, res -> {
            String lastModified = res.headers().get("last-modified");
            this.assertTrue(lastModified.endsWith("GMT"));
        }, 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testChangeDefaultContentEncoding() throws Exception {
        this.stat.setDefaultContentEncoding("ISO-8859-1");
        this.testRequest(HttpMethod.GET, "/otherpage.html", null, res -> {
            String contentType = res.headers().get("Content-Type");
            this.assertEquals("text/html;charset=ISO-8859-1", contentType);
        }, 200, "OK", "<html><body>Other page</body></html>");
    }

    @Test
    public void testHandlerAfter() throws Exception {
        this.router.get().handler(ctx -> ctx.response().end("Howdy!"));
        this.testRequest(HttpMethod.GET, "/not-existing-file.html", 200, "OK", "Howdy!");
    }

    @Test
    public void testWriteResponseWhenAlreadyClosed() throws Exception {
        this.router.clear();
        this.router.route().handler(rc -> {
            rc.next();
            rc.response().end("OtherResponse");
        }).handler((Handler)this.stat);
        this.testRequest(HttpMethod.GET, "/index.html", 200, "OK", "OtherResponse");
    }

    @Test
    public void testEscapeWindows() throws Exception {
        this.router.clear();
        this.router.route().handler((Handler)this.stat);
        this.testRequest(HttpMethod.GET, "/%5c..%5cindex.html", 200, "OK");
    }

    @Test
    public void testWithClassLoader() throws Exception {
        File tmp = File.createTempFile("vertx_", ".txt");
        tmp.deleteOnExit();
        final URL url = tmp.toURI().toURL();
        Files.write(tmp.toPath(), "hello".getBytes(), new OpenOption[0]);
        final AtomicBoolean used = new AtomicBoolean();
        ClassLoader classLoader = new ClassLoader(Thread.currentThread().getContextClassLoader()){

            @Override
            public URL getResource(String name) {
                if (name.equals("webroot/index.html")) {
                    used.set(true);
                    return url;
                }
                return super.getResource(name);
            }
        };
        this.server.close();
        CountDownLatch latch = new CountDownLatch(1);
        this.vertx.deployVerticle((Verticle)new AbstractVerticle(){

            public void start(Promise<Void> startPromise) throws Exception {
                StaticHandlerTest.this.server = this.vertx.createHttpServer(StaticHandlerTest.this.getHttpServerOptions());
                StaticHandlerTest.this.server.requestHandler((Handler)StaticHandlerTest.this.router).listen().mapEmpty().onComplete(startPromise);
            }
        }, new DeploymentOptions().setClassLoader(classLoader), this.onSuccess(v -> latch.countDown()));
        this.awaitLatch(latch);
        this.testRequest(HttpMethod.GET, "/index.html", 200, "OK", "hello");
        this.assertTrue(used.get());
    }

    private long toDateTime(String header) {
        try {
            return Utils.parseRFC1123DateTime((String)header);
        }
        catch (Exception e) {
            this.fail(e.getMessage());
            return -1L;
        }
    }

    private long fileSize(String filename) {
        return new File(filename).length();
    }
}

