package optimajet.workflow.ignite;

import com.fasterxml.jackson.core.type.TypeReference;
import optimajet.workflow.core.DefiningParametersSerializer;
import optimajet.workflow.core.HashHelper;
import optimajet.workflow.core.fault.*;
import optimajet.workflow.core.model.*;
import optimajet.workflow.core.persistence.ErrorLevel;
import optimajet.workflow.core.persistence.ProcessStatus;
import optimajet.workflow.core.provider.WorkflowDocumentProvider;
import optimajet.workflow.core.runtime.ParametersSerializer;
import optimajet.workflow.core.runtime.ProcessHistoryItem;
import optimajet.workflow.core.runtime.TimerToExecute;
import optimajet.workflow.core.runtime.WorkflowRuntime;
import optimajet.workflow.core.util.CollectionUtil;
import optimajet.workflow.core.util.DocumentUtil;
import optimajet.workflow.core.util.JsonConvert;
import optimajet.workflow.core.util.StringUtil;
import org.apache.ignite.Ignite;
import org.apache.ignite.IgniteCache;
import org.apache.ignite.cache.CacheMode;
import org.apache.ignite.cache.CacheWriteSynchronizationMode;
import org.apache.ignite.cache.query.QueryCursor;
import org.apache.ignite.cache.query.SqlQuery;
import org.apache.ignite.configuration.CacheConfiguration;
import org.apache.ignite.transactions.Transaction;
import org.w3c.dom.Document;

import javax.cache.Cache;
import java.util.*;

public class IgniteProvider implements WorkflowDocumentProvider {

    private static final CacheConfiguration<UUID, WorkflowProcessInstance> WORKFLOW_PROCESS_INSTANCE_CACHE_CONFIG =
            getCacheConfiguration(UUID.class, WorkflowProcessInstance.class, "WorkflowProcessInstance");
    private static final CacheConfiguration<UUID, WorkflowProcessInstanceStatus> WORKFLOW_PROCESS_INSTANCE_STATUS_CACHE_CONFIG =
            getCacheConfiguration(UUID.class, WorkflowProcessInstanceStatus.class, "WorkflowProcessInstanceStatus");
    private static final CacheConfiguration<UUID, WorkflowProcessScheme> WORKFLOW_PROCESS_SCHEME_CACHE_CONFIG =
            getCacheConfiguration(UUID.class, WorkflowProcessScheme.class, "WorkflowProcessScheme");
    private static final CacheConfiguration<UUID, WorkflowProcessTransitionHistory> WORKFLOW_PROCESS_TRANSITION_HISTORY_CACHE_CONFIG =
            getCacheConfiguration(UUID.class, WorkflowProcessTransitionHistory.class, "WorkflowProcessTransitionHistory");
    private static final CacheConfiguration<String, WorkflowScheme> WORKFLOW_SCHEME_CACHE_CONFIG =
            getCacheConfiguration(String.class, WorkflowScheme.class, "WorkflowScheme");
    private static final CacheConfiguration<UUID, WorkflowProcessTimer> WORKFLOW_PROCESS_TIMER_CACHE_CONFIG =
            getCacheConfiguration(UUID.class, WorkflowProcessTimer.class, "WorkflowProcessTimer");
    private static final CacheConfiguration<UUID, WorkflowGlobalParameter> WORKFLOW_GLOBAL_PARAMETER_CACHE_CONFIG =
            getCacheConfiguration(UUID.class, WorkflowGlobalParameter.class, "WorkflowGlobalParameter");

    private final Ignite ignite;
    private WorkflowRuntime runtime;

    public IgniteProvider(Ignite ignite) {
        this.ignite = ignite;
    }

    private static <I, T> CacheConfiguration<I, T> getCacheConfiguration(Class<I> keyClass, Class<T> valueClass, String name) {
        return getCacheConfiguration(keyClass, valueClass, name, CacheWriteSynchronizationMode.PRIMARY_SYNC);
    }

    private static <I, T> CacheConfiguration<I, T> getCacheConfiguration(Class<I> keyClass, Class<T> valueClass, String name,
                                                                         CacheWriteSynchronizationMode cacheWriteSynchronizationMode) {
        CacheConfiguration<I, T> cacheCfg = new CacheConfiguration<>(name);
        cacheCfg.setCacheMode(CacheMode.PARTITIONED);
        cacheCfg.setWriteSynchronizationMode(cacheWriteSynchronizationMode);
        cacheCfg.setIndexedTypes(keyClass, valueClass);
        return cacheCfg;
    }

    @Override
    public void init(WorkflowRuntime runtime) {
        this.runtime = runtime;
    }

    @Override
    public void initializeProcess(ProcessInstance processInstance) {
        try (IgniteCache<UUID, WorkflowProcessInstance> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_CACHE_CONFIG)) {
            if (cache.containsKey(processInstance.getProcessId())) {
                throw new ProcessAlreadyExistsException(processInstance.getProcessId());
            }

            WorkflowProcessInstance newProcess = new WorkflowProcessInstance();
            newProcess.setId(processInstance.getProcessId());
            newProcess.setSchemeId(processInstance.getSchemeId());
            newProcess.setActivityName(processInstance.getProcessScheme().getInitialActivity().getName());
            newProcess.setStateName(processInstance.getProcessScheme().getInitialActivity().getState());
            newProcess.setRootProcessId(processInstance.getRootProcessId());
            newProcess.setParentProcessId(processInstance.getParentProcessId());
            newProcess.setPersistence(new ArrayList<WorkflowProcessInstancePersistence>());

            cache.put(processInstance.getProcessId(), newProcess);
        }
    }

    @Override
    public void bindProcessToNewScheme(ProcessInstance processInstance) throws ProcessNotFoundException {
        bindProcessToNewScheme(processInstance, false);
    }

    @Override
    public void bindProcessToNewScheme(ProcessInstance processInstance, boolean resetIsDeterminingParametersChanged) throws ProcessNotFoundException {
        IgniteCache<UUID, WorkflowProcessInstance> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_CACHE_CONFIG);
        WorkflowProcessInstance oldProcess = cache.get(processInstance.getProcessId());
        if (oldProcess == null) {
            throw new ProcessNotFoundException(processInstance.getProcessId());
        }

        oldProcess.setSchemeId(processInstance.getSchemeId());
        if (resetIsDeterminingParametersChanged) {
            oldProcess.setDeterminingParametersChanged(false);
        }
        cache.put(oldProcess.getId(), oldProcess);
    }

    @Override
    public void fillProcessParameters(ProcessInstance processInstance) {
        processInstance.addParameters(getProcessParameters(processInstance.getProcessId(), processInstance.getProcessScheme()));
    }

    @Override
    public void fillPersistedProcessParameters(ProcessInstance processInstance) {
        processInstance.addParameters(getPersistedProcessParameters(processInstance.getProcessId(), processInstance.getProcessScheme()));
    }

    @Override
    public void fillSystemProcessParameters(ProcessInstance processInstance) {
        processInstance.addParameters(getSystemProcessParameters(processInstance.getProcessId(), processInstance.getProcessScheme()));
    }

    @Override
    public void savePersistenceParameters(ProcessInstance processInstance) {
        List<ParameterDefinitionWithValue> parametersToPersistList = new ArrayList<>();
        for (ParameterDefinitionWithValue ptp : processInstance.getProcessParameters()) {
            if (ptp.getPurpose() == ParameterPurpose.Persistence) {
                ParameterDefinitionWithValue parameterDefinitionWithValue = new ParameterDefinitionWithValue();
                parameterDefinitionWithValue.setParameterDefinition(ptp);
                if (ptp.getType().equals(UnknownParameterType.class)) {
                    parameterDefinitionWithValue.setValue(ptp.getValue());
                } else {
                    parameterDefinitionWithValue.setValue(ParametersSerializer.serialize(ptp.getValue()));
                }
                parametersToPersistList.add(parameterDefinitionWithValue);
            }
        }

        IgniteCache<UUID, WorkflowProcessInstance> cache =
                ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_CACHE_CONFIG);
        WorkflowProcessInstance process = cache.get(processInstance.getProcessId());
        if (process.getPersistence() == null) {
            return;
        }
        for (final ParameterDefinitionWithValue parameterDefinitionWithValue : parametersToPersistList) {
            WorkflowProcessInstancePersistence persistence = CollectionUtil.singleOrDefault(process.getPersistence(),
                    new CollectionUtil.ItemCondition<WorkflowProcessInstancePersistence>() {
                        @Override
                        public boolean check(WorkflowProcessInstancePersistence pp) {
                            return pp.getParameterName().equals(parameterDefinitionWithValue.getName());
                        }
                    });

            if (persistence == null) {
                if (parameterDefinitionWithValue.getValue() instanceof String) {
                    persistence = new WorkflowProcessInstancePersistence();
                    persistence.setParameterName(parameterDefinitionWithValue.getName());
                    persistence.setValue((String) parameterDefinitionWithValue.getValue());
                    process.getPersistence().add(persistence);
                }
            } else {
                if (parameterDefinitionWithValue.getValue() instanceof String) {
                    persistence.setValue((String) parameterDefinitionWithValue.getValue());
                } else {
                    process.getPersistence().remove(persistence);
                }
            }
        }
        cache.put(process.getId(), process);
    }

    @Override
    public void setWorkflowInitialized(ProcessInstance processInstance) {
        setCustomStatus(processInstance.getProcessId(), ProcessStatus.Initialized, true);
    }

    @Override
    public void setWorkflowIdled(ProcessInstance processInstance) {
        setCustomStatus(processInstance.getProcessId(), ProcessStatus.Idled);
    }

    @Override
    public void setWorkflowRunning(ProcessInstance processInstance) {
        try (IgniteCache<UUID, WorkflowProcessInstanceStatus> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_STATUS_CACHE_CONFIG);
             Transaction tx = ignite.transactions().txStart()) {
            WorkflowProcessInstanceStatus instanceStatus = cache.get(processInstance.getProcessId());
            if (instanceStatus == null) {
                throw new StatusNotDefinedException();
            }

            if (instanceStatus.getStatus() == ProcessStatus.Running.getId()) {
                throw new ImpossibleToSetStatusException();
            }

            instanceStatus.setStatus(ProcessStatus.Running.getId());
            cache.put(instanceStatus.getId(), instanceStatus);

            tx.commit();
        }
    }

    @Override
    public void setWorkflowFinalized(ProcessInstance processInstance) {
        setCustomStatus(processInstance.getProcessId(), ProcessStatus.Finalized);
    }

    @Override
    public void setWorkflowTerminated(ProcessInstance processInstance, ErrorLevel level, String errorMessage) {
        setCustomStatus(processInstance.getProcessId(), ProcessStatus.Terminated);
    }

    @Override
    public void resetWorkflowRunning() {
        SqlQuery<UUID, WorkflowProcessInstanceStatus> sqlQuery =
                new SqlQuery<UUID, WorkflowProcessInstanceStatus>(WorkflowProcessInstanceStatus.class, "status = ?")
                        .setArgs(ProcessStatus.Running.getId());

        try (IgniteCache<UUID, WorkflowProcessInstanceStatus> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_STATUS_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowProcessInstanceStatus>> cursor = cache.query(sqlQuery)) {
            Map<UUID, WorkflowProcessInstanceStatus> map = new HashMap<>();
            for (Cache.Entry<UUID, WorkflowProcessInstanceStatus> entry : cursor) {
                entry.getValue().setStatus(ProcessStatus.Idled.getId());
                map.put(entry.getKey(), entry.getValue());
            }
            cache.putAll(map);
        }
    }

    @Override
    public void updatePersistenceState(ProcessInstance processInstance, TransitionDefinition transition) {
        ParameterDefinitionWithValue paramIdentityId = processInstance.getParameter(DefaultDefinitions.PARAMETER_IDENTITY_ID);
        ParameterDefinitionWithValue paramImpIdentityId = processInstance.getParameter(DefaultDefinitions.PARAMETER_IMPERSONATED_IDENTITY_ID);

        String identityId = paramIdentityId == null ? StringUtil.EMPTY : (String) paramIdentityId.getValue();
        String impIdentityId = paramImpIdentityId == null ? identityId : (String) paramImpIdentityId.getValue();

        try (IgniteCache<UUID, WorkflowProcessInstance> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_CACHE_CONFIG)) {
            WorkflowProcessInstance instance = cache.get(processInstance.getProcessId());

            if (instance != null) {
                if (!StringUtil.isNullOrEmpty(transition.getTo().getState())) {
                    instance.setStateName(transition.getTo().getState());
                }

                instance.setActivityName(transition.getTo().getName());
                instance.setPreviousActivity(transition.getFrom().getName());

                if (!StringUtil.isNullOrEmpty(transition.getFrom().getState())) {
                    instance.setPreviousState(transition.getFrom().getState());
                }

                if (transition.getClassifier() == TransitionClassifier.Direct) {
                    instance.setPreviousActivityForDirect(transition.getFrom().getName());

                    if (!StringUtil.isNullOrEmpty(transition.getFrom().getState())) {
                        instance.setPreviousStateForDirect(transition.getFrom().getState());
                    }
                } else if (transition.getClassifier() == TransitionClassifier.Reverse) {
                    instance.setPreviousActivityForReverse(transition.getFrom().getName());
                    if (!StringUtil.isNullOrEmpty(transition.getFrom().getState())) {
                        instance.setPreviousStateForReverse(transition.getFrom().getState());
                    }
                }

                instance.setParentProcessId(processInstance.getParentProcessId());
                instance.setRootProcessId(processInstance.getRootProcessId());

                cache.put(instance.getId(), instance);
            }

            WorkflowProcessTransitionHistory history = new WorkflowProcessTransitionHistory();
            history.setActorIdentityId(impIdentityId);
            history.setExecutorIdentityId(identityId);
            history.setId(UUID.randomUUID());
            history.setFinalised(transition.getTo().isFinalActivity());
            history.setProcessId(processInstance.getProcessId());
            history.setFromActivityName(transition.getFrom().getName());
            history.setFromStateName(transition.getFrom().getState());
            history.setToActivityName(transition.getTo().getName());
            history.setToStateName(transition.getTo().getState());
            history.setTransitionClassifier(transition.getClassifier().toString());
            history.setTransitionTime(runtime.getRuntimeDateTimeNow());
            history.setTriggerName(StringUtil.isNullOrEmpty(processInstance.getExecutedTimer()) ? processInstance.getCurrentCommand() : processInstance.getExecutedTimer());

            try (IgniteCache<UUID, WorkflowProcessTransitionHistory> transitionHistoryCache = ignite.getOrCreateCache(WORKFLOW_PROCESS_TRANSITION_HISTORY_CACHE_CONFIG)) {
                transitionHistoryCache.put(history.getId(), history);
            }
        }
    }

    @Override
    public boolean isProcessExists(UUID processId) {
        IgniteCache<UUID, WorkflowProcessInstance> cache =
                ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_CACHE_CONFIG);
        return cache.containsKey(processId);
    }

    @Override
    public ProcessStatus getInstanceStatus(UUID processId) {
        try (IgniteCache<UUID, WorkflowProcessInstanceStatus> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_STATUS_CACHE_CONFIG)) {
            WorkflowProcessInstanceStatus instance = cache.get(processId);
            if (instance == null) {
                return ProcessStatus.NotFound;
            }

            return ProcessStatus.getById(instance.getStatus());
        }
    }

    private void setCustomStatus(UUID processId, ProcessStatus status) {
        setCustomStatus(processId, status, false);
    }

    private void setCustomStatus(UUID processId, ProcessStatus status, boolean createIfNotDefined) {
        try (IgniteCache<UUID, WorkflowProcessInstanceStatus> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_STATUS_CACHE_CONFIG)) {
            WorkflowProcessInstanceStatus instanceStatus = cache.get(processId);
            if (instanceStatus == null) {
                if (!createIfNotDefined) {
                    throw new StatusNotDefinedException();
                }

                instanceStatus = new WorkflowProcessInstanceStatus();
                instanceStatus.setId(processId);
                instanceStatus.setStatus(ProcessStatus.Initialized.getId());
            } else {
                instanceStatus.setStatus(status.getId());
            }

            cache.put(instanceStatus.getId(), instanceStatus);
        }
    }

    private Collection<ParameterDefinitionWithValue> getProcessParameters(UUID processId, ProcessDefinition processDefinition) {
        List<ParameterDefinitionWithValue> parameters = new ArrayList<>(processDefinition.getParameters().size());
        parameters.addAll(getPersistedProcessParameters(processId, processDefinition));
        parameters.addAll(getSystemProcessParameters(processId, processDefinition));
        return parameters;
    }

    private Collection<ParameterDefinitionWithValue> getSystemProcessParameters(UUID processId,
                                                                                ProcessDefinition processDefinition) {
        WorkflowProcessInstance processInstance = getProcessInstance(processId);

        Collection<ParameterDefinition> systemParameters = CollectionUtil.where(processDefinition.getParameters(),
                new CollectionUtil.ItemCondition<ParameterDefinition>() {
                    @Override
                    public boolean check(ParameterDefinition p) {
                        return p.getPurpose() == ParameterPurpose.System;
                    }
                });

        List<ParameterDefinitionWithValue> parameters = new ArrayList<>(systemParameters.size());
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_PROCESS_ID, processId));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_PREVIOUS_STATE, processInstance.getPreviousState()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_CURRENT_STATE, processInstance.getStateName()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_PREVIOUS_STATE_FOR_DIRECT, processInstance.getPreviousStateForDirect()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_PREVIOUS_STATE_FOR_REVERSE, processInstance.getPreviousStateForReverse()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_PREVIOUS_ACTIVITY, processInstance.getPreviousActivity()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_CURRENT_ACTIVITY, processInstance.getActivityName()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_PREVIOUS_ACTIVITY_FOR_DIRECT, processInstance.getPreviousActivityForDirect()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_PREVIOUS_ACTIVITY_FOR_REVERSE, processInstance.getPreviousActivityForReverse()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_SCHEME_CODE, processDefinition.getName()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_SCHEME_ID, processInstance.getSchemeId()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_IS_PRE_EXECUTION, Boolean.FALSE));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_PARENT_PROCESS_ID, processInstance.getParentProcessId()));
        parameters.add(createSystemParameter(systemParameters, DefaultDefinitions.PARAMETER_ROOT_PROCESS_ID, processInstance.getRootProcessId()));

        return parameters;
    }

    private ParameterDefinitionWithValue createSystemParameter(Collection<ParameterDefinition> systemParameters,
                                                               final ParameterDefinition definition,
                                                               Object value) {
        ParameterDefinition parameterDefinition = CollectionUtil.single(systemParameters,
                new CollectionUtil.ItemCondition<ParameterDefinition>() {
                    @Override
                    public boolean check(ParameterDefinition sp) {
                        return sp.getName().equals(definition.getName());
                    }
                });
        return ParameterDefinitionWithValue.create(parameterDefinition, value);
    }

    private Collection<ParameterDefinitionWithValue> getPersistedProcessParameters(UUID processId, ProcessDefinition processDefinition) {
        Collection<ParameterDefinition> persistenceParameters = processDefinition.getPersistenceParameters();
        List<ParameterDefinitionWithValue> parameters = new ArrayList<>();

        WorkflowProcessInstance workflowProcessInstance = getProcessInstance(processId);
        List<WorkflowProcessInstancePersistence> persistedParameters = workflowProcessInstance.getPersistence();
        if (persistedParameters == null) {
            return parameters;
        }

        for (final WorkflowProcessInstancePersistence persistedParameter : persistedParameters) {
            ParameterDefinition parameterDefinition = CollectionUtil.firstOrDefault(persistenceParameters,
                    new CollectionUtil.ItemCondition<ParameterDefinition>() {
                        @Override
                        public boolean check(ParameterDefinition p) {
                            return p.getName().equals(persistedParameter.getParameterName());
                        }
                    });
            if (parameterDefinition == null) {
                parameterDefinition = ParameterDefinition.create(persistedParameter.getParameterName(),
                        UnknownParameterType.class, ParameterPurpose.Persistence);
                parameters.add(ParameterDefinitionWithValue.create(parameterDefinition, persistedParameter.getValue()));
            } else {
                parameters.add(ParameterDefinitionWithValue.create(parameterDefinition,
                        ParametersSerializer.deserialize(persistedParameter.getValue(), parameterDefinition.getType())));
            }
        }

        return parameters;
    }

    private WorkflowProcessInstance getProcessInstance(UUID processId) throws ProcessNotFoundException {
        IgniteCache<UUID, WorkflowProcessInstance> cache =
                ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_CACHE_CONFIG);

        WorkflowProcessInstance processInstance = cache.get(processId);
        if (processInstance == null) {
            throw new ProcessNotFoundException(processId);
        }

        return processInstance;
    }

    @Override
    public void deleteProcess(UUID[] processIds) {
        for (UUID processId : processIds) {
            deleteProcess(processId);
        }
    }

    @Override
    public <T> void saveGlobalParameter(final String type, final String name, T value) {
        WorkflowGlobalParameter parameter = getWorkflowGlobalParameter(type, name);
        if (parameter == null) {
            parameter = new WorkflowGlobalParameter();
            parameter.setId(UUID.randomUUID());
            parameter.setType(type);
            parameter.setName(name);
        }

        parameter.setValue(JsonConvert.serializeObject(value));

        try (IgniteCache<UUID, WorkflowGlobalParameter> cache = ignite.getOrCreateCache(WORKFLOW_GLOBAL_PARAMETER_CACHE_CONFIG)) {
            cache.put(parameter.getId(), parameter);
        }
    }

    @Override
    public <T> T loadGlobalParameter(String type, String name, Class<T> cls) {
        WorkflowGlobalParameter parameter = getWorkflowGlobalParameter(type, name);
        if (parameter == null) {
            return null;
        }

        return JsonConvert.deserializeObject(parameter.getValue(), cls);
    }

    @Override
    public <T> T loadGlobalParameter(String type, String name, TypeReference<T> typeReference) {
        WorkflowGlobalParameter parameter = getWorkflowGlobalParameter(type, name);
        if (parameter == null) {
            return null;
        }

        return JsonConvert.deserializeTypeObject(parameter.getValue(), typeReference);
    }

    @Override
    public <T> Collection<T> loadGlobalParameters(final String type, final Class<T> cls) {
        SqlQuery<UUID, WorkflowGlobalParameter> sqlQuery = new SqlQuery<UUID, WorkflowGlobalParameter>(WorkflowGlobalParameter.class, "type = ?")
                .setArgs(type);

        List<T> result = new ArrayList<>();
        try (IgniteCache<UUID, WorkflowGlobalParameter> cache = ignite.getOrCreateCache(WORKFLOW_GLOBAL_PARAMETER_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowGlobalParameter>> cursor = cache.query(sqlQuery)) {
            for (Cache.Entry<UUID, WorkflowGlobalParameter> entry : cursor) {
                result.add(JsonConvert.deserializeObject(entry.getValue().getValue(), cls));
            }
        }
        return result;
    }

    @Override
    public void deleteGlobalParameters(String type) {
        deleteGlobalParameters(type, null);
    }

    @Override
    public void deleteGlobalParameters(String type, String name) {
        final SqlQuery<UUID, WorkflowGlobalParameter> sqlQuery;
        if (StringUtil.isNullOrEmpty(name)) {
            sqlQuery = new SqlQuery<UUID, WorkflowGlobalParameter>(WorkflowGlobalParameter.class, "type = ?")
                    .setArgs(type);
        } else {
            sqlQuery = new SqlQuery<UUID, WorkflowGlobalParameter>(WorkflowGlobalParameter.class, "name = ? and name = ?")
                    .setArgs(type, name);
        }

        try (IgniteCache<UUID, WorkflowGlobalParameter> cache = ignite.getOrCreateCache(WORKFLOW_GLOBAL_PARAMETER_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowGlobalParameter>> cursor = cache.query(sqlQuery)) {
            Set<UUID> ids = new HashSet<>();
            for (Cache.Entry<UUID, WorkflowGlobalParameter> entry : cursor) {
                ids.add(entry.getKey());
            }

            cache.removeAll(ids);
        }
    }

    @Override
    public void deleteProcess(final UUID processId) {
        IgniteCache<UUID, WorkflowProcessInstance> cacheInstance =
                ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_CACHE_CONFIG);
        cacheInstance.remove(processId);

        IgniteCache<UUID, WorkflowProcessInstanceStatus> cacheStatus = ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_STATUS_CACHE_CONFIG);
        cacheStatus.remove(processId);

        IgniteCache<UUID, WorkflowProcessTransitionHistory> cacheTransition = ignite.getOrCreateCache(WORKFLOW_PROCESS_TRANSITION_HISTORY_CACHE_CONFIG);
        SqlQuery<UUID, WorkflowProcessTransitionHistory> sqlTransition =
                new SqlQuery<UUID, WorkflowProcessTransitionHistory>(WorkflowProcessTransitionHistory.class, "processId = ?")
                        .setArgs(processId);
        try (QueryCursor<Cache.Entry<UUID, WorkflowProcessTransitionHistory>> cursor = cacheTransition.query(sqlTransition)) {
            for (Cache.Entry<UUID, WorkflowProcessTransitionHistory> entry : cursor) {
                cacheTransition.remove(entry.getKey());
            }
        }

        SqlQuery<UUID, WorkflowProcessTimer> sql = new SqlQuery<UUID, WorkflowProcessTimer>(WorkflowProcessTimer.class, "processId = ?")
                .setArgs(processId);
        IgniteCache<UUID, WorkflowProcessTimer> cacheTimer = ignite.getOrCreateCache(WORKFLOW_PROCESS_TIMER_CACHE_CONFIG);

        try (QueryCursor<Cache.Entry<UUID, WorkflowProcessTimer>> cursor = cacheTimer.query(sql)) {
            for (Cache.Entry<UUID, WorkflowProcessTimer> entry : cursor) {
                cacheTimer.remove(entry.getKey());
            }
        }
    }

    @Override
    public void registerTimer(UUID processId, String name, Date nextExecutionDateTime, boolean notOverrideIfExists) {
        SqlQuery<UUID, WorkflowProcessTimer> sqlQuery = new SqlQuery<UUID, WorkflowProcessTimer>(WorkflowProcessTimer.class, "processId = ? and name = ?")
                .setArgs(processId, name);
        try (IgniteCache<UUID, WorkflowProcessTimer> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_TIMER_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowProcessTimer>> cursor = cache.query(sqlQuery)) {
            Cache.Entry<UUID, WorkflowProcessTimer> entry = CollectionUtil.firstOrDefault(cursor);
            final WorkflowProcessTimer timer;
            if (entry == null) {
                timer = new WorkflowProcessTimer();
                timer.setId(UUID.randomUUID());
                timer.setName(name);
                timer.setNextExecutionDateTime(nextExecutionDateTime);
                timer.setProcessId(processId);
                timer.setIgnore(false);
            } else {
                timer = entry.getValue();
                timer.setNextExecutionDateTime(nextExecutionDateTime);
            }
            cache.put(timer.getId(), timer);
        }
    }

    @Override
    public void clearTimers(UUID processId, Collection<String> timersIgnoreList) {
        final List<Object> params = new ArrayList<>();
        params.add(processId);
        String sql = "processId = ?";
        if (!timersIgnoreList.isEmpty()) {
            String placeHolders = StringUtil.join(",", CollectionUtil.select(timersIgnoreList,
                    new CollectionUtil.ItemTransformer<String, String>() {
                        @Override
                        public String transform(String s) {
                            params.add(s);
                            return "?";
                        }
                    }));
            sql += " and name not in (" + placeHolders + ")";
        }

        SqlQuery<UUID, WorkflowProcessTimer> sqlQuery =
                new SqlQuery<UUID, WorkflowProcessTimer>(WorkflowProcessTimer.class, sql)
                        .setArgs(params.toArray());

        try (IgniteCache<UUID, WorkflowProcessTimer> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_TIMER_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowProcessTimer>> cursor = cache.query(sqlQuery)) {
            Set<UUID> ids = new HashSet<>();
            for (Cache.Entry<UUID, WorkflowProcessTimer> entry : cursor) {
                ids.add(entry.getKey());
            }

            cache.removeAll(ids);
        }
    }

    @Override
    public void clearTimersIgnore() {
        SqlQuery<UUID, WorkflowProcessTimer> sqlQuery = new SqlQuery<UUID, WorkflowProcessTimer>(WorkflowProcessTimer.class, "ignore = ?")
                .setArgs(Boolean.TRUE);

        try (IgniteCache<UUID, WorkflowProcessTimer> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_TIMER_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowProcessTimer>> cursor = cache.query(sqlQuery)) {
            Map<UUID, WorkflowProcessTimer> map = new HashMap<>();
            for (Cache.Entry<UUID, WorkflowProcessTimer> entry : cursor) {
                entry.getValue().setIgnore(false);
                map.put(entry.getKey(), entry.getValue());
            }

            cache.putAll(map);
        }
    }

    @Override
    public void clearTimerIgnore(UUID timerId) {
        try (IgniteCache<UUID, WorkflowProcessTimer> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_TIMER_CACHE_CONFIG)) {
            WorkflowProcessTimer timer = cache.get(timerId);
            timer.setIgnore(false);
            cache.put(timer.getId(), timer);
        }
    }

    @Override
    public void clearTimer(UUID timerId) {
        try (IgniteCache<UUID, WorkflowProcessTimer> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_TIMER_CACHE_CONFIG)) {
            cache.remove(timerId);
        }
    }

    @Override
    public Date getCloseExecutionDateTime() {
        SqlQuery<UUID, WorkflowProcessTimer> sql = new SqlQuery<>(WorkflowProcessTimer.class, "order by nextExecutionDateTime");
        sql.setPageSize(1);
        try (IgniteCache<UUID, WorkflowProcessTimer> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_TIMER_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowProcessTimer>> cursor = cache.query(sql)) {
            Cache.Entry<UUID, WorkflowProcessTimer> entry = CollectionUtil.firstOrDefault(cursor);
            if (entry == null) {
                return null;
            }
            return entry.getValue().getNextExecutionDateTime();
        }
    }

    @Override
    public List<TimerToExecute> getTimersToExecute() {
        Date now = runtime.getRuntimeDateTimeNow();

        SqlQuery<UUID, WorkflowProcessTimer> sqlQuery =
                new SqlQuery<UUID, WorkflowProcessTimer>(WorkflowProcessTimer.class, "ignore = ? and nextExecutionDateTime <= ?")
                        .setArgs(Boolean.FALSE, now);

        try (IgniteCache<UUID, WorkflowProcessTimer> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_TIMER_CACHE_CONFIG);
             Transaction tx = ignite.transactions().txStart();
             QueryCursor<Cache.Entry<UUID, WorkflowProcessTimer>> cursor = cache.query(sqlQuery)) {
            Map<UUID, WorkflowProcessTimer> map = new HashMap<>();
            List<TimerToExecute> result = new ArrayList<>();
            for (Cache.Entry<UUID, WorkflowProcessTimer> entry : cursor) {
                WorkflowProcessTimer t = entry.getValue();
                TimerToExecute timerToExecute = new TimerToExecute();
                timerToExecute.setName(t.getName());
                timerToExecute.setProcessId(t.getProcessId());
                timerToExecute.setTimerId(t.getId());
                result.add(timerToExecute);

                t.setIgnore(true);
                map.put(entry.getKey(), entry.getValue());
            }

            cache.putAll(map);
            tx.commit();
            return result;
        }
    }

    @Override
    public List<ProcessHistoryItem> getProcessHistory(UUID processId) {
        SqlQuery<UUID, WorkflowProcessTransitionHistory> sqlQuery =
                new SqlQuery<>(WorkflowProcessTransitionHistory.class, "processId = ?");

        List<ProcessHistoryItem> result = new ArrayList<>();
        try (IgniteCache<UUID, WorkflowProcessTransitionHistory> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_TRANSITION_HISTORY_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowProcessTransitionHistory>> cursor = cache.query(sqlQuery)) {
            for (Cache.Entry<UUID, WorkflowProcessTransitionHistory> entry : cursor) {
                WorkflowProcessTransitionHistory item = entry.getValue();
                ProcessHistoryItem processHistoryItem = new ProcessHistoryItem();
                processHistoryItem.setActorIdentityId(item.getActorIdentityId());
                processHistoryItem.setExecutorIdentityId(item.getExecutorIdentityId());
                processHistoryItem.setFromActivityName(item.getFromActivityName());
                processHistoryItem.setFromStateName(item.getFromStateName());
                processHistoryItem.setFinalised(item.isFinalised());
                processHistoryItem.setProcessId(item.getProcessId());
                processHistoryItem.setToActivityName(item.getToActivityName());
                processHistoryItem.setToStateName(item.getToStateName());
                processHistoryItem.setTransitionClassifier(TransitionClassifier.valueOf(item.getTransitionClassifier()));
                processHistoryItem.setTransitionTime(item.getTransitionTime());
                processHistoryItem.setTriggerName(item.getTriggerName());
                result.add(processHistoryItem);
            }
        }
        return result;
    }

    @Override
    public SchemeDefinition<Document> getProcessSchemeByProcessId(UUID processId) {
        try (IgniteCache<UUID, WorkflowProcessInstance> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_INSTANCE_CACHE_CONFIG)) {
            WorkflowProcessInstance processInstance = cache.get(processId);
            if (processInstance == null) {
                throw new ProcessNotFoundException(processId);
            }
            if (processInstance.getSchemeId() == null) {
                throw new SchemeNotFoundException(processId, SchemeLocation.WorkflowProcessInstance);
            }
            SchemeDefinition<Document> schemeDefinition = getProcessSchemeBySchemeId(processInstance.getSchemeId());
            schemeDefinition.setDeterminingParametersChanged(processInstance.isDeterminingParametersChanged());
            return schemeDefinition;
        }
    }

    @Override
    public SchemeDefinition<Document> getProcessSchemeBySchemeId(UUID schemeId) {
        try (IgniteCache<UUID, WorkflowProcessScheme> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_SCHEME_CACHE_CONFIG)) {
            WorkflowProcessScheme processScheme = cache.get(schemeId);
            if (processScheme == null || StringUtil.isNullOrEmpty(processScheme.getScheme())) {
                throw new SchemeNotFoundException(schemeId, SchemeLocation.WorkflowProcessScheme);
            }
            return convertToSchemeDefinition(processScheme);
        }
    }

    @Override
    public SchemeDefinition<Document> getProcessSchemeWithParameters(String schemeCode, final String definingParameters,
                                                                     UUID rootSchemeId, boolean ignoreObsolete) {
        String hash = HashHelper.generateStringHash(definingParameters);

        List<Object> params = new ArrayList<>();
        params.add(schemeCode);
        params.add(hash);

        String baseSql = "schemeCode = ? and definingParametersHash = ?";
        if (rootSchemeId != null) {
            baseSql += " and rootSchemeId = ?";
            params.add(rootSchemeId);
        }

        final SqlQuery<UUID, WorkflowProcessScheme> sqlQuery;
        if (ignoreObsolete) {
            params.add(Boolean.FALSE);
            sqlQuery = new SqlQuery<UUID, WorkflowProcessScheme>(WorkflowProcessScheme.class, baseSql + " and obsolete = ?")
                    .setArgs(params.toArray());
        } else {
            sqlQuery = new SqlQuery<UUID, WorkflowProcessScheme>(WorkflowProcessScheme.class, baseSql)
                    .setArgs(params.toArray());
        }

        try (IgniteCache<UUID, WorkflowProcessScheme> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_SCHEME_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowProcessScheme>> cursor = cache.query(sqlQuery)) {
            Cache.Entry<UUID, WorkflowProcessScheme> entry = CollectionUtil.firstOrDefault(cursor,
                    new CollectionUtil.ItemCondition<Cache.Entry<UUID, WorkflowProcessScheme>>() {
                        @Override
                        public boolean check(Cache.Entry<UUID, WorkflowProcessScheme> entry) {
                            return entry.getValue().getDefiningParameters().equals(definingParameters);
                        }
                    });

            if (entry != null) {
                return convertToSchemeDefinition(entry.getValue());
            }
            throw new SchemeNotFoundException(schemeCode, SchemeLocation.WorkflowProcessScheme, definingParameters);
        }
    }

    @Override
    public void setSchemeIsObsolete(String schemeCode, Map<String, Object> parameters) {
        String definingParameters = DefiningParametersSerializer.serialize(parameters);
        String definingParametersHash = HashHelper.generateStringHash(definingParameters);

        SqlQuery<UUID, WorkflowProcessScheme> sql = new SqlQuery<UUID, WorkflowProcessScheme>(WorkflowProcessScheme.class,
                "(schemeCode = ? or rootSchemeCode = ?) and definingParametersHash = ?")
                .setArgs(schemeCode, schemeCode, definingParametersHash);
        setSchemeIsObsolete(sql);
    }

    @Override
    public void setSchemeIsObsolete(String schemeCode) {
        SqlQuery<UUID, WorkflowProcessScheme> sql =
                new SqlQuery<UUID, WorkflowProcessScheme>(WorkflowProcessScheme.class, "schemeCode = ? or rootSchemeCode = ?")
                        .setArgs(schemeCode, schemeCode);
        setSchemeIsObsolete(sql);
    }

    @Override
    public void saveScheme(SchemeDefinition<Document> scheme) {
        final String definingParameters = scheme.getDefiningParameters();
        String definingParametersHash = HashHelper.generateStringHash(definingParameters);

        SqlQuery<UUID, WorkflowProcessScheme> sqlQuery = new SqlQuery<UUID, WorkflowProcessScheme>(WorkflowProcessScheme.class,
                "definingParametersHash = ? and schemeCode = ? and obsolete = ?")
                .setArgs(definingParametersHash, scheme.getSchemeCode(), scheme.isObsolete());
        try (IgniteCache<UUID, WorkflowProcessScheme> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_SCHEME_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowProcessScheme>> cursor = cache.query(
                     sqlQuery)) {
            if (cursor.iterator().hasNext()) {
                throw new SchemeAlreadyExistsException(scheme.getSchemeCode(), SchemeLocation.WorkflowProcessScheme,
                        scheme.getDefiningParameters());
            }

            WorkflowProcessScheme newProcessScheme = new WorkflowProcessScheme();
            newProcessScheme.setId(scheme.getId());
            newProcessScheme.setDefiningParameters(definingParameters);
            newProcessScheme.setDefiningParametersHash(definingParametersHash);
            newProcessScheme.setScheme(DocumentUtil.serialize(scheme.getScheme()));
            newProcessScheme.setSchemeCode(scheme.getSchemeCode());
            newProcessScheme.setRootSchemeCode(scheme.getRootSchemeCode());
            newProcessScheme.setRootSchemeId(scheme.getRootSchemeId());
            newProcessScheme.setAllowedActivities(JsonConvert.serializeObject(scheme.getAllowedActivities()));
            newProcessScheme.setStartingTransition(scheme.getStartingTransition());
            newProcessScheme.setObsolete(scheme.isObsolete());

            cache.put(newProcessScheme.getId(), newProcessScheme);
        }
    }

    @Override
    public void saveScheme(String schemaCode, String scheme) {
        IgniteCache<String, WorkflowScheme> cache = ignite.getOrCreateCache(WORKFLOW_SCHEME_CACHE_CONFIG);

        WorkflowScheme wfScheme = cache.get(schemaCode);
        if (wfScheme == null) {
            wfScheme = new WorkflowScheme();
            wfScheme.setCode(schemaCode);
            wfScheme.setScheme(scheme);
            cache.put(schemaCode, wfScheme);
        } else {
            wfScheme.setScheme(scheme);
            cache.put(schemaCode, wfScheme);
        }
    }

    @Override
    public Document getScheme(String code) {
        IgniteCache<String, WorkflowScheme> cache = ignite.getOrCreateCache(WORKFLOW_SCHEME_CACHE_CONFIG);
        WorkflowScheme workflowScheme = cache.get(code);
        if (workflowScheme == null || StringUtil.isNullOrEmpty(workflowScheme.getScheme())) {
            throw new SchemeNotFoundException(code, SchemeLocation.WorkflowScheme);
        }

        return DocumentUtil.parse(workflowScheme.getScheme());
    }

    @Override
    public Document generate(String schemeCode, UUID schemeId, Map<String, Object> parameters) {
        if (!parameters.isEmpty()) {
            throw new IllegalArgumentException("Parameters not supported");
        }

        return getScheme(schemeCode);
    }

    private void setSchemeIsObsolete(SqlQuery<UUID, WorkflowProcessScheme> sql) {
        try (IgniteCache<UUID, WorkflowProcessScheme> cache = ignite.getOrCreateCache(WORKFLOW_PROCESS_SCHEME_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowProcessScheme>> cursor = cache.query(sql)) {
            Map<UUID, WorkflowProcessScheme> map = new HashMap<>();
            for (Cache.Entry<UUID, WorkflowProcessScheme> entry : cursor) {
                entry.getValue().setObsolete(true);
                map.put(entry.getKey(), entry.getValue());
            }
            cache.putAll(map);
        }
    }

    private WorkflowGlobalParameter getWorkflowGlobalParameter(final String type, final String name) {
        SqlQuery<UUID, WorkflowGlobalParameter> sql = new SqlQuery<UUID, WorkflowGlobalParameter>(WorkflowGlobalParameter.class, "name = ? and type = ?")
                .setArgs(name, type);

        try (IgniteCache<UUID, WorkflowGlobalParameter> cache = ignite.getOrCreateCache(WORKFLOW_GLOBAL_PARAMETER_CACHE_CONFIG);
             QueryCursor<Cache.Entry<UUID, WorkflowGlobalParameter>> cursor = cache.query(sql)) {
            Cache.Entry<UUID, WorkflowGlobalParameter> entry = CollectionUtil.firstOrDefault(cursor);
            return entry == null ? null : entry.getValue();
        }
    }

    private SchemeDefinition<Document> convertToSchemeDefinition(WorkflowProcessScheme workflowProcessScheme) {
        List<String> allowedActivities = null;
        if (workflowProcessScheme.getAllowedActivities() != null) {
            //noinspection unchecked
            allowedActivities = JsonConvert.deserializeObject(workflowProcessScheme.getAllowedActivities(), List.class);
        }

        return new SchemeDefinition<>(workflowProcessScheme.getId(), workflowProcessScheme.getRootSchemeId(),
                workflowProcessScheme.getSchemeCode(), workflowProcessScheme.getRootSchemeCode(),
                DocumentUtil.parse(workflowProcessScheme.getScheme()), workflowProcessScheme.isObsolete(), false,
                allowedActivities,
                workflowProcessScheme.getStartingTransition(),
                workflowProcessScheme.getDefiningParameters());
    }
}