package io.youi.net

import io.circe.Decoder.Result
import io.circe.{Decoder, DecodingFailure, Encoder, HCursor, Json}

case class ContentType(`type`: String,
                       subType: String,
                       extras: Map[String, String] = Map.empty) {
  def this(mimeType: String) = {
    this(mimeType.substring(0, mimeType.indexOf('/')), mimeType.substring(mimeType.indexOf('/') + 1))
  }

  lazy val mimeType: String = s"${`type`}/$subType"
  lazy val boundary: String = extras("boundary")
  lazy val charSet: String = extras("charset")
  lazy val start: String = extras("start")
  lazy val outputString: String = {
    val b = new StringBuilder(mimeType)
    extras.foreach {
      case (key, value) => b.append(s"; $key=$value")
    }
    b.toString()
  }
  lazy val extension: Option[String] = ContentType.mimeType2Extensions.getOrElse(`type`, Nil).headOption

  def withExtra(key: String, value: String): ContentType = copy(extras = extras + (key -> value))
  def withCharSet(charSet: String): ContentType = withExtra("charset", charSet)

  def is(contentType: ContentType): Boolean = contentType.mimeType == mimeType

  override def equals(obj: scala.Any): Boolean = obj match {
    case ct: ContentType => ct.is(this)
    case _ => false
  }

  override def hashCode(): Int = mimeType.hashCode

  override def toString: String = outputString
}

object ContentType {
  private var extension2MimeType = Map.empty[String, String]
  private var mimeType2Extensions = Map.empty[String, List[String]]
  private def map(extension: String, mimeType: String): Unit = {
    extension2MimeType += extension -> mimeType
    val extensions = mimeType2Extensions.getOrElse(mimeType, Nil)
    mimeType2Extensions += mimeType -> (extensions ::: List(extension))
  }

  lazy val `application/andrew-inset`: ContentType = ContentType("application", "andrew-inset")
  lazy val `application/applixware`: ContentType = ContentType("application", "applixware")
  lazy val `application/atomcat+xml`: ContentType = ContentType("application", "atomcat+xml")
  lazy val `application/atomsvc+xml`: ContentType = ContentType("application", "atomsvc+xml")
  lazy val `application/atom+xml`: ContentType = ContentType("application", "atom+xml")
  lazy val `application/ccxml+xml`: ContentType = ContentType("application", "ccxml+xml")
  lazy val `application/cdmi-capability`: ContentType = ContentType("application", "cdmi-capability")
  lazy val `application/cdmi-container`: ContentType = ContentType("application", "cdmi-container")
  lazy val `application/cdmi-domain`: ContentType = ContentType("application", "cdmi-domain")
  lazy val `application/cdmi-object`: ContentType = ContentType("application", "cdmi-object")
  lazy val `application/cdmi-queue`: ContentType = ContentType("application", "cdmi-queue")
  lazy val `application/cu-seeme`: ContentType = ContentType("application", "cu-seeme")
  lazy val `application/davmount+xml`: ContentType = ContentType("application", "davmount+xml")
  lazy val `application/docbook+xml`: ContentType = ContentType("application", "docbook+xml")
  lazy val `application/dssc+der`: ContentType = ContentType("application", "dssc+der")
  lazy val `application/dssc+xml`: ContentType = ContentType("application", "dssc+xml")
  lazy val `application/ecmascript`: ContentType = ContentType("application", "ecmascript")
  lazy val `application/emma+xml`: ContentType = ContentType("application", "emma+xml")
  lazy val `application/epub+zip`: ContentType = ContentType("application", "epub+zip")
  lazy val `application/exi`: ContentType = ContentType("application", "exi")
  lazy val `application/font-tdpfr`: ContentType = ContentType("application", "font-tdpfr")
  lazy val `application/font-woff`: ContentType = ContentType("application", "font-woff")
  lazy val `application/gml+xml`: ContentType = ContentType("application", "gml+xml")
  lazy val `application/gpx+xml`: ContentType = ContentType("application", "gpx+xml")
  lazy val `application/gxf`: ContentType = ContentType("application", "gxf")
  lazy val `application/hyperstudio`: ContentType = ContentType("application", "hyperstudio")
  lazy val `application/inkml+xml`: ContentType = ContentType("application", "inkml+xml")
  lazy val `application/ipfix`: ContentType = ContentType("application", "ipfix")
  lazy val `application/java-archive`: ContentType = ContentType("application", "java-archive")
  lazy val `application/javascript`: ContentType = ContentType("application", "javascript")
  lazy val `application/java-serialized-object`: ContentType = ContentType("application", "java-serialized-object")
  lazy val `application/java-vm`: ContentType = ContentType("application", "java-vm")
  lazy val `application/json`: ContentType = ContentType("application", "json")
  lazy val `application/jsonml+json`: ContentType = ContentType("application", "jsonml+json")
  lazy val `application/lost+xml`: ContentType = ContentType("application", "lost+xml")
  lazy val `application/mac-binhex40`: ContentType = ContentType("application", "mac-binhex40")
  lazy val `application/mac-compactpro`: ContentType = ContentType("application", "mac-compactpro")
  lazy val `application/mads+xml`: ContentType = ContentType("application", "mads+xml")
  lazy val `application/marc`: ContentType = ContentType("application", "marc")
  lazy val `application/marcxml+xml`: ContentType = ContentType("application", "marcxml+xml")
  lazy val `application/mathematica`: ContentType = ContentType("application", "mathematica")
  lazy val `application/mathml+xml`: ContentType = ContentType("application", "mathml+xml")
  lazy val `application/mbox`: ContentType = ContentType("application", "mbox")
  lazy val `application/mediaservercontrol+xml`: ContentType = ContentType("application", "mediaservercontrol+xml")
  lazy val `application/metalink4+xml`: ContentType = ContentType("application", "metalink4+xml")
  lazy val `application/metalink+xml`: ContentType = ContentType("application", "metalink+xml")
  lazy val `application/mets+xml`: ContentType = ContentType("application", "mets+xml")
  lazy val `application/mods+xml`: ContentType = ContentType("application", "mods+xml")
  lazy val `application/mp21`: ContentType = ContentType("application", "mp21")
  lazy val `application/mp4`: ContentType = ContentType("application", "mp4")
  lazy val `application/msword`: ContentType = ContentType("application", "msword")
  lazy val `application/mxf`: ContentType = ContentType("application", "mxf")
  lazy val `application/octet-stream`: ContentType = ContentType("application", "octet-stream")
  lazy val `application/oda`: ContentType = ContentType("application", "oda")
  lazy val `application/oebps-package+xml`: ContentType = ContentType("application", "oebps-package+xml")
  lazy val `application/ogg`: ContentType = ContentType("application", "ogg")
  lazy val `application/omdoc+xml`: ContentType = ContentType("application", "omdoc+xml")
  lazy val `application/onenote`: ContentType = ContentType("application", "onenote")
  lazy val `application/oxps`: ContentType = ContentType("application", "oxps")
  lazy val `application/patch-ops-error+xml`: ContentType = ContentType("application", "patch-ops-error+xml")
  lazy val `application/pdf`: ContentType = ContentType("application", "pdf")
  lazy val `application/pgp-encrypted`: ContentType = ContentType("application", "pgp-encrypted")
  lazy val `application/pgp-signature`: ContentType = ContentType("application", "pgp-signature")
  lazy val `application/pics-rules`: ContentType = ContentType("application", "pics-rules")
  lazy val `application/pkcs10`: ContentType = ContentType("application", "pkcs10")
  lazy val `application/pkcs7-mime`: ContentType = ContentType("application", "pkcs7-mime")
  lazy val `application/pkcs7-signature`: ContentType = ContentType("application", "pkcs7-signature")
  lazy val `application/pkcs8`: ContentType = ContentType("application", "pkcs8")
  lazy val `application/pkix-attr-cert`: ContentType = ContentType("application", "pkix-attr-cert")
  lazy val `application/pkix-cert`: ContentType = ContentType("application", "pkix-cert")
  lazy val `application/pkixcmp`: ContentType = ContentType("application", "pkixcmp")
  lazy val `application/pkix-crl`: ContentType = ContentType("application", "pkix-crl")
  lazy val `application/pkix-pkipath`: ContentType = ContentType("application", "pkix-pkipath")
  lazy val `application/pls+xml`: ContentType = ContentType("application", "pls+xml")
  lazy val `application/postscript`: ContentType = ContentType("application", "postscript")
  lazy val `application/prs.cww`: ContentType = ContentType("application", "prs.cww")
  lazy val `application/pskc+xml`: ContentType = ContentType("application", "pskc+xml")
  lazy val `application/rdf+xml`: ContentType = ContentType("application", "rdf+xml")
  lazy val `application/reginfo+xml`: ContentType = ContentType("application", "reginfo+xml")
  lazy val `application/relax-ng-compact-syntax`: ContentType = ContentType("application", "relax-ng-compact-syntax")
  lazy val `application/resource-lists-diff+xml`: ContentType = ContentType("application", "resource-lists-diff+xml")
  lazy val `application/resource-lists+xml`: ContentType = ContentType("application", "resource-lists+xml")
  lazy val `application/rls-services+xml`: ContentType = ContentType("application", "rls-services+xml")
  lazy val `application/rpki-ghostbusters`: ContentType = ContentType("application", "rpki-ghostbusters")
  lazy val `application/rpki-manifest`: ContentType = ContentType("application", "rpki-manifest")
  lazy val `application/rpki-roa`: ContentType = ContentType("application", "rpki-roa")
  lazy val `application/rsd+xml`: ContentType = ContentType("application", "rsd+xml")
  lazy val `application/rss+xml`: ContentType = ContentType("application", "rss+xml")
  lazy val `application/rtf`: ContentType = ContentType("application", "rtf")
  lazy val `application/sbml+xml`: ContentType = ContentType("application", "sbml+xml")
  lazy val `application/scvp-cv-request`: ContentType = ContentType("application", "scvp-cv-request")
  lazy val `application/scvp-cv-response`: ContentType = ContentType("application", "scvp-cv-response")
  lazy val `application/scvp-vp-request`: ContentType = ContentType("application", "scvp-vp-request")
  lazy val `application/scvp-vp-response`: ContentType = ContentType("application", "scvp-vp-response")
  lazy val `application/sdp`: ContentType = ContentType("application", "sdp")
  lazy val `application/set-payment-initiation`: ContentType = ContentType("application", "set-payment-initiation")
  lazy val `application/set-registration-initiation`: ContentType = ContentType("application", "set-registration-initiation")
  lazy val `application/shf+xml`: ContentType = ContentType("application", "shf+xml")
  lazy val `application/smil+xml`: ContentType = ContentType("application", "smil+xml")
  lazy val `application/sparql-query`: ContentType = ContentType("application", "sparql-query")
  lazy val `application/sparql-results+xml`: ContentType = ContentType("application", "sparql-results+xml")
  lazy val `application/srgs`: ContentType = ContentType("application", "srgs")
  lazy val `application/srgs+xml`: ContentType = ContentType("application", "srgs+xml")
  lazy val `application/sru+xml`: ContentType = ContentType("application", "sru+xml")
  lazy val `application/ssdl+xml`: ContentType = ContentType("application", "ssdl+xml")
  lazy val `application/ssml+xml`: ContentType = ContentType("application", "ssml+xml")
  lazy val `application/tei+xml`: ContentType = ContentType("application", "tei+xml")
  lazy val `application/thraud+xml`: ContentType = ContentType("application", "thraud+xml")
  lazy val `application/timestamped-data`: ContentType = ContentType("application", "timestamped-data")
  lazy val `application/vnd.3gpp2.tcap`: ContentType = ContentType("application", "vnd.3gpp2.tcap")
  lazy val `application/vnd.3gpp.pic-bw-large`: ContentType = ContentType("application", "vnd.3gpp.pic-bw-large")
  lazy val `application/vnd.3gpp.pic-bw-small`: ContentType = ContentType("application", "vnd.3gpp.pic-bw-small")
  lazy val `application/vnd.3gpp.pic-bw-var`: ContentType = ContentType("application", "vnd.3gpp.pic-bw-var")
  lazy val `application/vnd.3m.post-it-notes`: ContentType = ContentType("application", "vnd.3m.post-it-notes")
  lazy val `application/vnd.accpac.simply.aso`: ContentType = ContentType("application", "vnd.accpac.simply.aso")
  lazy val `application/vnd.accpac.simply.imp`: ContentType = ContentType("application", "vnd.accpac.simply.imp")
  lazy val `application/vnd.acucobol`: ContentType = ContentType("application", "vnd.acucobol")
  lazy val `application/vnd.acucorp`: ContentType = ContentType("application", "vnd.acucorp")
  lazy val `application/vnd.adobe.air-application-installer-package+zip`: ContentType = ContentType("application", "vnd.adobe.air-application-installer-package+zip")
  lazy val `application/vnd.adobe.formscentral.fcdt`: ContentType = ContentType("application", "vnd.adobe.formscentral.fcdt")
  lazy val `application/vnd.adobe.fxp`: ContentType = ContentType("application", "vnd.adobe.fxp")
  lazy val `application/vnd.adobe.xdp+xml`: ContentType = ContentType("application", "vnd.adobe.xdp+xml")
  lazy val `application/vnd.adobe.xfdf`: ContentType = ContentType("application", "vnd.adobe.xfdf")
  lazy val `application/vnd.ahead.space`: ContentType = ContentType("application", "vnd.ahead.space")
  lazy val `application/vnd.airzip.filesecure.azf`: ContentType = ContentType("application", "vnd.airzip.filesecure.azf")
  lazy val `application/vnd.airzip.filesecure.azs`: ContentType = ContentType("application", "vnd.airzip.filesecure.azs")
  lazy val `application/vnd.amazon.ebook`: ContentType = ContentType("application", "vnd.amazon.ebook")
  lazy val `application/vnd.americandynamics.acc`: ContentType = ContentType("application", "vnd.americandynamics.acc")
  lazy val `application/vnd.amiga.ami`: ContentType = ContentType("application", "vnd.amiga.ami")
  lazy val `application/vnd.android.package-archive`: ContentType = ContentType("application", "vnd.android.package-archive")
  lazy val `application/vnd.anser-web-certificate-issue-initiation`: ContentType = ContentType("application", "vnd.anser-web-certificate-issue-initiation")
  lazy val `application/vnd.anser-web-funds-transfer-initiation`: ContentType = ContentType("application", "vnd.anser-web-funds-transfer-initiation")
  lazy val `application/vnd.antix.game-component`: ContentType = ContentType("application", "vnd.antix.game-component")
  lazy val `application/vnd.apple.installer+xml`: ContentType = ContentType("application", "vnd.apple.installer+xml")
  lazy val `application/vnd.apple.mpegurl`: ContentType = ContentType("application", "vnd.apple.mpegurl")
  lazy val `application/vnd.aristanetworks.swi`: ContentType = ContentType("application", "vnd.aristanetworks.swi")
  lazy val `application/vnd.astraea-software.iota`: ContentType = ContentType("application", "vnd.astraea-software.iota")
  lazy val `application/vnd.audiograph`: ContentType = ContentType("application", "vnd.audiograph")
  lazy val `application/vnd.blueice.multipass`: ContentType = ContentType("application", "vnd.blueice.multipass")
  lazy val `application/vnd.bmi`: ContentType = ContentType("application", "vnd.bmi")
  lazy val `application/vnd.businessobjects`: ContentType = ContentType("application", "vnd.businessobjects")
  lazy val `application/vnd.chemdraw+xml`: ContentType = ContentType("application", "vnd.chemdraw+xml")
  lazy val `application/vnd.chipnuts.karaoke-mmd`: ContentType = ContentType("application", "vnd.chipnuts.karaoke-mmd")
  lazy val `application/vnd.cinderella`: ContentType = ContentType("application", "vnd.cinderella")
  lazy val `application/vnd.claymore`: ContentType = ContentType("application", "vnd.claymore")
  lazy val `application/vnd.cloanto.rp9`: ContentType = ContentType("application", "vnd.cloanto.rp9")
  lazy val `application/vnd.clonk.c4group`: ContentType = ContentType("application", "vnd.clonk.c4group")
  lazy val `application/vnd.cluetrust.cartomobile-config`: ContentType = ContentType("application", "vnd.cluetrust.cartomobile-config")
  lazy val `application/vnd.cluetrust.cartomobile-config-pkg`: ContentType = ContentType("application", "vnd.cluetrust.cartomobile-config-pkg")
  lazy val `application/vnd.commonspace`: ContentType = ContentType("application", "vnd.commonspace")
  lazy val `application/vnd.contact.cmsg`: ContentType = ContentType("application", "vnd.contact.cmsg")
  lazy val `application/vnd.cosmocaller`: ContentType = ContentType("application", "vnd.cosmocaller")
  lazy val `application/vnd.crick.clicker`: ContentType = ContentType("application", "vnd.crick.clicker")
  lazy val `application/vnd.crick.clicker.keyboard`: ContentType = ContentType("application", "vnd.crick.clicker.keyboard")
  lazy val `application/vnd.crick.clicker.palette`: ContentType = ContentType("application", "vnd.crick.clicker.palette")
  lazy val `application/vnd.crick.clicker.template`: ContentType = ContentType("application", "vnd.crick.clicker.template")
  lazy val `application/vnd.crick.clicker.wordbank`: ContentType = ContentType("application", "vnd.crick.clicker.wordbank")
  lazy val `application/vnd.criticaltools.wbs+xml`: ContentType = ContentType("application", "vnd.criticaltools.wbs+xml")
  lazy val `application/vnd.ctc-posml`: ContentType = ContentType("application", "vnd.ctc-posml")
  lazy val `application/vnd.cups-ppd`: ContentType = ContentType("application", "vnd.cups-ppd")
  lazy val `application/vnd.curl.car`: ContentType = ContentType("application", "vnd.curl.car")
  lazy val `application/vnd.curl.pcurl`: ContentType = ContentType("application", "vnd.curl.pcurl")
  lazy val `application/vnd.dart`: ContentType = ContentType("application", "vnd.dart")
  lazy val `application/vnd.data-vision.rdz`: ContentType = ContentType("application", "vnd.data-vision.rdz")
  lazy val `application/vnd.dece.data`: ContentType = ContentType("application", "vnd.dece.data")
  lazy val `application/vnd.dece.ttml+xml`: ContentType = ContentType("application", "vnd.dece.ttml+xml")
  lazy val `application/vnd.dece.unspecified`: ContentType = ContentType("application", "vnd.dece.unspecified")
  lazy val `application/vnd.dece.zip`: ContentType = ContentType("application", "vnd.dece.zip")
  lazy val `application/vnd.denovo.fcselayout-link`: ContentType = ContentType("application", "vnd.denovo.fcselayout-link")
  lazy val `application/vnd.dna`: ContentType = ContentType("application", "vnd.dna")
  lazy val `application/vnd.dolby.mlp`: ContentType = ContentType("application", "vnd.dolby.mlp")
  lazy val `application/vnd.dpgraph`: ContentType = ContentType("application", "vnd.dpgraph")
  lazy val `application/vnd.dreamfactory`: ContentType = ContentType("application", "vnd.dreamfactory")
  lazy val `application/vnd.ds-keypoint`: ContentType = ContentType("application", "vnd.ds-keypoint")
  lazy val `application/vnd.dvb.ait`: ContentType = ContentType("application", "vnd.dvb.ait")
  lazy val `application/vnd.dvb.service`: ContentType = ContentType("application", "vnd.dvb.service")
  lazy val `application/vnd.dynageo`: ContentType = ContentType("application", "vnd.dynageo")
  lazy val `application/vnd.ecowin.chart`: ContentType = ContentType("application", "vnd.ecowin.chart")
  lazy val `application/vnd.enliven`: ContentType = ContentType("application", "vnd.enliven")
  lazy val `application/vnd.epson.esf`: ContentType = ContentType("application", "vnd.epson.esf")
  lazy val `application/vnd.epson.msf`: ContentType = ContentType("application", "vnd.epson.msf")
  lazy val `application/vnd.epson.quickanime`: ContentType = ContentType("application", "vnd.epson.quickanime")
  lazy val `application/vnd.epson.salt`: ContentType = ContentType("application", "vnd.epson.salt")
  lazy val `application/vnd.epson.ssf`: ContentType = ContentType("application", "vnd.epson.ssf")
  lazy val `application/vnd.eszigno3+xml`: ContentType = ContentType("application", "vnd.eszigno3+xml")
  lazy val `application/vnd.ezpix-album`: ContentType = ContentType("application", "vnd.ezpix-album")
  lazy val `application/vnd.ezpix-package`: ContentType = ContentType("application", "vnd.ezpix-package")
  lazy val `application/vnd.fdf`: ContentType = ContentType("application", "vnd.fdf")
  lazy val `application/vnd.fdsn.mseed`: ContentType = ContentType("application", "vnd.fdsn.mseed")
  lazy val `application/vnd.fdsn.seed`: ContentType = ContentType("application", "vnd.fdsn.seed")
  lazy val `application/vnd.flographit`: ContentType = ContentType("application", "vnd.flographit")
  lazy val `application/vnd.fluxtime.clip`: ContentType = ContentType("application", "vnd.fluxtime.clip")
  lazy val `application/vnd.framemaker`: ContentType = ContentType("application", "vnd.framemaker")
  lazy val `application/vnd.frogans.fnc`: ContentType = ContentType("application", "vnd.frogans.fnc")
  lazy val `application/vnd.frogans.ltf`: ContentType = ContentType("application", "vnd.frogans.ltf")
  lazy val `application/vnd.fsc.weblaunch`: ContentType = ContentType("application", "vnd.fsc.weblaunch")
  lazy val `application/vnd.fujitsu.oasys`: ContentType = ContentType("application", "vnd.fujitsu.oasys")
  lazy val `application/vnd.fujitsu.oasys2`: ContentType = ContentType("application", "vnd.fujitsu.oasys2")
  lazy val `application/vnd.fujitsu.oasys3`: ContentType = ContentType("application", "vnd.fujitsu.oasys3")
  lazy val `application/vnd.fujitsu.oasysgp`: ContentType = ContentType("application", "vnd.fujitsu.oasysgp")
  lazy val `application/vnd.fujitsu.oasysprs`: ContentType = ContentType("application", "vnd.fujitsu.oasysprs")
  lazy val `application/vnd.fujixerox.ddd`: ContentType = ContentType("application", "vnd.fujixerox.ddd")
  lazy val `application/vnd.fujixerox.docuworks`: ContentType = ContentType("application", "vnd.fujixerox.docuworks")
  lazy val `application/vnd.fujixerox.docuworks.binder`: ContentType = ContentType("application", "vnd.fujixerox.docuworks.binder")
  lazy val `application/vnd.fuzzysheet`: ContentType = ContentType("application", "vnd.fuzzysheet")
  lazy val `application/vnd.genomatix.tuxedo`: ContentType = ContentType("application", "vnd.genomatix.tuxedo")
  lazy val `application/vnd.geogebra.file`: ContentType = ContentType("application", "vnd.geogebra.file")
  lazy val `application/vnd.geogebra.tool`: ContentType = ContentType("application", "vnd.geogebra.tool")
  lazy val `application/vnd.geometry-explorer`: ContentType = ContentType("application", "vnd.geometry-explorer")
  lazy val `application/vnd.geonext`: ContentType = ContentType("application", "vnd.geonext")
  lazy val `application/vnd.geoplan`: ContentType = ContentType("application", "vnd.geoplan")
  lazy val `application/vnd.geospace`: ContentType = ContentType("application", "vnd.geospace")
  lazy val `application/vnd.gmx`: ContentType = ContentType("application", "vnd.gmx")
  lazy val `application/vnd.google-earth.kml+xml`: ContentType = ContentType("application", "vnd.google-earth.kml+xml")
  lazy val `application/vnd.google-earth.kmz`: ContentType = ContentType("application", "vnd.google-earth.kmz")
  lazy val `application/vnd.grafeq`: ContentType = ContentType("application", "vnd.grafeq")
  lazy val `application/vnd.groove-account`: ContentType = ContentType("application", "vnd.groove-account")
  lazy val `application/vnd.groove-help`: ContentType = ContentType("application", "vnd.groove-help")
  lazy val `application/vnd.groove-identity-message`: ContentType = ContentType("application", "vnd.groove-identity-message")
  lazy val `application/vnd.groove-injector`: ContentType = ContentType("application", "vnd.groove-injector")
  lazy val `application/vnd.groove-tool-message`: ContentType = ContentType("application", "vnd.groove-tool-message")
  lazy val `application/vnd.groove-tool-template`: ContentType = ContentType("application", "vnd.groove-tool-template")
  lazy val `application/vnd.groove-vcard`: ContentType = ContentType("application", "vnd.groove-vcard")
  lazy val `application/vnd.hal+xml`: ContentType = ContentType("application", "vnd.hal+xml")
  lazy val `application/vnd.handheld-entertainment+xml`: ContentType = ContentType("application", "vnd.handheld-entertainment+xml")
  lazy val `application/vnd.hbci`: ContentType = ContentType("application", "vnd.hbci")
  lazy val `application/vnd.hhe.lesson-player`: ContentType = ContentType("application", "vnd.hhe.lesson-player")
  lazy val `application/vnd.hp-hpgl`: ContentType = ContentType("application", "vnd.hp-hpgl")
  lazy val `application/vnd.hp-hpid`: ContentType = ContentType("application", "vnd.hp-hpid")
  lazy val `application/vnd.hp-hps`: ContentType = ContentType("application", "vnd.hp-hps")
  lazy val `application/vnd.hp-jlyt`: ContentType = ContentType("application", "vnd.hp-jlyt")
  lazy val `application/vnd.hp-pcl`: ContentType = ContentType("application", "vnd.hp-pcl")
  lazy val `application/vnd.hp-pclxl`: ContentType = ContentType("application", "vnd.hp-pclxl")
  lazy val `application/vnd.hydrostatix.sof-data`: ContentType = ContentType("application", "vnd.hydrostatix.sof-data")
  lazy val `application/vnd.ibm.minipay`: ContentType = ContentType("application", "vnd.ibm.minipay")
  lazy val `application/vnd.ibm.modcap`: ContentType = ContentType("application", "vnd.ibm.modcap")
  lazy val `application/vnd.ibm.rights-management`: ContentType = ContentType("application", "vnd.ibm.rights-management")
  lazy val `application/vnd.ibm.secure-container`: ContentType = ContentType("application", "vnd.ibm.secure-container")
  lazy val `application/vnd.iccprofile`: ContentType = ContentType("application", "vnd.iccprofile")
  lazy val `application/vnd.igloader`: ContentType = ContentType("application", "vnd.igloader")
  lazy val `application/vnd.immervision-ivp`: ContentType = ContentType("application", "vnd.immervision-ivp")
  lazy val `application/vnd.immervision-ivu`: ContentType = ContentType("application", "vnd.immervision-ivu")
  lazy val `application/vnd.insors.igm`: ContentType = ContentType("application", "vnd.insors.igm")
  lazy val `application/vnd.intercon.formnet`: ContentType = ContentType("application", "vnd.intercon.formnet")
  lazy val `application/vnd.intergeo`: ContentType = ContentType("application", "vnd.intergeo")
  lazy val `application/vnd.intu.qbo`: ContentType = ContentType("application", "vnd.intu.qbo")
  lazy val `application/vnd.intu.qfx`: ContentType = ContentType("application", "vnd.intu.qfx")
  lazy val `application/vnd.ipunplugged.rcprofile`: ContentType = ContentType("application", "vnd.ipunplugged.rcprofile")
  lazy val `application/vnd.irepository.package+xml`: ContentType = ContentType("application", "vnd.irepository.package+xml")
  lazy val `application/vnd.isac.fcs`: ContentType = ContentType("application", "vnd.isac.fcs")
  lazy val `application/vnd.is-xpr`: ContentType = ContentType("application", "vnd.is-xpr")
  lazy val `application/vnd.jam`: ContentType = ContentType("application", "vnd.jam")
  lazy val `application/vnd.jcp.javame.midlet-rms`: ContentType = ContentType("application", "vnd.jcp.javame.midlet-rms")
  lazy val `application/vnd.jisp`: ContentType = ContentType("application", "vnd.jisp")
  lazy val `application/vnd.joost.joda-archive`: ContentType = ContentType("application", "vnd.joost.joda-archive")
  lazy val `application/vnd.kahootz`: ContentType = ContentType("application", "vnd.kahootz")
  lazy val `application/vnd.kde.karbon`: ContentType = ContentType("application", "vnd.kde.karbon")
  lazy val `application/vnd.kde.kchart`: ContentType = ContentType("application", "vnd.kde.kchart")
  lazy val `application/vnd.kde.kformula`: ContentType = ContentType("application", "vnd.kde.kformula")
  lazy val `application/vnd.kde.kivio`: ContentType = ContentType("application", "vnd.kde.kivio")
  lazy val `application/vnd.kde.kontour`: ContentType = ContentType("application", "vnd.kde.kontour")
  lazy val `application/vnd.kde.kpresenter`: ContentType = ContentType("application", "vnd.kde.kpresenter")
  lazy val `application/vnd.kde.kspread`: ContentType = ContentType("application", "vnd.kde.kspread")
  lazy val `application/vnd.kde.kword`: ContentType = ContentType("application", "vnd.kde.kword")
  lazy val `application/vnd.kenameaapp`: ContentType = ContentType("application", "vnd.kenameaapp")
  lazy val `application/vnd.kidspiration`: ContentType = ContentType("application", "vnd.kidspiration")
  lazy val `application/vnd.kinar`: ContentType = ContentType("application", "vnd.kinar")
  lazy val `application/vnd.koan`: ContentType = ContentType("application", "vnd.koan")
  lazy val `application/vnd.kodak-descriptor`: ContentType = ContentType("application", "vnd.kodak-descriptor")
  lazy val `application/vnd.las.las+xml`: ContentType = ContentType("application", "vnd.las.las+xml")
  lazy val `application/vnd.llamagraphics.life-balance.desktop`: ContentType = ContentType("application", "vnd.llamagraphics.life-balance.desktop")
  lazy val `application/vnd.llamagraphics.life-balance.exchange+xml`: ContentType = ContentType("application", "vnd.llamagraphics.life-balance.exchange+xml")
  lazy val `application/vnd.lotus-1-2-3`: ContentType = ContentType("application", "vnd.lotus-1-2-3")
  lazy val `application/vnd.lotus-approach`: ContentType = ContentType("application", "vnd.lotus-approach")
  lazy val `application/vnd.lotus-freelance`: ContentType = ContentType("application", "vnd.lotus-freelance")
  lazy val `application/vnd.lotus-notes`: ContentType = ContentType("application", "vnd.lotus-notes")
  lazy val `application/vnd.lotus-organizer`: ContentType = ContentType("application", "vnd.lotus-organizer")
  lazy val `application/vnd.lotus-screencam`: ContentType = ContentType("application", "vnd.lotus-screencam")
  lazy val `application/vnd.lotus-wordpro`: ContentType = ContentType("application", "vnd.lotus-wordpro")
  lazy val `application/vnd.macports.portpkg`: ContentType = ContentType("application", "vnd.macports.portpkg")
  lazy val `application/vnd.mcd`: ContentType = ContentType("application", "vnd.mcd")
  lazy val `application/vnd.medcalcdata`: ContentType = ContentType("application", "vnd.medcalcdata")
  lazy val `application/vnd.mediastation.cdkey`: ContentType = ContentType("application", "vnd.mediastation.cdkey")
  lazy val `application/vnd.mfer`: ContentType = ContentType("application", "vnd.mfer")
  lazy val `application/vnd.mfmp`: ContentType = ContentType("application", "vnd.mfmp")
  lazy val `application/vnd.micrografx.flo`: ContentType = ContentType("application", "vnd.micrografx.flo")
  lazy val `application/vnd.micrografx.igx`: ContentType = ContentType("application", "vnd.micrografx.igx")
  lazy val `application/vnd.mif`: ContentType = ContentType("application", "vnd.mif")
  lazy val `application/vnd.mobius.daf`: ContentType = ContentType("application", "vnd.mobius.daf")
  lazy val `application/vnd.mobius.dis`: ContentType = ContentType("application", "vnd.mobius.dis")
  lazy val `application/vnd.mobius.mbk`: ContentType = ContentType("application", "vnd.mobius.mbk")
  lazy val `application/vnd.mobius.mqy`: ContentType = ContentType("application", "vnd.mobius.mqy")
  lazy val `application/vnd.mobius.msl`: ContentType = ContentType("application", "vnd.mobius.msl")
  lazy val `application/vnd.mobius.plc`: ContentType = ContentType("application", "vnd.mobius.plc")
  lazy val `application/vnd.mobius.txf`: ContentType = ContentType("application", "vnd.mobius.txf")
  lazy val `application/vnd.mophun.application`: ContentType = ContentType("application", "vnd.mophun.application")
  lazy val `application/vnd.mophun.certificate`: ContentType = ContentType("application", "vnd.mophun.certificate")
  lazy val `application/vnd.mozilla.xul+xml`: ContentType = ContentType("application", "vnd.mozilla.xul+xml")
  lazy val `application/vnd.ms-artgalry`: ContentType = ContentType("application", "vnd.ms-artgalry")
  lazy val `application/vnd.ms-cab-compressed`: ContentType = ContentType("application", "vnd.ms-cab-compressed")
  lazy val `application/vnd.mseq`: ContentType = ContentType("application", "vnd.mseq")
  lazy val `application/vnd.ms-excel`: ContentType = ContentType("application", "vnd.ms-excel")
  lazy val `application/vnd.ms-excel.addin.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-excel.addin.macroenabled.12")
  lazy val `application/vnd.ms-excel.sheet.binary.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-excel.sheet.binary.macroenabled.12")
  lazy val `application/vnd.ms-excel.sheet.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-excel.sheet.macroenabled.12")
  lazy val `application/vnd.ms-excel.template.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-excel.template.macroenabled.12")
  lazy val `application/vnd.ms-fontobject`: ContentType = ContentType("application", "vnd.ms-fontobject")
  lazy val `application/vnd.ms-htmlhelp`: ContentType = ContentType("application", "vnd.ms-htmlhelp")
  lazy val `application/vnd.ms-ims`: ContentType = ContentType("application", "vnd.ms-ims")
  lazy val `application/vnd.ms-lrm`: ContentType = ContentType("application", "vnd.ms-lrm")
  lazy val `application/vnd.ms-officetheme`: ContentType = ContentType("application", "vnd.ms-officetheme")
  lazy val `application/vnd.ms-pki.seccat`: ContentType = ContentType("application", "vnd.ms-pki.seccat")
  lazy val `application/vnd.ms-pki.stl`: ContentType = ContentType("application", "vnd.ms-pki.stl")
  lazy val `application/vnd.ms-powerpoint`: ContentType = ContentType("application", "vnd.ms-powerpoint")
  lazy val `application/vnd.ms-powerpoint.addin.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-powerpoint.addin.macroenabled.12")
  lazy val `application/vnd.ms-powerpoint.presentation.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-powerpoint.presentation.macroenabled.12")
  lazy val `application/vnd.ms-powerpoint.slide.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-powerpoint.slide.macroenabled.12")
  lazy val `application/vnd.ms-powerpoint.slideshow.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-powerpoint.slideshow.macroenabled.12")
  lazy val `application/vnd.ms-powerpoint.template.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-powerpoint.template.macroenabled.12")
  lazy val `application/vnd.ms-project`: ContentType = ContentType("application", "vnd.ms-project")
  lazy val `application/vnd.ms-word.document.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-word.document.macroenabled.12")
  lazy val `application/vnd.ms-word.template.macroenabled.12`: ContentType = ContentType("application", "vnd.ms-word.template.macroenabled.12")
  lazy val `application/vnd.ms-works`: ContentType = ContentType("application", "vnd.ms-works")
  lazy val `application/vnd.ms-wpl`: ContentType = ContentType("application", "vnd.ms-wpl")
  lazy val `application/vnd.ms-xpsdocument`: ContentType = ContentType("application", "vnd.ms-xpsdocument")
  lazy val `application/vnd.musician`: ContentType = ContentType("application", "vnd.musician")
  lazy val `application/vnd.muvee.style`: ContentType = ContentType("application", "vnd.muvee.style")
  lazy val `application/vnd.mynfc`: ContentType = ContentType("application", "vnd.mynfc")
  lazy val `application/vnd.neurolanguage.nlu`: ContentType = ContentType("application", "vnd.neurolanguage.nlu")
  lazy val `application/vnd.nitf`: ContentType = ContentType("application", "vnd.nitf")
  lazy val `application/vnd.noblenet-directory`: ContentType = ContentType("application", "vnd.noblenet-directory")
  lazy val `application/vnd.noblenet-sealer`: ContentType = ContentType("application", "vnd.noblenet-sealer")
  lazy val `application/vnd.noblenet-web`: ContentType = ContentType("application", "vnd.noblenet-web")
  lazy val `application/vnd.nokia.n-gage.data`: ContentType = ContentType("application", "vnd.nokia.n-gage.data")
  lazy val `application/vnd.nokia.n-gage.symbian.install`: ContentType = ContentType("application", "vnd.nokia.n-gage.symbian.install")
  lazy val `application/vnd.nokia.radio-preset`: ContentType = ContentType("application", "vnd.nokia.radio-preset")
  lazy val `application/vnd.nokia.radio-presets`: ContentType = ContentType("application", "vnd.nokia.radio-presets")
  lazy val `application/vnd.novadigm.edm`: ContentType = ContentType("application", "vnd.novadigm.edm")
  lazy val `application/vnd.novadigm.edx`: ContentType = ContentType("application", "vnd.novadigm.edx")
  lazy val `application/vnd.novadigm.ext`: ContentType = ContentType("application", "vnd.novadigm.ext")
  lazy val `application/vnd.oasis.opendocument.chart`: ContentType = ContentType("application", "vnd.oasis.opendocument.chart")
  lazy val `application/vnd.oasis.opendocument.chart-template`: ContentType = ContentType("application", "vnd.oasis.opendocument.chart-template")
  lazy val `application/vnd.oasis.opendocument.database`: ContentType = ContentType("application", "vnd.oasis.opendocument.database")
  lazy val `application/vnd.oasis.opendocument.formula`: ContentType = ContentType("application", "vnd.oasis.opendocument.formula")
  lazy val `application/vnd.oasis.opendocument.formula-template`: ContentType = ContentType("application", "vnd.oasis.opendocument.formula-template")
  lazy val `application/vnd.oasis.opendocument.graphics`: ContentType = ContentType("application", "vnd.oasis.opendocument.graphics")
  lazy val `application/vnd.oasis.opendocument.graphics-template`: ContentType = ContentType("application", "vnd.oasis.opendocument.graphics-template")
  lazy val `application/vnd.oasis.opendocument.image`: ContentType = ContentType("application", "vnd.oasis.opendocument.image")
  lazy val `application/vnd.oasis.opendocument.image-template`: ContentType = ContentType("application", "vnd.oasis.opendocument.image-template")
  lazy val `application/vnd.oasis.opendocument.presentation`: ContentType = ContentType("application", "vnd.oasis.opendocument.presentation")
  lazy val `application/vnd.oasis.opendocument.presentation-template`: ContentType = ContentType("application", "vnd.oasis.opendocument.presentation-template")
  lazy val `application/vnd.oasis.opendocument.spreadsheet`: ContentType = ContentType("application", "vnd.oasis.opendocument.spreadsheet")
  lazy val `application/vnd.oasis.opendocument.spreadsheet-template`: ContentType = ContentType("application", "vnd.oasis.opendocument.spreadsheet-template")
  lazy val `application/vnd.oasis.opendocument.text`: ContentType = ContentType("application", "vnd.oasis.opendocument.text")
  lazy val `application/vnd.oasis.opendocument.text-master`: ContentType = ContentType("application", "vnd.oasis.opendocument.text-master")
  lazy val `application/vnd.oasis.opendocument.text-template`: ContentType = ContentType("application", "vnd.oasis.opendocument.text-template")
  lazy val `application/vnd.oasis.opendocument.text-web`: ContentType = ContentType("application", "vnd.oasis.opendocument.text-web")
  lazy val `application/vnd.olpc-sugar`: ContentType = ContentType("application", "vnd.olpc-sugar")
  lazy val `application/vnd.oma.dd2+xml`: ContentType = ContentType("application", "vnd.oma.dd2+xml")
  lazy val `application/vnd.openofficeorg.extension`: ContentType = ContentType("application", "vnd.openofficeorg.extension")
  lazy val `application/vnd.openxmlformats-officedocument.presentationml.presentation`: ContentType = ContentType("application", "vnd.openxmlformats-officedocument.presentationml.presentation")
  lazy val `application/vnd.openxmlformats-officedocument.presentationml.slide`: ContentType = ContentType("application", "vnd.openxmlformats-officedocument.presentationml.slide")
  lazy val `application/vnd.openxmlformats-officedocument.presentationml.slideshow`: ContentType = ContentType("application", "vnd.openxmlformats-officedocument.presentationml.slideshow")
  lazy val `application/vnd.openxmlformats-officedocument.presentationml.template`: ContentType = ContentType("application", "vnd.openxmlformats-officedocument.presentationml.template")
  lazy val `application/vnd.openxmlformats-officedocument.spreadsheetml.sheet`: ContentType = ContentType("application", "vnd.openxmlformats-officedocument.spreadsheetml.sheet")
  lazy val `application/vnd.openxmlformats-officedocument.spreadsheetml.template`: ContentType = ContentType("application", "vnd.openxmlformats-officedocument.spreadsheetml.template")
  lazy val `application/vnd.openxmlformats-officedocument.wordprocessingml.document`: ContentType = ContentType("application", "vnd.openxmlformats-officedocument.wordprocessingml.document")
  lazy val `application/vnd.openxmlformats-officedocument.wordprocessingml.template`: ContentType = ContentType("application", "vnd.openxmlformats-officedocument.wordprocessingml.template")
  lazy val `application/vnd.osgeo.mapguide.package`: ContentType = ContentType("application", "vnd.osgeo.mapguide.package")
  lazy val `application/vnd.osgi.dp`: ContentType = ContentType("application", "vnd.osgi.dp")
  lazy val `application/vnd.osgi.subsystem`: ContentType = ContentType("application", "vnd.osgi.subsystem")
  lazy val `application/vnd.palm`: ContentType = ContentType("application", "vnd.palm")
  lazy val `application/vnd.pawaafile`: ContentType = ContentType("application", "vnd.pawaafile")
  lazy val `application/vnd.pg.format`: ContentType = ContentType("application", "vnd.pg.format")
  lazy val `application/vnd.pg.osasli`: ContentType = ContentType("application", "vnd.pg.osasli")
  lazy val `application/vnd.picsel`: ContentType = ContentType("application", "vnd.picsel")
  lazy val `application/vnd.pmi.widget`: ContentType = ContentType("application", "vnd.pmi.widget")
  lazy val `application/vnd.pocketlearn`: ContentType = ContentType("application", "vnd.pocketlearn")
  lazy val `application/vnd.powerbuilder6`: ContentType = ContentType("application", "vnd.powerbuilder6")
  lazy val `application/vnd.previewsystems.box`: ContentType = ContentType("application", "vnd.previewsystems.box")
  lazy val `application/vnd.proteus.magazine`: ContentType = ContentType("application", "vnd.proteus.magazine")
  lazy val `application/vnd.publishare-delta-tree`: ContentType = ContentType("application", "vnd.publishare-delta-tree")
  lazy val `application/vnd.pvi.ptid1`: ContentType = ContentType("application", "vnd.pvi.ptid1")
  lazy val `application/vnd.quark.quarkxpress`: ContentType = ContentType("application", "vnd.quark.quarkxpress")
  lazy val `application/vnd.realvnc.bed`: ContentType = ContentType("application", "vnd.realvnc.bed")
  lazy val `application/vnd.recordare.musicxml`: ContentType = ContentType("application", "vnd.recordare.musicxml")
  lazy val `application/vnd.recordare.musicxml+xml`: ContentType = ContentType("application", "vnd.recordare.musicxml+xml")
  lazy val `application/vnd.rig.cryptonote`: ContentType = ContentType("application", "vnd.rig.cryptonote")
  lazy val `application/vnd.rim.cod`: ContentType = ContentType("application", "vnd.rim.cod")
  lazy val `application/vnd.rn-realmedia`: ContentType = ContentType("application", "vnd.rn-realmedia")
  lazy val `application/vnd.rn-realmedia-vbr`: ContentType = ContentType("application", "vnd.rn-realmedia-vbr")
  lazy val `application/vnd.route66.link66+xml`: ContentType = ContentType("application", "vnd.route66.link66+xml")
  lazy val `application/vnd.sailingtracker.track`: ContentType = ContentType("application", "vnd.sailingtracker.track")
  lazy val `application/vnd.seemail`: ContentType = ContentType("application", "vnd.seemail")
  lazy val `application/vnd.sema`: ContentType = ContentType("application", "vnd.sema")
  lazy val `application/vnd.semd`: ContentType = ContentType("application", "vnd.semd")
  lazy val `application/vnd.semf`: ContentType = ContentType("application", "vnd.semf")
  lazy val `application/vnd.shana.informed.formdata`: ContentType = ContentType("application", "vnd.shana.informed.formdata")
  lazy val `application/vnd.shana.informed.formtemplate`: ContentType = ContentType("application", "vnd.shana.informed.formtemplate")
  lazy val `application/vnd.shana.informed.interchange`: ContentType = ContentType("application", "vnd.shana.informed.interchange")
  lazy val `application/vnd.shana.informed.package`: ContentType = ContentType("application", "vnd.shana.informed.package")
  lazy val `application/vnd.simtech-mindmapper`: ContentType = ContentType("application", "vnd.simtech-mindmapper")
  lazy val `application/vnd.smaf`: ContentType = ContentType("application", "vnd.smaf")
  lazy val `application/vnd.smart.teacher`: ContentType = ContentType("application", "vnd.smart.teacher")
  lazy val `application/vnd.solent.sdkm+xml`: ContentType = ContentType("application", "vnd.solent.sdkm+xml")
  lazy val `application/vnd.spotfire.dxp`: ContentType = ContentType("application", "vnd.spotfire.dxp")
  lazy val `application/vnd.spotfire.sfs`: ContentType = ContentType("application", "vnd.spotfire.sfs")
  lazy val `application/vnd.stardivision.calc`: ContentType = ContentType("application", "vnd.stardivision.calc")
  lazy val `application/vnd.stardivision.draw`: ContentType = ContentType("application", "vnd.stardivision.draw")
  lazy val `application/vnd.stardivision.impress`: ContentType = ContentType("application", "vnd.stardivision.impress")
  lazy val `application/vnd.stardivision.math`: ContentType = ContentType("application", "vnd.stardivision.math")
  lazy val `application/vnd.stardivision.writer`: ContentType = ContentType("application", "vnd.stardivision.writer")
  lazy val `application/vnd.stardivision.writer-global`: ContentType = ContentType("application", "vnd.stardivision.writer-global")
  lazy val `application/vnd.stepmania.package`: ContentType = ContentType("application", "vnd.stepmania.package")
  lazy val `application/vnd.stepmania.stepchart`: ContentType = ContentType("application", "vnd.stepmania.stepchart")
  lazy val `application/vnd.sun.xml.calc`: ContentType = ContentType("application", "vnd.sun.xml.calc")
  lazy val `application/vnd.sun.xml.calc.template`: ContentType = ContentType("application", "vnd.sun.xml.calc.template")
  lazy val `application/vnd.sun.xml.draw`: ContentType = ContentType("application", "vnd.sun.xml.draw")
  lazy val `application/vnd.sun.xml.draw.template`: ContentType = ContentType("application", "vnd.sun.xml.draw.template")
  lazy val `application/vnd.sun.xml.impress`: ContentType = ContentType("application", "vnd.sun.xml.impress")
  lazy val `application/vnd.sun.xml.impress.template`: ContentType = ContentType("application", "vnd.sun.xml.impress.template")
  lazy val `application/vnd.sun.xml.math`: ContentType = ContentType("application", "vnd.sun.xml.math")
  lazy val `application/vnd.sun.xml.writer`: ContentType = ContentType("application", "vnd.sun.xml.writer")
  lazy val `application/vnd.sun.xml.writer.global`: ContentType = ContentType("application", "vnd.sun.xml.writer.global")
  lazy val `application/vnd.sun.xml.writer.template`: ContentType = ContentType("application", "vnd.sun.xml.writer.template")
  lazy val `application/vnd.sus-calendar`: ContentType = ContentType("application", "vnd.sus-calendar")
  lazy val `application/vnd.svd`: ContentType = ContentType("application", "vnd.svd")
  lazy val `application/vnd.symbian.install`: ContentType = ContentType("application", "vnd.symbian.install")
  lazy val `application/vnd.syncml.dm+wbxml`: ContentType = ContentType("application", "vnd.syncml.dm+wbxml")
  lazy val `application/vnd.syncml.dm+xml`: ContentType = ContentType("application", "vnd.syncml.dm+xml")
  lazy val `application/vnd.syncml+xml`: ContentType = ContentType("application", "vnd.syncml+xml")
  lazy val `application/vnd.tao.intent-module-archive`: ContentType = ContentType("application", "vnd.tao.intent-module-archive")
  lazy val `application/vnd.tcpdump.pcap`: ContentType = ContentType("application", "vnd.tcpdump.pcap")
  lazy val `application/vnd.tmobile-livetv`: ContentType = ContentType("application", "vnd.tmobile-livetv")
  lazy val `application/vnd.trid.tpt`: ContentType = ContentType("application", "vnd.trid.tpt")
  lazy val `application/vnd.triscape.mxs`: ContentType = ContentType("application", "vnd.triscape.mxs")
  lazy val `application/vnd.trueapp`: ContentType = ContentType("application", "vnd.trueapp")
  lazy val `application/vnd.ufdl`: ContentType = ContentType("application", "vnd.ufdl")
  lazy val `application/vnd.uiq.theme`: ContentType = ContentType("application", "vnd.uiq.theme")
  lazy val `application/vnd.umajin`: ContentType = ContentType("application", "vnd.umajin")
  lazy val `application/vnd.unity`: ContentType = ContentType("application", "vnd.unity")
  lazy val `application/vnd.uoml+xml`: ContentType = ContentType("application", "vnd.uoml+xml")
  lazy val `application/vnd.vcx`: ContentType = ContentType("application", "vnd.vcx")
  lazy val `application/vnd.visio`: ContentType = ContentType("application", "vnd.visio")
  lazy val `application/vnd.visionary`: ContentType = ContentType("application", "vnd.visionary")
  lazy val `application/vnd.vsf`: ContentType = ContentType("application", "vnd.vsf")
  lazy val `application/vnd.wap.wbxml`: ContentType = ContentType("application", "vnd.wap.wbxml")
  lazy val `application/vnd.wap.wmlc`: ContentType = ContentType("application", "vnd.wap.wmlc")
  lazy val `application/vnd.wap.wmlscriptc`: ContentType = ContentType("application", "vnd.wap.wmlscriptc")
  lazy val `application/vnd.webturbo`: ContentType = ContentType("application", "vnd.webturbo")
  lazy val `application/vnd.wolfram.player`: ContentType = ContentType("application", "vnd.wolfram.player")
  lazy val `application/vnd.wordperfect`: ContentType = ContentType("application", "vnd.wordperfect")
  lazy val `application/vnd.wqd`: ContentType = ContentType("application", "vnd.wqd")
  lazy val `application/vnd.wt.stf`: ContentType = ContentType("application", "vnd.wt.stf")
  lazy val `application/vnd.xara`: ContentType = ContentType("application", "vnd.xara")
  lazy val `application/vnd.xfdl`: ContentType = ContentType("application", "vnd.xfdl")
  lazy val `application/vnd.yamaha.hv-dic`: ContentType = ContentType("application", "vnd.yamaha.hv-dic")
  lazy val `application/vnd.yamaha.hv-script`: ContentType = ContentType("application", "vnd.yamaha.hv-script")
  lazy val `application/vnd.yamaha.hv-voice`: ContentType = ContentType("application", "vnd.yamaha.hv-voice")
  lazy val `application/vnd.yamaha.openscoreformat`: ContentType = ContentType("application", "vnd.yamaha.openscoreformat")
  lazy val `application/vnd.yamaha.openscoreformat.osfpvg+xml`: ContentType = ContentType("application", "vnd.yamaha.openscoreformat.osfpvg+xml")
  lazy val `application/vnd.yamaha.smaf-audio`: ContentType = ContentType("application", "vnd.yamaha.smaf-audio")
  lazy val `application/vnd.yamaha.smaf-phrase`: ContentType = ContentType("application", "vnd.yamaha.smaf-phrase")
  lazy val `application/vnd.yellowriver-custom-menu`: ContentType = ContentType("application", "vnd.yellowriver-custom-menu")
  lazy val `application/vnd.zul`: ContentType = ContentType("application", "vnd.zul")
  lazy val `application/vnd.zzazz.deck+xml`: ContentType = ContentType("application", "vnd.zzazz.deck+xml")
  lazy val `application/voicexml+xml`: ContentType = ContentType("application", "voicexml+xml")
  lazy val `application/widget`: ContentType = ContentType("application", "widget")
  lazy val `application/winhlp`: ContentType = ContentType("application", "winhlp")
  lazy val `application/wsdl+xml`: ContentType = ContentType("application", "wsdl+xml")
  lazy val `application/wspolicy+xml`: ContentType = ContentType("application", "wspolicy+xml")
  lazy val `application/x-7z-compressed`: ContentType = ContentType("application", "x-7z-compressed")
  lazy val `application/x-abiword`: ContentType = ContentType("application", "x-abiword")
  lazy val `application/x-ace-compressed`: ContentType = ContentType("application", "x-ace-compressed")
  lazy val `application/xaml+xml`: ContentType = ContentType("application", "xaml+xml")
  lazy val `application/x-apple-diskimage`: ContentType = ContentType("application", "x-apple-diskimage")
  lazy val `application/x-authorware-bin`: ContentType = ContentType("application", "x-authorware-bin")
  lazy val `application/x-authorware-map`: ContentType = ContentType("application", "x-authorware-map")
  lazy val `application/x-authorware-seg`: ContentType = ContentType("application", "x-authorware-seg")
  lazy val `application/x-bcpio`: ContentType = ContentType("application", "x-bcpio")
  lazy val `application/x-bittorrent`: ContentType = ContentType("application", "x-bittorrent")
  lazy val `application/x-blorb`: ContentType = ContentType("application", "x-blorb")
  lazy val `application/x-bzip`: ContentType = ContentType("application", "x-bzip")
  lazy val `application/x-bzip2`: ContentType = ContentType("application", "x-bzip2")
  lazy val `application/xcap-diff+xml`: ContentType = ContentType("application", "xcap-diff+xml")
  lazy val `application/x-cbr`: ContentType = ContentType("application", "x-cbr")
  lazy val `application/x-cdlink`: ContentType = ContentType("application", "x-cdlink")
  lazy val `application/x-cfs-compressed`: ContentType = ContentType("application", "x-cfs-compressed")
  lazy val `application/x-chat`: ContentType = ContentType("application", "x-chat")
  lazy val `application/x-chess-pgn`: ContentType = ContentType("application", "x-chess-pgn")
  lazy val `application/x-conference`: ContentType = ContentType("application", "x-conference")
  lazy val `application/x-cpio`: ContentType = ContentType("application", "x-cpio")
  lazy val `application/x-csh`: ContentType = ContentType("application", "x-csh")
  lazy val `application/x-debian-package`: ContentType = ContentType("application", "x-debian-package")
  lazy val `application/x-dgc-compressed`: ContentType = ContentType("application", "x-dgc-compressed")
  lazy val `application/x-director`: ContentType = ContentType("application", "x-director")
  lazy val `application/x-doom`: ContentType = ContentType("application", "x-doom")
  lazy val `application/x-dtbncx+xml`: ContentType = ContentType("application", "x-dtbncx+xml")
  lazy val `application/x-dtbook+xml`: ContentType = ContentType("application", "x-dtbook+xml")
  lazy val `application/x-dtbresource+xml`: ContentType = ContentType("application", "x-dtbresource+xml")
  lazy val `application/x-dvi`: ContentType = ContentType("application", "x-dvi")
  lazy val `application/xenc+xml`: ContentType = ContentType("application", "xenc+xml")
  lazy val `application/x-envoy`: ContentType = ContentType("application", "x-envoy")
  lazy val `application/x-eva`: ContentType = ContentType("application", "x-eva")
  lazy val `application/x-font-bdf`: ContentType = ContentType("application", "x-font-bdf")
  lazy val `application/x-font-ghostscript`: ContentType = ContentType("application", "x-font-ghostscript")
  lazy val `application/x-font-linux-psf`: ContentType = ContentType("application", "x-font-linux-psf")
  lazy val `application/x-font-otf`: ContentType = ContentType("application", "x-font-otf")
  lazy val `application/x-font-pcf`: ContentType = ContentType("application", "x-font-pcf")
  lazy val `application/x-font-snf`: ContentType = ContentType("application", "x-font-snf")
  lazy val `application/x-font-ttf`: ContentType = ContentType("application", "x-font-ttf")
  lazy val `application/x-font-type1`: ContentType = ContentType("application", "x-font-type1")
  lazy val `application/x-freearc`: ContentType = ContentType("application", "x-freearc")
  lazy val `application/x-futuresplash`: ContentType = ContentType("application", "x-futuresplash")
  lazy val `application/x-gca-compressed`: ContentType = ContentType("application", "x-gca-compressed")
  lazy val `application/x-glulx`: ContentType = ContentType("application", "x-glulx")
  lazy val `application/x-gnumeric`: ContentType = ContentType("application", "x-gnumeric")
  lazy val `application/x-gramps-xml`: ContentType = ContentType("application", "x-gramps-xml")
  lazy val `application/x-gtar`: ContentType = ContentType("application", "x-gtar")
  lazy val `application/x-hdf`: ContentType = ContentType("application", "x-hdf")
  lazy val `application/xhtml+xml`: ContentType = ContentType("application", "xhtml+xml")
  lazy val `application/x-install-instructions`: ContentType = ContentType("application", "x-install-instructions")
  lazy val `application/x-iso9660-image`: ContentType = ContentType("application", "x-iso9660-image")
  lazy val `application/x-java-jnlp-file`: ContentType = ContentType("application", "x-java-jnlp-file")
  lazy val `application/x-latex`: ContentType = ContentType("application", "x-latex")
  lazy val `application/x-lzh-compressed`: ContentType = ContentType("application", "x-lzh-compressed")
  lazy val `application/x-mie`: ContentType = ContentType("application", "x-mie")
  lazy val `application/x-www-form-urlencoded` = ContentType("application", "x-www-form-urlencoded")
  lazy val `application/xml`: ContentType = ContentType("application", "xml")
  lazy val `application/xml-dtd`: ContentType = ContentType("application", "xml-dtd")
  lazy val `application/x-mobipocket-ebook`: ContentType = ContentType("application", "x-mobipocket-ebook")
  lazy val `application/x-msaccess`: ContentType = ContentType("application", "x-msaccess")
  lazy val `application/x-ms-application`: ContentType = ContentType("application", "x-ms-application")
  lazy val `application/x-msbinder`: ContentType = ContentType("application", "x-msbinder")
  lazy val `application/x-mscardfile`: ContentType = ContentType("application", "x-mscardfile")
  lazy val `application/x-msclip`: ContentType = ContentType("application", "x-msclip")
  lazy val `application/x-msdownload`: ContentType = ContentType("application", "x-msdownload")
  lazy val `application/x-msmediaview`: ContentType = ContentType("application", "x-msmediaview")
  lazy val `application/x-msmetafile`: ContentType = ContentType("application", "x-msmetafile")
  lazy val `application/x-msmoney`: ContentType = ContentType("application", "x-msmoney")
  lazy val `application/x-mspublisher`: ContentType = ContentType("application", "x-mspublisher")
  lazy val `application/x-msschedule`: ContentType = ContentType("application", "x-msschedule")
  lazy val `application/x-ms-shortcut`: ContentType = ContentType("application", "x-ms-shortcut")
  lazy val `application/x-msterminal`: ContentType = ContentType("application", "x-msterminal")
  lazy val `application/x-ms-wmd`: ContentType = ContentType("application", "x-ms-wmd")
  lazy val `application/x-ms-wmz`: ContentType = ContentType("application", "x-ms-wmz")
  lazy val `application/x-mswrite`: ContentType = ContentType("application", "x-mswrite")
  lazy val `application/x-ms-xbap`: ContentType = ContentType("application", "x-ms-xbap")
  lazy val `application/x-netcdf`: ContentType = ContentType("application", "x-netcdf")
  lazy val `application/x-nzb`: ContentType = ContentType("application", "x-nzb")
  lazy val `application/xop+xml`: ContentType = ContentType("application", "xop+xml")
  lazy val `application/x-pkcs12`: ContentType = ContentType("application", "x-pkcs12")
  lazy val `application/x-pkcs7-certificates`: ContentType = ContentType("application", "x-pkcs7-certificates")
  lazy val `application/x-pkcs7-certreqresp`: ContentType = ContentType("application", "x-pkcs7-certreqresp")
  lazy val `application/xproc+xml`: ContentType = ContentType("application", "xproc+xml")
  lazy val `application/x-rar-compressed`: ContentType = ContentType("application", "x-rar-compressed")
  lazy val `application/x-research-info-systems`: ContentType = ContentType("application", "x-research-info-systems")
  lazy val `application/x-sh`: ContentType = ContentType("application", "x-sh")
  lazy val `application/x-shar`: ContentType = ContentType("application", "x-shar")
  lazy val `application/x-shockwave-flash`: ContentType = ContentType("application", "x-shockwave-flash")
  lazy val `application/x-silverlight-app`: ContentType = ContentType("application", "x-silverlight-app")
  lazy val `application/xslt+xml`: ContentType = ContentType("application", "xslt+xml")
  lazy val `application/xspf+xml`: ContentType = ContentType("application", "xspf+xml")
  lazy val `application/x-sql`: ContentType = ContentType("application", "x-sql")
  lazy val `application/x-stuffit`: ContentType = ContentType("application", "x-stuffit")
  lazy val `application/x-stuffitx`: ContentType = ContentType("application", "x-stuffitx")
  lazy val `application/x-subrip`: ContentType = ContentType("application", "x-subrip")
  lazy val `application/x-sv4cpio`: ContentType = ContentType("application", "x-sv4cpio")
  lazy val `application/x-sv4crc`: ContentType = ContentType("application", "x-sv4crc")
  lazy val `application/x-t3vm-image`: ContentType = ContentType("application", "x-t3vm-image")
  lazy val `application/x-tads`: ContentType = ContentType("application", "x-tads")
  lazy val `application/x-tar`: ContentType = ContentType("application", "x-tar")
  lazy val `application/x-tcl`: ContentType = ContentType("application", "x-tcl")
  lazy val `application/x-tex`: ContentType = ContentType("application", "x-tex")
  lazy val `application/x-texinfo`: ContentType = ContentType("application", "x-texinfo")
  lazy val `application/x-tex-tfm`: ContentType = ContentType("application", "x-tex-tfm")
  lazy val `application/x-tgif`: ContentType = ContentType("application", "x-tgif")
  lazy val `application/x-ustar`: ContentType = ContentType("application", "x-ustar")
  lazy val `application/xv+xml`: ContentType = ContentType("application", "xv+xml")
  lazy val `application/x-wais-source`: ContentType = ContentType("application", "x-wais-source")
  lazy val `application/x-x509-ca-cert`: ContentType = ContentType("application", "x-x509-ca-cert")
  lazy val `application/x-xfig`: ContentType = ContentType("application", "x-xfig")
  lazy val `application/x-xliff+xml`: ContentType = ContentType("application", "x-xliff+xml")
  lazy val `application/x-xpinstall`: ContentType = ContentType("application", "x-xpinstall")
  lazy val `application/x-xz`: ContentType = ContentType("application", "x-xz")
  lazy val `application/x-zmachine`: ContentType = ContentType("application", "x-zmachine")
  lazy val `application/yang`: ContentType = ContentType("application", "yang")
  lazy val `application/yin+xml`: ContentType = ContentType("application", "yin+xml")
  lazy val `application/zip`: ContentType = ContentType("application", "zip")
  lazy val `audio/adpcm`: ContentType = ContentType("audio", "adpcm")
  lazy val `audio/basic`: ContentType = ContentType("audio", "basic")
  lazy val `audio/midi`: ContentType = ContentType("audio", "midi")
  lazy val `audio/mp4`: ContentType = ContentType("audio", "mp4")
  lazy val `audio/mpeg`: ContentType = ContentType("audio", "mpeg")
  lazy val `audio/ogg`: ContentType = ContentType("audio", "ogg")
  lazy val `audio/s3m`: ContentType = ContentType("audio", "s3m")
  lazy val `audio/silk`: ContentType = ContentType("audio", "silk")
  lazy val `audio/vnd.dece.audio`: ContentType = ContentType("audio", "vnd.dece.audio")
  lazy val `audio/vnd.digital-winds`: ContentType = ContentType("audio", "vnd.digital-winds")
  lazy val `audio/vnd.dra`: ContentType = ContentType("audio", "vnd.dra")
  lazy val `audio/vnd.dts`: ContentType = ContentType("audio", "vnd.dts")
  lazy val `audio/vnd.dts.hd`: ContentType = ContentType("audio", "vnd.dts.hd")
  lazy val `audio/vnd.lucent.voice`: ContentType = ContentType("audio", "vnd.lucent.voice")
  lazy val `audio/vnd.ms-playready.media.pya`: ContentType = ContentType("audio", "vnd.ms-playready.media.pya")
  lazy val `audio/vnd.nuera.ecelp4800`: ContentType = ContentType("audio", "vnd.nuera.ecelp4800")
  lazy val `audio/vnd.nuera.ecelp7470`: ContentType = ContentType("audio", "vnd.nuera.ecelp7470")
  lazy val `audio/vnd.nuera.ecelp9600`: ContentType = ContentType("audio", "vnd.nuera.ecelp9600")
  lazy val `audio/vnd.rip`: ContentType = ContentType("audio", "vnd.rip")
  lazy val `audio/webm`: ContentType = ContentType("audio", "webm")
  lazy val `audio/x-aac`: ContentType = ContentType("audio", "x-aac")
  lazy val `audio/x-aiff`: ContentType = ContentType("audio", "x-aiff")
  lazy val `audio/x-caf`: ContentType = ContentType("audio", "x-caf")
  lazy val `audio/x-flac`: ContentType = ContentType("audio", "x-flac")
  lazy val `audio/xm`: ContentType = ContentType("audio", "xm")
  lazy val `audio/x-matroska`: ContentType = ContentType("audio", "x-matroska")
  lazy val `audio/x-mpegurl`: ContentType = ContentType("audio", "x-mpegurl")
  lazy val `audio/x-ms-wax`: ContentType = ContentType("audio", "x-ms-wax")
  lazy val `audio/x-ms-wma`: ContentType = ContentType("audio", "x-ms-wma")
  lazy val `audio/x-pn-realaudio`: ContentType = ContentType("audio", "x-pn-realaudio")
  lazy val `audio/x-pn-realaudio-plugin`: ContentType = ContentType("audio", "x-pn-realaudio-plugin")
  lazy val `audio/x-wav`: ContentType = ContentType("audio", "x-wav")
  lazy val `chemical/x-cdx`: ContentType = ContentType("chemical", "x-cdx")
  lazy val `chemical/x-cif`: ContentType = ContentType("chemical", "x-cif")
  lazy val `chemical/x-cmdf`: ContentType = ContentType("chemical", "x-cmdf")
  lazy val `chemical/x-cml`: ContentType = ContentType("chemical", "x-cml")
  lazy val `chemical/x-csml`: ContentType = ContentType("chemical", "x-csml")
  lazy val `chemical/x-xyz`: ContentType = ContentType("chemical", "x-xyz")
  lazy val `image/bmp`: ContentType = ContentType("image", "bmp")
  lazy val `image/cgm`: ContentType = ContentType("image", "cgm")
  lazy val `image/g3fax`: ContentType = ContentType("image", "g3fax")
  lazy val `image/gif`: ContentType = ContentType("image", "gif")
  lazy val `image/ief`: ContentType = ContentType("image", "ief")
  lazy val `image/jpeg`: ContentType = ContentType("image", "jpeg")
  lazy val `image/ktx`: ContentType = ContentType("image", "ktx")
  lazy val `image/png`: ContentType = ContentType("image", "png")
  lazy val `image/prs.btif`: ContentType = ContentType("image", "prs.btif")
  lazy val `image/sgi`: ContentType = ContentType("image", "sgi")
  lazy val `image/svg+xml`: ContentType = ContentType("image", "svg+xml")
  lazy val `image/tiff`: ContentType = ContentType("image", "tiff")
  lazy val `image/vnd.adobe.photoshop`: ContentType = ContentType("image", "vnd.adobe.photoshop")
  lazy val `image/vnd.dece.graphic`: ContentType = ContentType("image", "vnd.dece.graphic")
  lazy val `image/vnd.djvu`: ContentType = ContentType("image", "vnd.djvu")
  lazy val `image/vnd.dvb.subtitle`: ContentType = ContentType("image", "vnd.dvb.subtitle")
  lazy val `image/vnd.dwg`: ContentType = ContentType("image", "vnd.dwg")
  lazy val `image/vnd.dxf`: ContentType = ContentType("image", "vnd.dxf")
  lazy val `image/vnd.fastbidsheet`: ContentType = ContentType("image", "vnd.fastbidsheet")
  lazy val `image/vnd.fpx`: ContentType = ContentType("image", "vnd.fpx")
  lazy val `image/vnd.fst`: ContentType = ContentType("image", "vnd.fst")
  lazy val `image/vnd.fujixerox.edmics-mmr`: ContentType = ContentType("image", "vnd.fujixerox.edmics-mmr")
  lazy val `image/vnd.fujixerox.edmics-rlc`: ContentType = ContentType("image", "vnd.fujixerox.edmics-rlc")
  lazy val `image/vnd.ms-modi`: ContentType = ContentType("image", "vnd.ms-modi")
  lazy val `image/vnd.ms-photo`: ContentType = ContentType("image", "vnd.ms-photo")
  lazy val `image/vnd.net-fpx`: ContentType = ContentType("image", "vnd.net-fpx")
  lazy val `image/vnd.wap.wbmp`: ContentType = ContentType("image", "vnd.wap.wbmp")
  lazy val `image/vnd.xiff`: ContentType = ContentType("image", "vnd.xiff")
  lazy val `image/webp`: ContentType = ContentType("image", "webp")
  lazy val `image/x-3ds`: ContentType = ContentType("image", "x-3ds")
  lazy val `image/x-cmu-raster`: ContentType = ContentType("image", "x-cmu-raster")
  lazy val `image/x-cmx`: ContentType = ContentType("image", "x-cmx")
  lazy val `image/x-freehand`: ContentType = ContentType("image", "x-freehand")
  lazy val `image/x-icon`: ContentType = ContentType("image", "x-icon")
  lazy val `image/x-mrsid-image`: ContentType = ContentType("image", "x-mrsid-image")
  lazy val `image/x-pcx`: ContentType = ContentType("image", "x-pcx")
  lazy val `image/x-pict`: ContentType = ContentType("image", "x-pict")
  lazy val `image/x-portable-anymap`: ContentType = ContentType("image", "x-portable-anymap")
  lazy val `image/x-portable-bitmap`: ContentType = ContentType("image", "x-portable-bitmap")
  lazy val `image/x-portable-graymap`: ContentType = ContentType("image", "x-portable-graymap")
  lazy val `image/x-portable-pixmap`: ContentType = ContentType("image", "x-portable-pixmap")
  lazy val `image/x-rgb`: ContentType = ContentType("image", "x-rgb")
  lazy val `image/x-tga`: ContentType = ContentType("image", "x-tga")
  lazy val `image/x-xbitmap`: ContentType = ContentType("image", "x-xbitmap")
  lazy val `image/x-xpixmap`: ContentType = ContentType("image", "x-xpixmap")
  lazy val `image/x-xwindowdump`: ContentType = ContentType("image", "x-xwindowdump")
  lazy val `message/rfc822`: ContentType = ContentType("message", "rfc822")
  lazy val `model/iges`: ContentType = ContentType("model", "iges")
  lazy val `model/mesh`: ContentType = ContentType("model", "mesh")
  lazy val `model/vnd.collada+xml`: ContentType = ContentType("model", "vnd.collada+xml")
  lazy val `model/vnd.dwf`: ContentType = ContentType("model", "vnd.dwf")
  lazy val `model/vnd.gdl`: ContentType = ContentType("model", "vnd.gdl")
  lazy val `model/vnd.gtw`: ContentType = ContentType("model", "vnd.gtw")
  lazy val `model/vnd.mts`: ContentType = ContentType("model", "vnd.mts")
  lazy val `model/vnd.vtu`: ContentType = ContentType("model", "vnd.vtu")
  lazy val `model/vrml`: ContentType = ContentType("model", "vrml")
  lazy val `model/x3d+binary`: ContentType = ContentType("model", "x3d+binary")
  lazy val `model/x3d+vrml`: ContentType = ContentType("model", "x3d+vrml")
  lazy val `model/x3d+xml`: ContentType = ContentType("model", "x3d+xml")
  lazy val `multipart/form-data` = ContentType("multipart", "form-data")
  lazy val `text/cache-manifest`: ContentType = ContentType("text", "cache-manifest")
  lazy val `text/calendar`: ContentType = ContentType("text", "calendar")
  lazy val `text/css`: ContentType = ContentType("text", "css")
  lazy val `text/csv`: ContentType = ContentType("text", "csv")
  lazy val `text/html`: ContentType = ContentType("text", "html")
  lazy val `text/n3`: ContentType = ContentType("text", "n3")
  lazy val `text/plain`: ContentType = ContentType("text", "plain")
  lazy val `text/prs.lines.tag`: ContentType = ContentType("text", "prs.lines.tag")
  lazy val `text/richtext`: ContentType = ContentType("text", "richtext")
  lazy val `text/sgml`: ContentType = ContentType("text", "sgml")
  lazy val `text/tab-separated-values`: ContentType = ContentType("text", "tab-separated-values")
  lazy val `text/troff`: ContentType = ContentType("text", "troff")
  lazy val `text/turtle`: ContentType = ContentType("text", "turtle")
  lazy val `text/uri-list`: ContentType = ContentType("text", "uri-list")
  lazy val `text/vcard`: ContentType = ContentType("text", "vcard")
  lazy val `text/vnd.curl`: ContentType = ContentType("text", "vnd.curl")
  lazy val `text/vnd.curl.dcurl`: ContentType = ContentType("text", "vnd.curl.dcurl")
  lazy val `text/vnd.curl.mcurl`: ContentType = ContentType("text", "vnd.curl.mcurl")
  lazy val `text/vnd.curl.scurl`: ContentType = ContentType("text", "vnd.curl.scurl")
  lazy val `text/vnd.dvb.subtitle`: ContentType = ContentType("text", "vnd.dvb.subtitle")
  lazy val `text/vnd.fly`: ContentType = ContentType("text", "vnd.fly")
  lazy val `text/vnd.fmi.flexstor`: ContentType = ContentType("text", "vnd.fmi.flexstor")
  lazy val `text/vnd.graphviz`: ContentType = ContentType("text", "vnd.graphviz")
  lazy val `text/vnd.in3d.3dml`: ContentType = ContentType("text", "vnd.in3d.3dml")
  lazy val `text/vnd.in3d.spot`: ContentType = ContentType("text", "vnd.in3d.spot")
  lazy val `text/vnd.sun.j2me.app-descriptor`: ContentType = ContentType("text", "vnd.sun.j2me.app-descriptor")
  lazy val `text/vnd.wap.wml`: ContentType = ContentType("text", "vnd.wap.wml")
  lazy val `text/vnd.wap.wmlscript`: ContentType = ContentType("text", "vnd.wap.wmlscript")
  lazy val `text/x-asm`: ContentType = ContentType("text", "x-asm")
  lazy val `text/x-c`: ContentType = ContentType("text", "x-c")
  lazy val `text/x-fortran`: ContentType = ContentType("text", "x-fortran")
  lazy val `text/x-java-source`: ContentType = ContentType("text", "x-java-source")
  lazy val `text/x-nfo`: ContentType = ContentType("text", "x-nfo")
  lazy val `text/x-opml`: ContentType = ContentType("text", "x-opml")
  lazy val `text/x-pascal`: ContentType = ContentType("text", "x-pascal")
  lazy val `text/x-setext`: ContentType = ContentType("text", "x-setext")
  lazy val `text/x-sfv`: ContentType = ContentType("text", "x-sfv")
  lazy val `text/x-uuencode`: ContentType = ContentType("text", "x-uuencode")
  lazy val `text/x-vcalendar`: ContentType = ContentType("text", "x-vcalendar")
  lazy val `text/x-vcard`: ContentType = ContentType("text", "x-vcard")
  lazy val `text/xml`: ContentType = ContentType("text", "xml")
  lazy val `video/3gpp`: ContentType = ContentType("video", "3gpp")
  lazy val `video/3gpp2`: ContentType = ContentType("video", "3gpp2")
  lazy val `video/h261`: ContentType = ContentType("video", "h261")
  lazy val `video/h263`: ContentType = ContentType("video", "h263")
  lazy val `video/h264`: ContentType = ContentType("video", "h264")
  lazy val `video/jpeg`: ContentType = ContentType("video", "jpeg")
  lazy val `video/jpm`: ContentType = ContentType("video", "jpm")
  lazy val `video/mj2`: ContentType = ContentType("video", "mj2")
  lazy val `video/mp4`: ContentType = ContentType("video", "mp4")
  lazy val `video/mpeg`: ContentType = ContentType("video", "mpeg")
  lazy val `video/ogg`: ContentType = ContentType("video", "ogg")
  lazy val `video/quicktime`: ContentType = ContentType("video", "quicktime")
  lazy val `video/vnd.dece.hd`: ContentType = ContentType("video", "vnd.dece.hd")
  lazy val `video/vnd.dece.mobile`: ContentType = ContentType("video", "vnd.dece.mobile")
  lazy val `video/vnd.dece.pd`: ContentType = ContentType("video", "vnd.dece.pd")
  lazy val `video/vnd.dece.sd`: ContentType = ContentType("video", "vnd.dece.sd")
  lazy val `video/vnd.dece.video`: ContentType = ContentType("video", "vnd.dece.video")
  lazy val `video/vnd.dvb.file`: ContentType = ContentType("video", "vnd.dvb.file")
  lazy val `video/vnd.fvt`: ContentType = ContentType("video", "vnd.fvt")
  lazy val `video/vnd.mpegurl`: ContentType = ContentType("video", "vnd.mpegurl")
  lazy val `video/vnd.ms-playready.media.pyv`: ContentType = ContentType("video", "vnd.ms-playready.media.pyv")
  lazy val `video/vnd.uvvu.mp4`: ContentType = ContentType("video", "vnd.uvvu.mp4")
  lazy val `video/vnd.vivo`: ContentType = ContentType("video", "vnd.vivo")
  lazy val `video/webm`: ContentType = ContentType("video", "webm")
  lazy val `video/x-f4v`: ContentType = ContentType("video", "x-f4v")
  lazy val `video/x-fli`: ContentType = ContentType("video", "x-fli")
  lazy val `video/x-flv`: ContentType = ContentType("video", "x-flv")
  lazy val `video/x-m4v`: ContentType = ContentType("video", "x-m4v")
  lazy val `video/x-matroska`: ContentType = ContentType("video", "x-matroska")
  lazy val `video/x-mng`: ContentType = ContentType("video", "x-mng")
  lazy val `video/x-ms-asf`: ContentType = ContentType("video", "x-ms-asf")
  lazy val `video/x-msvideo`: ContentType = ContentType("video", "x-msvideo")
  lazy val `video/x-ms-vob`: ContentType = ContentType("video", "x-ms-vob")
  lazy val `video/x-ms-wm`: ContentType = ContentType("video", "x-ms-wm")
  lazy val `video/x-ms-wmv`: ContentType = ContentType("video", "x-ms-wmv")
  lazy val `video/x-ms-wmx`: ContentType = ContentType("video", "x-ms-wmx")
  lazy val `video/x-ms-wvx`: ContentType = ContentType("video", "x-ms-wvx")
  lazy val `video/x-sgi-movie`: ContentType = ContentType("video", "x-sgi-movie")
  lazy val `video/x-smv`: ContentType = ContentType("video", "x-smv")
  lazy val `x-conference/x-cooltalk`: ContentType = ContentType("x-conference", "x-cooltalk")

  map("123", "application/vnd.lotus-1-2-3")
  map("3dml", "text/vnd.in3d.3dml")
  map("3ds", "image/x-3ds")
  map("3g2", "video/3gpp2")
  map("3gp", "video/3gpp")
  map("7z", "application/x-7z-compressed")
  map("aab", "application/x-authorware-bin")
  map("aac", "audio/x-aac")
  map("aam", "application/x-authorware-map")
  map("aas", "application/x-authorware-seg")
  map("abw", "application/x-abiword")
  map("ac", "application/pkix-attr-cert")
  map("acc", "application/vnd.americandynamics.acc")
  map("ace", "application/x-ace-compressed")
  map("acu", "application/vnd.acucobol")
  map("acutc", "application/vnd.acucorp")
  map("adp", "audio/adpcm")
  map("aep", "application/vnd.audiograph")
  map("afm", "application/x-font-type1")
  map("afp", "application/vnd.ibm.modcap")
  map("ahead", "application/vnd.ahead.space")
  map("ai", "application/postscript")
  map("aif", "audio/x-aiff")
  map("aifc", "audio/x-aiff")
  map("aiff", "audio/x-aiff")
  map("air", "application/vnd.adobe.air-application-installer-package+zip")
  map("ait", "application/vnd.dvb.ait")
  map("ami", "application/vnd.amiga.ami")
  map("apk", "application/vnd.android.package-archive")
  map("appcache", "text/cache-manifest")
  map("application", "application/x-ms-application")
  map("apr", "application/vnd.lotus-approach")
  map("arc", "application/x-freearc")
  map("asc", "application/pgp-signature")
  map("asf", "video/x-ms-asf")
  map("asm", "text/x-asm")
  map("aso", "application/vnd.accpac.simply.aso")
  map("asx", "video/x-ms-asf")
  map("atc", "application/vnd.acucorp")
  map("atom", "application/atom+xml")
  map("atomcat", "application/atomcat+xml")
  map("atomsvc", "application/atomsvc+xml")
  map("atx", "application/vnd.antix.game-component")
  map("au", "audio/basic")
  map("avi", "video/x-msvideo")
  map("aw", "application/applixware")
  map("azf", "application/vnd.airzip.filesecure.azf")
  map("azs", "application/vnd.airzip.filesecure.azs")
  map("azw", "application/vnd.amazon.ebook")
  map("bat", "application/x-msdownload")
  map("bcpio", "application/x-bcpio")
  map("bdf", "application/x-font-bdf")
  map("bdm", "application/vnd.syncml.dm+wbxml")
  map("bed", "application/vnd.realvnc.bed")
  map("bh2", "application/vnd.fujitsu.oasysprs")
  map("bin", "application/octet-stream")
  map("blb", "application/x-blorb")
  map("blorb", "application/x-blorb")
  map("bmi", "application/vnd.bmi")
  map("bmp", "image/bmp")
  map("book", "application/vnd.framemaker")
  map("box", "application/vnd.previewsystems.box")
  map("boz", "application/x-bzip2")
  map("bpk", "application/octet-stream")
  map("btif", "image/prs.btif")
  map("bz2", "application/x-bzip2")
  map("bz", "application/x-bzip")
  map("c11amc", "application/vnd.cluetrust.cartomobile-config")
  map("c11amz", "application/vnd.cluetrust.cartomobile-config-pkg")
  map("c4d", "application/vnd.clonk.c4group")
  map("c4f", "application/vnd.clonk.c4group")
  map("c4g", "application/vnd.clonk.c4group")
  map("c4p", "application/vnd.clonk.c4group")
  map("c4u", "application/vnd.clonk.c4group")
  map("cab", "application/vnd.ms-cab-compressed")
  map("caf", "audio/x-caf")
  map("cap", "application/vnd.tcpdump.pcap")
  map("car", "application/vnd.curl.car")
  map("cat", "application/vnd.ms-pki.seccat")
  map("cb7", "application/x-cbr")
  map("cba", "application/x-cbr")
  map("cbr", "application/x-cbr")
  map("cbt", "application/x-cbr")
  map("cbz", "application/x-cbr")
  map("cct", "application/x-director")
  map("cc", "text/x-c")
  map("ccxml", "application/ccxml+xml")
  map("cdbcmsg", "application/vnd.contact.cmsg")
  map("cdf", "application/x-netcdf")
  map("cdkey", "application/vnd.mediastation.cdkey")
  map("cdmia", "application/cdmi-capability")
  map("cdmic", "application/cdmi-container")
  map("cdmid", "application/cdmi-domain")
  map("cdmio", "application/cdmi-object")
  map("cdmiq", "application/cdmi-queue")
  map("cdx", "chemical/x-cdx")
  map("cdxml", "application/vnd.chemdraw+xml")
  map("cdy", "application/vnd.cinderella")
  map("cer", "application/pkix-cert")
  map("cfs", "application/x-cfs-compressed")
  map("cgm", "image/cgm")
  map("chat", "application/x-chat")
  map("chm", "application/vnd.ms-htmlhelp")
  map("chrt", "application/vnd.kde.kchart")
  map("cif", "chemical/x-cif")
  map("cii", "application/vnd.anser-web-certificate-issue-initiation")
  map("cil", "application/vnd.ms-artgalry")
  map("cla", "application/vnd.claymore")
  map("class", "application/java-vm")
  map("clkk", "application/vnd.crick.clicker.keyboard")
  map("clkp", "application/vnd.crick.clicker.palette")
  map("clkt", "application/vnd.crick.clicker.template")
  map("clkw", "application/vnd.crick.clicker.wordbank")
  map("clkx", "application/vnd.crick.clicker")
  map("clp", "application/x-msclip")
  map("cmc", "application/vnd.cosmocaller")
  map("cmdf", "chemical/x-cmdf")
  map("cml", "chemical/x-cml")
  map("cmp", "application/vnd.yellowriver-custom-menu")
  map("cmx", "image/x-cmx")
  map("cod", "application/vnd.rim.cod")
  map("com", "application/x-msdownload")
  map("conf", "text/plain")
  map("cpio", "application/x-cpio")
  map("cpp", "text/x-c")
  map("cpt", "application/mac-compactpro")
  map("crd", "application/x-mscardfile")
  map("crl", "application/pkix-crl")
  map("crt", "application/x-x509-ca-cert")
  map("cryptonote", "application/vnd.rig.cryptonote")
  map("csh", "application/x-csh")
  map("csml", "chemical/x-csml")
  map("csp", "application/vnd.commonspace")
  map("css", "text/css")
  map("cst", "application/x-director")
  map("csv", "text/csv")
  map("c", "text/x-c")
  map("cu", "application/cu-seeme")
  map("curl", "text/vnd.curl")
  map("cww", "application/prs.cww")
  map("cxt", "application/x-director")
  map("cxx", "text/x-c")
  map("dae", "model/vnd.collada+xml")
  map("daf", "application/vnd.mobius.daf")
  map("dart", "application/vnd.dart")
  map("dataless", "application/vnd.fdsn.seed")
  map("davmount", "application/davmount+xml")
  map("dbk", "application/docbook+xml")
  map("dcr", "application/x-director")
  map("dcurl", "text/vnd.curl.dcurl")
  map("dd2", "application/vnd.oma.dd2+xml")
  map("ddd", "application/vnd.fujixerox.ddd")
  map("deb", "application/x-debian-package")
  map("def", "text/plain")
  map("deploy", "application/octet-stream")
  map("der", "application/x-x509-ca-cert")
  map("dfac", "application/vnd.dreamfactory")
  map("dgc", "application/x-dgc-compressed")
  map("dic", "text/x-c")
  map("dir", "application/x-director")
  map("dis", "application/vnd.mobius.dis")
  map("dist", "application/octet-stream")
  map("distz", "application/octet-stream")
  map("djv", "image/vnd.djvu")
  map("djvu", "image/vnd.djvu")
  map("dll", "application/x-msdownload")
  map("dmg", "application/x-apple-diskimage")
  map("dmp", "application/vnd.tcpdump.pcap")
  map("dms", "application/octet-stream")
  map("dna", "application/vnd.dna")
  map("doc", "application/msword")
  map("docm", "application/vnd.ms-word.document.macroenabled.12")
  map("docx", "application/vnd.openxmlformats-officedocument.wordprocessingml.document")
  map("dot", "application/msword")
  map("dotm", "application/vnd.ms-word.template.macroenabled.12")
  map("dotx", "application/vnd.openxmlformats-officedocument.wordprocessingml.template")
  map("dp", "application/vnd.osgi.dp")
  map("dpg", "application/vnd.dpgraph")
  map("dra", "audio/vnd.dra")
  map("dsc", "text/prs.lines.tag")
  map("dssc", "application/dssc+der")
  map("dtb", "application/x-dtbook+xml")
  map("dtd", "application/xml-dtd")
  map("dts", "audio/vnd.dts")
  map("dtshd", "audio/vnd.dts.hd")
  map("dump", "application/octet-stream")
  map("dvb", "video/vnd.dvb.file")
  map("dvi", "application/x-dvi")
  map("dwf", "model/vnd.dwf")
  map("dwg", "image/vnd.dwg")
  map("dxf", "image/vnd.dxf")
  map("dxp", "application/vnd.spotfire.dxp")
  map("dxr", "application/x-director")
  map("ecelp4800", "audio/vnd.nuera.ecelp4800")
  map("ecelp7470", "audio/vnd.nuera.ecelp7470")
  map("ecelp9600", "audio/vnd.nuera.ecelp9600")
  map("ecma", "application/ecmascript")
  map("edm", "application/vnd.novadigm.edm")
  map("edx", "application/vnd.novadigm.edx")
  map("efif", "application/vnd.picsel")
  map("ei6", "application/vnd.pg.osasli")
  map("elc", "application/octet-stream")
  map("emf", "application/x-msmetafile")
  map("eml", "message/rfc822")
  map("emma", "application/emma+xml")
  map("emz", "application/x-msmetafile")
  map("eol", "audio/vnd.digital-winds")
  map("eot", "application/vnd.ms-fontobject")
  map("eps", "application/postscript")
  map("epub", "application/epub+zip")
  map("es3", "application/vnd.eszigno3+xml")
  map("esa", "application/vnd.osgi.subsystem")
  map("esf", "application/vnd.epson.esf")
  map("et3", "application/vnd.eszigno3+xml")
  map("etx", "text/x-setext")
  map("eva", "application/x-eva")
  map("evy", "application/x-envoy")
  map("exe", "application/x-msdownload")
  map("exi", "application/exi")
  map("ext", "application/vnd.novadigm.ext")
  map("ez2", "application/vnd.ezpix-album")
  map("ez3", "application/vnd.ezpix-package")
  map("ez", "application/andrew-inset")
  map("f4v", "video/x-f4v")
  map("f77", "text/x-fortran")
  map("f90", "text/x-fortran")
  map("fbs", "image/vnd.fastbidsheet")
  map("fcdt", "application/vnd.adobe.formscentral.fcdt")
  map("fcs", "application/vnd.isac.fcs")
  map("fdf", "application/vnd.fdf")
  map("fe_launch", "application/vnd.denovo.fcselayout-link")
  map("fg5", "application/vnd.fujitsu.oasysgp")
  map("fgd", "application/x-director")
  map("fh4", "image/x-freehand")
  map("fh5", "image/x-freehand")
  map("fh7", "image/x-freehand")
  map("fhc", "image/x-freehand")
  map("fh", "image/x-freehand")
  map("fig", "application/x-xfig")
  map("flac", "audio/x-flac")
  map("fli", "video/x-fli")
  map("flo", "application/vnd.micrografx.flo")
  map("flv", "video/x-flv")
  map("flw", "application/vnd.kde.kivio")
  map("flx", "text/vnd.fmi.flexstor")
  map("fly", "text/vnd.fly")
  map("fm", "application/vnd.framemaker")
  map("fnc", "application/vnd.frogans.fnc")
  map("for", "text/x-fortran")
  map("fpx", "image/vnd.fpx")
  map("frame", "application/vnd.framemaker")
  map("fsc", "application/vnd.fsc.weblaunch")
  map("fst", "image/vnd.fst")
  map("ftc", "application/vnd.fluxtime.clip")
  map("f", "text/x-fortran")
  map("fti", "application/vnd.anser-web-funds-transfer-initiation")
  map("fvt", "video/vnd.fvt")
  map("fxp", "application/vnd.adobe.fxp")
  map("fxpl", "application/vnd.adobe.fxp")
  map("fzs", "application/vnd.fuzzysheet")
  map("g2w", "application/vnd.geoplan")
  map("g3", "image/g3fax")
  map("g3w", "application/vnd.geospace")
  map("gac", "application/vnd.groove-account")
  map("gam", "application/x-tads")
  map("gbr", "application/rpki-ghostbusters")
  map("gca", "application/x-gca-compressed")
  map("gdl", "model/vnd.gdl")
  map("geo", "application/vnd.dynageo")
  map("gex", "application/vnd.geometry-explorer")
  map("ggb", "application/vnd.geogebra.file")
  map("ggt", "application/vnd.geogebra.tool")
  map("ghf", "application/vnd.groove-help")
  map("gif", "image/gif")
  map("gim", "application/vnd.groove-identity-message")
  map("gml", "application/gml+xml")
  map("gmx", "application/vnd.gmx")
  map("gnumeric", "application/x-gnumeric")
  map("gph", "application/vnd.flographit")
  map("gpx", "application/gpx+xml")
  map("gqf", "application/vnd.grafeq")
  map("gqs", "application/vnd.grafeq")
  map("gram", "application/srgs")
  map("gramps", "application/x-gramps-xml")
  map("gre", "application/vnd.geometry-explorer")
  map("grv", "application/vnd.groove-injector")
  map("grxml", "application/srgs+xml")
  map("gsf", "application/x-font-ghostscript")
  map("gtar", "application/x-gtar")
  map("gtm", "application/vnd.groove-tool-message")
  map("gtw", "model/vnd.gtw")
  map("gv", "text/vnd.graphviz")
  map("gxf", "application/gxf")
  map("gxt", "application/vnd.geonext")
  map("h261", "video/h261")
  map("h263", "video/h263")
  map("h264", "video/h264")
  map("hal", "application/vnd.hal+xml")
  map("hbci", "application/vnd.hbci")
  map("hdf", "application/x-hdf")
  map("hh", "text/x-c")
  map("hlp", "application/winhlp")
  map("hpgl", "application/vnd.hp-hpgl")
  map("hpid", "application/vnd.hp-hpid")
  map("hps", "application/vnd.hp-hps")
  map("hqx", "application/mac-binhex40")
  map("h", "text/x-c")
  map("htke", "application/vnd.kenameaapp")
  map("html", "text/html")
  map("htm", "text/html")
  map("hvd", "application/vnd.yamaha.hv-dic")
  map("hvp", "application/vnd.yamaha.hv-voice")
  map("hvs", "application/vnd.yamaha.hv-script")
  map("i2g", "application/vnd.intergeo")
  map("icc", "application/vnd.iccprofile")
  map("ice", "x-conference/x-cooltalk")
  map("icm", "application/vnd.iccprofile")
  map("ico", "image/x-icon")
  map("ics", "text/calendar")
  map("ief", "image/ief")
  map("ifb", "text/calendar")
  map("ifm", "application/vnd.shana.informed.formdata")
  map("iges", "model/iges")
  map("igl", "application/vnd.igloader")
  map("igm", "application/vnd.insors.igm")
  map("igs", "model/iges")
  map("igx", "application/vnd.micrografx.igx")
  map("iif", "application/vnd.shana.informed.interchange")
  map("imp", "application/vnd.accpac.simply.imp")
  map("ims", "application/vnd.ms-ims")
  map("ink", "application/inkml+xml")
  map("inkml", "application/inkml+xml")
  map("install", "application/x-install-instructions")
  map("in", "text/plain")
  map("iota", "application/vnd.astraea-software.iota")
  map("ipfix", "application/ipfix")
  map("ipk", "application/vnd.shana.informed.package")
  map("irm", "application/vnd.ibm.rights-management")
  map("irp", "application/vnd.irepository.package+xml")
  map("iso", "application/x-iso9660-image")
  map("itp", "application/vnd.shana.informed.formtemplate")
  map("ivp", "application/vnd.immervision-ivp")
  map("ivu", "application/vnd.immervision-ivu")
  map("jad", "text/vnd.sun.j2me.app-descriptor")
  map("jam", "application/vnd.jam")
  map("jar", "application/java-archive")
  map("java", "text/x-java-source")
  map("jisp", "application/vnd.jisp")
  map("jlt", "application/vnd.hp-jlyt")
  map("jnlp", "application/x-java-jnlp-file")
  map("joda", "application/vnd.joost.joda-archive")
  map("jpeg", "image/jpeg")
  map("jpe", "image/jpeg")
  map("jpg", "image/jpeg")
  map("jpgm", "video/jpm")
  map("jpgv", "video/jpeg")
  map("jpm", "video/jpm")
  map("js", "application/javascript")
  map("json", "application/json")
  map("jsonml", "application/jsonml+json")
  map("kar", "audio/midi")
  map("karbon", "application/vnd.kde.karbon")
  map("kfo", "application/vnd.kde.kformula")
  map("kia", "application/vnd.kidspiration")
  map("kml", "application/vnd.google-earth.kml+xml")
  map("kmz", "application/vnd.google-earth.kmz")
  map("kne", "application/vnd.kinar")
  map("knp", "application/vnd.kinar")
  map("kon", "application/vnd.kde.kontour")
  map("kpr", "application/vnd.kde.kpresenter")
  map("kpt", "application/vnd.kde.kpresenter")
  map("kpxx", "application/vnd.ds-keypoint")
  map("ksp", "application/vnd.kde.kspread")
  map("ktr", "application/vnd.kahootz")
  map("ktx", "image/ktx")
  map("ktz", "application/vnd.kahootz")
  map("kwd", "application/vnd.kde.kword")
  map("kwt", "application/vnd.kde.kword")
  map("lasxml", "application/vnd.las.las+xml")
  map("latex", "application/x-latex")
  map("lbd", "application/vnd.llamagraphics.life-balance.desktop")
  map("lbe", "application/vnd.llamagraphics.life-balance.exchange+xml")
  map("les", "application/vnd.hhe.lesson-player")
  map("lha", "application/x-lzh-compressed")
  map("link66", "application/vnd.route66.link66+xml")
  map("list3820", "application/vnd.ibm.modcap")
  map("listafp", "application/vnd.ibm.modcap")
  map("list", "text/plain")
  map("lnk", "application/x-ms-shortcut")
  map("log", "text/plain")
  map("lostxml", "application/lost+xml")
  map("lrf", "application/octet-stream")
  map("lrm", "application/vnd.ms-lrm")
  map("ltf", "application/vnd.frogans.ltf")
  map("lvp", "audio/vnd.lucent.voice")
  map("lwp", "application/vnd.lotus-wordpro")
  map("lzh", "application/x-lzh-compressed")
  map("m13", "application/x-msmediaview")
  map("m14", "application/x-msmediaview")
  map("m1v", "video/mpeg")
  map("m21", "application/mp21")
  map("m2a", "audio/mpeg")
  map("m2v", "video/mpeg")
  map("m3a", "audio/mpeg")
  map("m3u8", "application/vnd.apple.mpegurl")
  map("m3u", "audio/x-mpegurl")
  map("m4a", "audio/mp4")
  map("m4u", "video/vnd.mpegurl")
  map("m4v", "video/x-m4v")
  map("ma", "application/mathematica")
  map("mads", "application/mads+xml")
  map("mag", "application/vnd.ecowin.chart")
  map("maker", "application/vnd.framemaker")
  map("man", "text/troff")
  map("mar", "application/octet-stream")
  map("mathml", "application/mathml+xml")
  map("mb", "application/mathematica")
  map("mbk", "application/vnd.mobius.mbk")
  map("mbox", "application/mbox")
  map("mc1", "application/vnd.medcalcdata")
  map("mcd", "application/vnd.mcd")
  map("mcurl", "text/vnd.curl.mcurl")
  map("mdb", "application/x-msaccess")
  map("mdi", "image/vnd.ms-modi")
  map("mesh", "model/mesh")
  map("meta4", "application/metalink4+xml")
  map("metalink", "application/metalink+xml")
  map("me", "text/troff")
  map("mets", "application/mets+xml")
  map("mfm", "application/vnd.mfmp")
  map("mft", "application/rpki-manifest")
  map("mgp", "application/vnd.osgeo.mapguide.package")
  map("mgz", "application/vnd.proteus.magazine")
  map("mid", "audio/midi")
  map("midi", "audio/midi")
  map("mie", "application/x-mie")
  map("mif", "application/vnd.mif")
  map("mime", "message/rfc822")
  map("mj2", "video/mj2")
  map("mjp2", "video/mj2")
  map("mk3d", "video/x-matroska")
  map("mka", "audio/x-matroska")
  map("mks", "video/x-matroska")
  map("mkv", "video/x-matroska")
  map("mlp", "application/vnd.dolby.mlp")
  map("mmd", "application/vnd.chipnuts.karaoke-mmd")
  map("mmf", "application/vnd.smaf")
  map("mmr", "image/vnd.fujixerox.edmics-mmr")
  map("mng", "video/x-mng")
  map("mny", "application/x-msmoney")
  map("mobi", "application/x-mobipocket-ebook")
  map("mods", "application/mods+xml")
  map("movie", "video/x-sgi-movie")
  map("mov", "video/quicktime")
  map("mp21", "application/mp21")
  map("mp2a", "audio/mpeg")
  map("mp2", "audio/mpeg")
  map("mp3", "audio/mpeg")
  map("mp4a", "audio/mp4")
  map("mp4s", "application/mp4")
  map("mp4", "video/mp4")
  map("mp4v", "video/mp4")
  map("mpc", "application/vnd.mophun.certificate")
  map("mpeg", "video/mpeg")
  map("mpe", "video/mpeg")
  map("mpg4", "video/mp4")
  map("mpga", "audio/mpeg")
  map("mpg", "video/mpeg")
  map("mpkg", "application/vnd.apple.installer+xml")
  map("mpm", "application/vnd.blueice.multipass")
  map("mpn", "application/vnd.mophun.application")
  map("mpp", "application/vnd.ms-project")
  map("mpt", "application/vnd.ms-project")
  map("mpy", "application/vnd.ibm.minipay")
  map("mqy", "application/vnd.mobius.mqy")
  map("mrc", "application/marc")
  map("mrcx", "application/marcxml+xml")
  map("mscml", "application/mediaservercontrol+xml")
  map("mseed", "application/vnd.fdsn.mseed")
  map("mseq", "application/vnd.mseq")
  map("msf", "application/vnd.epson.msf")
  map("msh", "model/mesh")
  map("msi", "application/x-msdownload")
  map("msl", "application/vnd.mobius.msl")
  map("ms", "text/troff")
  map("msty", "application/vnd.muvee.style")
  map("mts", "model/vnd.mts")
  map("mus", "application/vnd.musician")
  map("musicxml", "application/vnd.recordare.musicxml+xml")
  map("mvb", "application/x-msmediaview")
  map("mwf", "application/vnd.mfer")
  map("mxf", "application/mxf")
  map("mxl", "application/vnd.recordare.musicxml")
  map("mxml", "application/xv+xml")
  map("mxs", "application/vnd.triscape.mxs")
  map("mxu", "video/vnd.mpegurl")
  map("n3", "text/n3")
  map("nb", "application/mathematica")
  map("nbp", "application/vnd.wolfram.player")
  map("nc", "application/x-netcdf")
  map("ncx", "application/x-dtbncx+xml")
  map("nfo", "text/x-nfo")
  map("n-gage", "application/vnd.nokia.n-gage.symbian.install")
  map("ngdat", "application/vnd.nokia.n-gage.data")
  map("nitf", "application/vnd.nitf")
  map("nlu", "application/vnd.neurolanguage.nlu")
  map("nml", "application/vnd.enliven")
  map("nnd", "application/vnd.noblenet-directory")
  map("nns", "application/vnd.noblenet-sealer")
  map("nnw", "application/vnd.noblenet-web")
  map("npx", "image/vnd.net-fpx")
  map("nsc", "application/x-conference")
  map("nsf", "application/vnd.lotus-notes")
  map("ntf", "application/vnd.nitf")
  map("nzb", "application/x-nzb")
  map("oa2", "application/vnd.fujitsu.oasys2")
  map("oa3", "application/vnd.fujitsu.oasys3")
  map("oas", "application/vnd.fujitsu.oasys")
  map("obd", "application/x-msbinder")
  map("obj", "application/x-tgif")
  map("oda", "application/oda")
  map("odb", "application/vnd.oasis.opendocument.database")
  map("odc", "application/vnd.oasis.opendocument.chart")
  map("odf", "application/vnd.oasis.opendocument.formula")
  map("odft", "application/vnd.oasis.opendocument.formula-template")
  map("odg", "application/vnd.oasis.opendocument.graphics")
  map("odi", "application/vnd.oasis.opendocument.image")
  map("odm", "application/vnd.oasis.opendocument.text-master")
  map("odp", "application/vnd.oasis.opendocument.presentation")
  map("ods", "application/vnd.oasis.opendocument.spreadsheet")
  map("odt", "application/vnd.oasis.opendocument.text")
  map("oga", "audio/ogg")
  map("ogg", "audio/ogg")
  map("ogv", "video/ogg")
  map("ogx", "application/ogg")
  map("omdoc", "application/omdoc+xml")
  map("onepkg", "application/onenote")
  map("onetmp", "application/onenote")
  map("onetoc2", "application/onenote")
  map("onetoc", "application/onenote")
  map("opf", "application/oebps-package+xml")
  map("opml", "text/x-opml")
  map("oprc", "application/vnd.palm")
  map("org", "application/vnd.lotus-organizer")
  map("osf", "application/vnd.yamaha.openscoreformat")
  map("osfpvg", "application/vnd.yamaha.openscoreformat.osfpvg+xml")
  map("otc", "application/vnd.oasis.opendocument.chart-template")
  map("otf", "application/x-font-otf")
  map("otg", "application/vnd.oasis.opendocument.graphics-template")
  map("oth", "application/vnd.oasis.opendocument.text-web")
  map("oti", "application/vnd.oasis.opendocument.image-template")
  map("otp", "application/vnd.oasis.opendocument.presentation-template")
  map("ots", "application/vnd.oasis.opendocument.spreadsheet-template")
  map("ott", "application/vnd.oasis.opendocument.text-template")
  map("oxps", "application/oxps")
  map("oxt", "application/vnd.openofficeorg.extension")
  map("p10", "application/pkcs10")
  map("p12", "application/x-pkcs12")
  map("p7b", "application/x-pkcs7-certificates")
  map("p7c", "application/pkcs7-mime")
  map("p7m", "application/pkcs7-mime")
  map("p7r", "application/x-pkcs7-certreqresp")
  map("p7s", "application/pkcs7-signature")
  map("p8", "application/pkcs8")
  map("pas", "text/x-pascal")
  map("paw", "application/vnd.pawaafile")
  map("pbd", "application/vnd.powerbuilder6")
  map("pbm", "image/x-portable-bitmap")
  map("pcap", "application/vnd.tcpdump.pcap")
  map("pcf", "application/x-font-pcf")
  map("pcl", "application/vnd.hp-pcl")
  map("pclxl", "application/vnd.hp-pclxl")
  map("pct", "image/x-pict")
  map("pcurl", "application/vnd.curl.pcurl")
  map("pcx", "image/x-pcx")
  map("pdb", "application/vnd.palm")
  map("pdf", "application/pdf")
  map("pfa", "application/x-font-type1")
  map("pfb", "application/x-font-type1")
  map("pfm", "application/x-font-type1")
  map("pfr", "application/font-tdpfr")
  map("pfx", "application/x-pkcs12")
  map("pgm", "image/x-portable-graymap")
  map("pgn", "application/x-chess-pgn")
  map("pgp", "application/pgp-encrypted")
  map("pic", "image/x-pict")
  map("pkg", "application/octet-stream")
  map("pki", "application/pkixcmp")
  map("pkipath", "application/pkix-pkipath")
  map("plb", "application/vnd.3gpp.pic-bw-large")
  map("plc", "application/vnd.mobius.plc")
  map("plf", "application/vnd.pocketlearn")
  map("pls", "application/pls+xml")
  map("pml", "application/vnd.ctc-posml")
  map("png", "image/png")
  map("pnm", "image/x-portable-anymap")
  map("portpkg", "application/vnd.macports.portpkg")
  map("pot", "application/vnd.ms-powerpoint")
  map("potm", "application/vnd.ms-powerpoint.template.macroenabled.12")
  map("potx", "application/vnd.openxmlformats-officedocument.presentationml.template")
  map("ppam", "application/vnd.ms-powerpoint.addin.macroenabled.12")
  map("ppd", "application/vnd.cups-ppd")
  map("ppm", "image/x-portable-pixmap")
  map("pps", "application/vnd.ms-powerpoint")
  map("ppsm", "application/vnd.ms-powerpoint.slideshow.macroenabled.12")
  map("ppsx", "application/vnd.openxmlformats-officedocument.presentationml.slideshow")
  map("ppt", "application/vnd.ms-powerpoint")
  map("pptm", "application/vnd.ms-powerpoint.presentation.macroenabled.12")
  map("pptx", "application/vnd.openxmlformats-officedocument.presentationml.presentation")
  map("pqa", "application/vnd.palm")
  map("prc", "application/x-mobipocket-ebook")
  map("pre", "application/vnd.lotus-freelance")
  map("prf", "application/pics-rules")
  map("ps", "application/postscript")
  map("psb", "application/vnd.3gpp.pic-bw-small")
  map("psd", "image/vnd.adobe.photoshop")
  map("psf", "application/x-font-linux-psf")
  map("pskcxml", "application/pskc+xml")
  map("p", "text/x-pascal")
  map("ptid", "application/vnd.pvi.ptid1")
  map("pub", "application/x-mspublisher")
  map("pvb", "application/vnd.3gpp.pic-bw-var")
  map("pwn", "application/vnd.3m.post-it-notes")
  map("pya", "audio/vnd.ms-playready.media.pya")
  map("pyv", "video/vnd.ms-playready.media.pyv")
  map("qam", "application/vnd.epson.quickanime")
  map("qbo", "application/vnd.intu.qbo")
  map("qfx", "application/vnd.intu.qfx")
  map("qps", "application/vnd.publishare-delta-tree")
  map("qt", "video/quicktime")
  map("qwd", "application/vnd.quark.quarkxpress")
  map("qwt", "application/vnd.quark.quarkxpress")
  map("qxb", "application/vnd.quark.quarkxpress")
  map("qxd", "application/vnd.quark.quarkxpress")
  map("qxl", "application/vnd.quark.quarkxpress")
  map("qxt", "application/vnd.quark.quarkxpress")
  map("ra", "audio/x-pn-realaudio")
  map("ram", "audio/x-pn-realaudio")
  map("rar", "application/x-rar-compressed")
  map("ras", "image/x-cmu-raster")
  map("rcprofile", "application/vnd.ipunplugged.rcprofile")
  map("rdf", "application/rdf+xml")
  map("rdz", "application/vnd.data-vision.rdz")
  map("rep", "application/vnd.businessobjects")
  map("res", "application/x-dtbresource+xml")
  map("rgb", "image/x-rgb")
  map("rif", "application/reginfo+xml")
  map("rip", "audio/vnd.rip")
  map("ris", "application/x-research-info-systems")
  map("rl", "application/resource-lists+xml")
  map("rlc", "image/vnd.fujixerox.edmics-rlc")
  map("rld", "application/resource-lists-diff+xml")
  map("rm", "application/vnd.rn-realmedia")
  map("rmi", "audio/midi")
  map("rmp", "audio/x-pn-realaudio-plugin")
  map("rms", "application/vnd.jcp.javame.midlet-rms")
  map("rmvb", "application/vnd.rn-realmedia-vbr")
  map("rnc", "application/relax-ng-compact-syntax")
  map("roa", "application/rpki-roa")
  map("roff", "text/troff")
  map("rp9", "application/vnd.cloanto.rp9")
  map("rpss", "application/vnd.nokia.radio-presets")
  map("rpst", "application/vnd.nokia.radio-preset")
  map("rq", "application/sparql-query")
  map("rs", "application/rls-services+xml")
  map("rsd", "application/rsd+xml")
  map("rss", "application/rss+xml")
  map("rtf", "application/rtf")
  map("rtx", "text/richtext")
  map("s3m", "audio/s3m")
  map("saf", "application/vnd.yamaha.smaf-audio")
  map("sbml", "application/sbml+xml")
  map("sc", "application/vnd.ibm.secure-container")
  map("scd", "application/x-msschedule")
  map("scm", "application/vnd.lotus-screencam")
  map("scq", "application/scvp-cv-request")
  map("scs", "application/scvp-cv-response")
  map("scurl", "text/vnd.curl.scurl")
  map("sda", "application/vnd.stardivision.draw")
  map("sdc", "application/vnd.stardivision.calc")
  map("sdd", "application/vnd.stardivision.impress")
  map("sdkd", "application/vnd.solent.sdkm+xml")
  map("sdkm", "application/vnd.solent.sdkm+xml")
  map("sdp", "application/sdp")
  map("sdw", "application/vnd.stardivision.writer")
  map("see", "application/vnd.seemail")
  map("seed", "application/vnd.fdsn.seed")
  map("sema", "application/vnd.sema")
  map("semd", "application/vnd.semd")
  map("semf", "application/vnd.semf")
  map("ser", "application/java-serialized-object")
  map("setpay", "application/set-payment-initiation")
  map("setreg", "application/set-registration-initiation")
  map("sfd-hdstx", "application/vnd.hydrostatix.sof-data")
  map("sfs", "application/vnd.spotfire.sfs")
  map("sfv", "text/x-sfv")
  map("sgi", "image/sgi")
  map("sgl", "application/vnd.stardivision.writer-global")
  map("sgml", "text/sgml")
  map("sgm", "text/sgml")
  map("sh", "application/x-sh")
  map("shar", "application/x-shar")
  map("shf", "application/shf+xml")
  map("sid", "image/x-mrsid-image")
  map("sig", "application/pgp-signature")
  map("sil", "audio/silk")
  map("silo", "model/mesh")
  map("sis", "application/vnd.symbian.install")
  map("sisx", "application/vnd.symbian.install")
  map("sit", "application/x-stuffit")
  map("sitx", "application/x-stuffitx")
  map("skd", "application/vnd.koan")
  map("skm", "application/vnd.koan")
  map("skp", "application/vnd.koan")
  map("skt", "application/vnd.koan")
  map("sldm", "application/vnd.ms-powerpoint.slide.macroenabled.12")
  map("sldx", "application/vnd.openxmlformats-officedocument.presentationml.slide")
  map("slt", "application/vnd.epson.salt")
  map("sm", "application/vnd.stepmania.stepchart")
  map("smf", "application/vnd.stardivision.math")
  map("smi", "application/smil+xml")
  map("smil", "application/smil+xml")
  map("smv", "video/x-smv")
  map("smzip", "application/vnd.stepmania.package")
  map("snd", "audio/basic")
  map("snf", "application/x-font-snf")
  map("so", "application/octet-stream")
  map("spc", "application/x-pkcs7-certificates")
  map("spf", "application/vnd.yamaha.smaf-phrase")
  map("spl", "application/x-futuresplash")
  map("spot", "text/vnd.in3d.spot")
  map("spp", "application/scvp-vp-response")
  map("spq", "application/scvp-vp-request")
  map("spx", "audio/ogg")
  map("sql", "application/x-sql")
  map("src", "application/x-wais-source")
  map("srt", "application/x-subrip")
  map("sru", "application/sru+xml")
  map("srx", "application/sparql-results+xml")
  map("ssdl", "application/ssdl+xml")
  map("sse", "application/vnd.kodak-descriptor")
  map("ssf", "application/vnd.epson.ssf")
  map("ssml", "application/ssml+xml")
  map("st", "application/vnd.sailingtracker.track")
  map("stc", "application/vnd.sun.xml.calc.template")
  map("std", "application/vnd.sun.xml.draw.template")
  map("s", "text/x-asm")
  map("stf", "application/vnd.wt.stf")
  map("sti", "application/vnd.sun.xml.impress.template")
  map("stk", "application/hyperstudio")
  map("stl", "application/vnd.ms-pki.stl")
  map("str", "application/vnd.pg.format")
  map("stw", "application/vnd.sun.xml.writer.template")
  map("sub", "image/vnd.dvb.subtitle")
  map("sub", "text/vnd.dvb.subtitle")
  map("sus", "application/vnd.sus-calendar")
  map("susp", "application/vnd.sus-calendar")
  map("sv4cpio", "application/x-sv4cpio")
  map("sv4crc", "application/x-sv4crc")
  map("svc", "application/vnd.dvb.service")
  map("svd", "application/vnd.svd")
  map("svg", "image/svg+xml")
  map("svgz", "image/svg+xml")
  map("swa", "application/x-director")
  map("swf", "application/x-shockwave-flash")
  map("swi", "application/vnd.aristanetworks.swi")
  map("sxc", "application/vnd.sun.xml.calc")
  map("sxd", "application/vnd.sun.xml.draw")
  map("sxg", "application/vnd.sun.xml.writer.global")
  map("sxi", "application/vnd.sun.xml.impress")
  map("sxm", "application/vnd.sun.xml.math")
  map("sxw", "application/vnd.sun.xml.writer")
  map("t3", "application/x-t3vm-image")
  map("taglet", "application/vnd.mynfc")
  map("tao", "application/vnd.tao.intent-module-archive")
  map("tar", "application/x-tar")
  map("tcap", "application/vnd.3gpp2.tcap")
  map("tcl", "application/x-tcl")
  map("teacher", "application/vnd.smart.teacher")
  map("tei", "application/tei+xml")
  map("teicorpus", "application/tei+xml")
  map("tex", "application/x-tex")
  map("texi", "application/x-texinfo")
  map("texinfo", "application/x-texinfo")
  map("text", "text/plain")
  map("tfi", "application/thraud+xml")
  map("tfm", "application/x-tex-tfm")
  map("tga", "image/x-tga")
  map("thmx", "application/vnd.ms-officetheme")
  map("tiff", "image/tiff")
  map("tif", "image/tiff")
  map("tmo", "application/vnd.tmobile-livetv")
  map("torrent", "application/x-bittorrent")
  map("tpl", "application/vnd.groove-tool-template")
  map("tpt", "application/vnd.trid.tpt")
  map("tra", "application/vnd.trueapp")
  map("trm", "application/x-msterminal")
  map("tr", "text/troff")
  map("tsd", "application/timestamped-data")
  map("tsv", "text/tab-separated-values")
  map("ttc", "application/x-font-ttf")
  map("t", "text/troff")
  map("ttf", "application/x-font-ttf")
  map("ttl", "text/turtle")
  map("twd", "application/vnd.simtech-mindmapper")
  map("twds", "application/vnd.simtech-mindmapper")
  map("txd", "application/vnd.genomatix.tuxedo")
  map("txf", "application/vnd.mobius.txf")
  map("txt", "text/plain")
  map("u32", "application/x-authorware-bin")
  map("udeb", "application/x-debian-package")
  map("ufd", "application/vnd.ufdl")
  map("ufdl", "application/vnd.ufdl")
  map("ulx", "application/x-glulx")
  map("umj", "application/vnd.umajin")
  map("unityweb", "application/vnd.unity")
  map("uoml", "application/vnd.uoml+xml")
  map("uris", "text/uri-list")
  map("uri", "text/uri-list")
  map("urls", "text/uri-list")
  map("ustar", "application/x-ustar")
  map("utz", "application/vnd.uiq.theme")
  map("uu", "text/x-uuencode")
  map("uva", "audio/vnd.dece.audio")
  map("uvd", "application/vnd.dece.data")
  map("uvf", "application/vnd.dece.data")
  map("uvg", "image/vnd.dece.graphic")
  map("uvh", "video/vnd.dece.hd")
  map("uvi", "image/vnd.dece.graphic")
  map("uvm", "video/vnd.dece.mobile")
  map("uvp", "video/vnd.dece.pd")
  map("uvs", "video/vnd.dece.sd")
  map("uvt", "application/vnd.dece.ttml+xml")
  map("uvu", "video/vnd.uvvu.mp4")
  map("uvva", "audio/vnd.dece.audio")
  map("uvvd", "application/vnd.dece.data")
  map("uvvf", "application/vnd.dece.data")
  map("uvvg", "image/vnd.dece.graphic")
  map("uvvh", "video/vnd.dece.hd")
  map("uvvi", "image/vnd.dece.graphic")
  map("uvvm", "video/vnd.dece.mobile")
  map("uvvp", "video/vnd.dece.pd")
  map("uvvs", "video/vnd.dece.sd")
  map("uvvt", "application/vnd.dece.ttml+xml")
  map("uvvu", "video/vnd.uvvu.mp4")
  map("uvv", "video/vnd.dece.video")
  map("uvvv", "video/vnd.dece.video")
  map("uvvx", "application/vnd.dece.unspecified")
  map("uvvz", "application/vnd.dece.zip")
  map("uvx", "application/vnd.dece.unspecified")
  map("uvz", "application/vnd.dece.zip")
  map("vcard", "text/vcard")
  map("vcd", "application/x-cdlink")
  map("vcf", "text/x-vcard")
  map("vcg", "application/vnd.groove-vcard")
  map("vcs", "text/x-vcalendar")
  map("vcx", "application/vnd.vcx")
  map("vis", "application/vnd.visionary")
  map("viv", "video/vnd.vivo")
  map("vob", "video/x-ms-vob")
  map("vor", "application/vnd.stardivision.writer")
  map("vox", "application/x-authorware-bin")
  map("vrml", "model/vrml")
  map("vsd", "application/vnd.visio")
  map("vsf", "application/vnd.vsf")
  map("vss", "application/vnd.visio")
  map("vst", "application/vnd.visio")
  map("vsw", "application/vnd.visio")
  map("vtu", "model/vnd.vtu")
  map("vxml", "application/voicexml+xml")
  map("w3d", "application/x-director")
  map("wad", "application/x-doom")
  map("wav", "audio/x-wav")
  map("wax", "audio/x-ms-wax")
  map("wbmp", "image/vnd.wap.wbmp")
  map("wbs", "application/vnd.criticaltools.wbs+xml")
  map("wbxml", "application/vnd.wap.wbxml")
  map("wcm", "application/vnd.ms-works")
  map("wdb", "application/vnd.ms-works")
  map("wdp", "image/vnd.ms-photo")
  map("weba", "audio/webm")
  map("webm", "video/webm")
  map("webp", "image/webp")
  map("wg", "application/vnd.pmi.widget")
  map("wgt", "application/widget")
  map("wks", "application/vnd.ms-works")
  map("wma", "audio/x-ms-wma")
  map("wmd", "application/x-ms-wmd")
  map("wmf", "application/x-msmetafile")
  map("wmlc", "application/vnd.wap.wmlc")
  map("wmlsc", "application/vnd.wap.wmlscriptc")
  map("wmls", "text/vnd.wap.wmlscript")
  map("wml", "text/vnd.wap.wml")
  map("wm", "video/x-ms-wm")
  map("wmv", "video/x-ms-wmv")
  map("wmx", "video/x-ms-wmx")
  map("wmz", "application/x-msmetafile")
  map("wmz", "application/x-ms-wmz")
  map("woff", "application/font-woff")
  map("wpd", "application/vnd.wordperfect")
  map("wpl", "application/vnd.ms-wpl")
  map("wps", "application/vnd.ms-works")
  map("wqd", "application/vnd.wqd")
  map("wri", "application/x-mswrite")
  map("wrl", "model/vrml")
  map("wsdl", "application/wsdl+xml")
  map("wspolicy", "application/wspolicy+xml")
  map("wtb", "application/vnd.webturbo")
  map("wvx", "video/x-ms-wvx")
  map("x32", "application/x-authorware-bin")
  map("x3db", "model/x3d+binary")
  map("x3dbz", "model/x3d+binary")
  map("x3d", "model/x3d+xml")
  map("x3dv", "model/x3d+vrml")
  map("x3dvz", "model/x3d+vrml")
  map("x3dz", "model/x3d+xml")
  map("xaml", "application/xaml+xml")
  map("xap", "application/x-silverlight-app")
  map("xar", "application/vnd.xara")
  map("xbap", "application/x-ms-xbap")
  map("xbd", "application/vnd.fujixerox.docuworks.binder")
  map("xbm", "image/x-xbitmap")
  map("xdf", "application/xcap-diff+xml")
  map("xdm", "application/vnd.syncml.dm+xml")
  map("xdp", "application/vnd.adobe.xdp+xml")
  map("xdssc", "application/dssc+xml")
  map("xdw", "application/vnd.fujixerox.docuworks")
  map("xenc", "application/xenc+xml")
  map("xer", "application/patch-ops-error+xml")
  map("xfdf", "application/vnd.adobe.xfdf")
  map("xfdl", "application/vnd.xfdl")
  map("xht", "application/xhtml+xml")
  map("xhtml", "application/xhtml+xml")
  map("xhvml", "application/xv+xml")
  map("xif", "image/vnd.xiff")
  map("xla", "application/vnd.ms-excel")
  map("xlam", "application/vnd.ms-excel.addin.macroenabled.12")
  map("xlc", "application/vnd.ms-excel")
  map("xlf", "application/x-xliff+xml")
  map("xlm", "application/vnd.ms-excel")
  map("xls", "application/vnd.ms-excel")
  map("xlsb", "application/vnd.ms-excel.sheet.binary.macroenabled.12")
  map("xlsm", "application/vnd.ms-excel.sheet.macroenabled.12")
  map("xlsx", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet")
  map("xlt", "application/vnd.ms-excel")
  map("xltm", "application/vnd.ms-excel.template.macroenabled.12")
  map("xltx", "application/vnd.openxmlformats-officedocument.spreadsheetml.template")
  map("xlw", "application/vnd.ms-excel")
  map("xm", "audio/xm")
  map("xml", "text/xml")
  map("xo", "application/vnd.olpc-sugar")
  map("xop", "application/xop+xml")
  map("xpi", "application/x-xpinstall")
  map("xpl", "application/xproc+xml")
  map("xpm", "image/x-xpixmap")
  map("xpr", "application/vnd.is-xpr")
  map("xps", "application/vnd.ms-xpsdocument")
  map("xpw", "application/vnd.intercon.formnet")
  map("xpx", "application/vnd.intercon.formnet")
  map("xsl", "application/xml")
  map("xslt", "application/xslt+xml")
  map("xsm", "application/vnd.syncml+xml")
  map("xspf", "application/xspf+xml")
  map("xul", "application/vnd.mozilla.xul+xml")
  map("xvm", "application/xv+xml")
  map("xvml", "application/xv+xml")
  map("xwd", "image/x-xwindowdump")
  map("xyz", "chemical/x-xyz")
  map("xz", "application/x-xz")
  map("yang", "application/yang")
  map("yin", "application/yin+xml")
  map("z1", "application/x-zmachine")
  map("z2", "application/x-zmachine")
  map("z3", "application/x-zmachine")
  map("z4", "application/x-zmachine")
  map("z5", "application/x-zmachine")
  map("z6", "application/x-zmachine")
  map("z7", "application/x-zmachine")
  map("z8", "application/x-zmachine")
  map("zaz", "application/vnd.zzazz.deck+xml")
  map("zip", "application/zip")
  map("zir", "application/vnd.zul")
  map("zirz", "application/vnd.zul")
  map("zmm", "application/vnd.handheld-entertainment+xml")

  def parse(contentTypeString: String): ContentType = {
    val parts = contentTypeString.split(';')
    val part = parts(0).trim
    val split = part.indexOf('/')
    val `type` = part.substring(0, split)
    val subType = part.substring(split + 1)
    var contentType = ContentType(`type`, subType)
    parts.tail.foreach { s =>
      val block = s.trim
      val divider = block.indexOf('=')
      if (divider == -1) {
        throw new RuntimeException(s"Unable to parse content type: [$contentTypeString]")
      }
      val name = block.substring(0, divider)
      val value = block.substring(divider + 1)
      contentType = contentType.withExtra(name, value)
    }
    contentType
  }

  def byFileName(fileName: String): ContentType = getByFileName(fileName).getOrElse(ContentType.`application/octet-stream`)
  def byExtension(extension: String): ContentType = getByExtension(extension).getOrElse(ContentType.`application/octet-stream`)

  def getByFileName(fileName: String): Option[ContentType] = {
    val index = fileName.lastIndexOf('.')
    if (index == -1) {
      None
    } else {
      val extension = fileName.substring(index + 1)
      getByExtension(extension)
    }
  }

  def getByExtension(extension: String): Option[ContentType] = {
    extension2MimeType.get(extension.toLowerCase).map(new ContentType(_))
  }

  implicit val encoder: Encoder[ContentType] = new Encoder[ContentType] {
    override def apply(ct: ContentType): Json = Json.fromString(ct.outputString)
  }

  implicit val decoder: Decoder[ContentType] = new Decoder[ContentType] {
    override def apply(c: HCursor): Result[ContentType] = c.value.asString match {
      case Some(value) => Right(parse(value))
      case None => Left(DecodingFailure(s"Unable to decode ContentType from ${c.value}", Nil))
    }
  }
}