package io.youi

class Key private[youi](val value: String, val description: String, val `type`: KeyType) {
  Key.register(this)

  override def toString: String = s"Key(value: $value, description: $description, type: ${`type`})"
}

class CharacterKey private[youi](val char: Char,
                                 description: String,
                                 `type`: KeyType,
                                 reverse: => CharacterKey) extends Key(char.toString, description, `type`) {
  def isLower: Boolean = char.isLower
  def isUpper: Boolean = char.isUpper
  def toUpper: CharacterKey = if (isUpper) this else reverse
  def toLower: CharacterKey = if (isLower) this else reverse

  override def toString: String = s"CharacterKey(char: $char, description: $description)"
}

class SymbolKey private[youi](val char: Char,
                              description: String) extends Key(char.toString, description, KeyType.Symbols) {
  override def toString: String = s"SymbolKey(char: $char, description: $description)"
}

object Key {
  private var map = Map.empty[String, Key]

  // Lowercase
  val a: CharacterKey = new CharacterKey('a', "Lowercase character 'a'.", KeyType.Lowercase, Key.A)
  val b: CharacterKey = new CharacterKey('b', "Lowercase character 'b'.", KeyType.Lowercase, Key.B)
  val c: CharacterKey = new CharacterKey('c', "Lowercase character 'c'.", KeyType.Lowercase, Key.C)
  val d: CharacterKey = new CharacterKey('d', "Lowercase character 'd'.", KeyType.Lowercase, Key.D)
  val e: CharacterKey = new CharacterKey('e', "Lowercase character 'e'.", KeyType.Lowercase, Key.E)
  val f: CharacterKey = new CharacterKey('f', "Lowercase character 'f'.", KeyType.Lowercase, Key.F)
  val g: CharacterKey = new CharacterKey('g', "Lowercase character 'g'.", KeyType.Lowercase, Key.G)
  val h: CharacterKey = new CharacterKey('h', "Lowercase character 'h'.", KeyType.Lowercase, Key.H)
  val i: CharacterKey = new CharacterKey('i', "Lowercase character 'i'.", KeyType.Lowercase, Key.I)
  val j: CharacterKey = new CharacterKey('j', "Lowercase character 'j'.", KeyType.Lowercase, Key.J)
  val k: CharacterKey = new CharacterKey('k', "Lowercase character 'k'.", KeyType.Lowercase, Key.K)
  val l: CharacterKey = new CharacterKey('l', "Lowercase character 'l'.", KeyType.Lowercase, Key.L)
  val m: CharacterKey = new CharacterKey('m', "Lowercase character 'm'.", KeyType.Lowercase, Key.M)
  val n: CharacterKey = new CharacterKey('n', "Lowercase character 'n'.", KeyType.Lowercase, Key.N)
  val o: CharacterKey = new CharacterKey('o', "Lowercase character 'o'.", KeyType.Lowercase, Key.O)
  val p: CharacterKey = new CharacterKey('p', "Lowercase character 'p'.", KeyType.Lowercase, Key.P)
  val q: CharacterKey = new CharacterKey('q', "Lowercase character 'q'.", KeyType.Lowercase, Key.Q)
  val r: CharacterKey = new CharacterKey('r', "Lowercase character 'r'.", KeyType.Lowercase, Key.R)
  val s: CharacterKey = new CharacterKey('s', "Lowercase character 's'.", KeyType.Lowercase, Key.S)
  val t: CharacterKey = new CharacterKey('t', "Lowercase character 't'.", KeyType.Lowercase, Key.T)
  val u: CharacterKey = new CharacterKey('u', "Lowercase character 'u'.", KeyType.Lowercase, Key.U)
  val v: CharacterKey = new CharacterKey('v', "Lowercase character 'v'.", KeyType.Lowercase, Key.V)
  val w: CharacterKey = new CharacterKey('w', "Lowercase character 'w'.", KeyType.Lowercase, Key.W)
  val x: CharacterKey = new CharacterKey('x', "Lowercase character 'x'.", KeyType.Lowercase, Key.X)
  val y: CharacterKey = new CharacterKey('y', "Lowercase character 'y'.", KeyType.Lowercase, Key.Y)
  val z: CharacterKey = new CharacterKey('z', "Lowercase character 'z'.", KeyType.Lowercase, Key.Z)

  // Uppercase
  val A: CharacterKey = new CharacterKey('A', "Uppercase character 'A'.", KeyType.Uppercase, Key.a)
  val B: CharacterKey = new CharacterKey('B', "Uppercase character 'B'.", KeyType.Uppercase, Key.b)
  val C: CharacterKey = new CharacterKey('C', "Uppercase character 'C'.", KeyType.Uppercase, Key.c)
  val D: CharacterKey = new CharacterKey('D', "Uppercase character 'D'.", KeyType.Uppercase, Key.d)
  val E: CharacterKey = new CharacterKey('E', "Uppercase character 'E'.", KeyType.Uppercase, Key.e)
  val F: CharacterKey = new CharacterKey('F', "Uppercase character 'F'.", KeyType.Uppercase, Key.f)
  val G: CharacterKey = new CharacterKey('G', "Uppercase character 'G'.", KeyType.Uppercase, Key.g)
  val H: CharacterKey = new CharacterKey('H', "Uppercase character 'H'.", KeyType.Uppercase, Key.h)
  val I: CharacterKey = new CharacterKey('I', "Uppercase character 'I'.", KeyType.Uppercase, Key.i)
  val J: CharacterKey = new CharacterKey('J', "Uppercase character 'J'.", KeyType.Uppercase, Key.j)
  val K: CharacterKey = new CharacterKey('K', "Uppercase character 'K'.", KeyType.Uppercase, Key.k)
  val L: CharacterKey = new CharacterKey('L', "Uppercase character 'L'.", KeyType.Uppercase, Key.l)
  val M: CharacterKey = new CharacterKey('M', "Uppercase character 'M'.", KeyType.Uppercase, Key.m)
  val N: CharacterKey = new CharacterKey('N', "Uppercase character 'N'.", KeyType.Uppercase, Key.n)
  val O: CharacterKey = new CharacterKey('O', "Uppercase character 'O'.", KeyType.Uppercase, Key.o)
  val P: CharacterKey = new CharacterKey('P', "Uppercase character 'P'.", KeyType.Uppercase, Key.p)
  val Q: CharacterKey = new CharacterKey('Q', "Uppercase character 'Q'.", KeyType.Uppercase, Key.q)
  val R: CharacterKey = new CharacterKey('R', "Uppercase character 'R'.", KeyType.Uppercase, Key.r)
  val S: CharacterKey = new CharacterKey('S', "Uppercase character 'S'.", KeyType.Uppercase, Key.s)
  val T: CharacterKey = new CharacterKey('T', "Uppercase character 'T'.", KeyType.Uppercase, Key.t)
  val U: CharacterKey = new CharacterKey('U', "Uppercase character 'U'.", KeyType.Uppercase, Key.u)
  val V: CharacterKey = new CharacterKey('V', "Uppercase character 'V'.", KeyType.Uppercase, Key.v)
  val W: CharacterKey = new CharacterKey('W', "Uppercase character 'W'.", KeyType.Uppercase, Key.w)
  val X: CharacterKey = new CharacterKey('X', "Uppercase character 'X'.", KeyType.Uppercase, Key.x)
  val Y: CharacterKey = new CharacterKey('Y', "Uppercase character 'Y'.", KeyType.Uppercase, Key.y)
  val Z: CharacterKey = new CharacterKey('Z', "Uppercase character 'Z'.", KeyType.Uppercase, Key.z)

  // Symbols
  val QuestionMark: SymbolKey = new SymbolKey('?', "Question Mark")
  val Equals: SymbolKey = new SymbolKey('=', "Equals")
  val GreaterThan: SymbolKey = new SymbolKey('>', "Greater Than")
  val LessThan: SymbolKey = new SymbolKey('<', "Less Than")
  val Semicolon: SymbolKey = new SymbolKey(';', "Semicolon")
  val Colon: SymbolKey = new SymbolKey(':', "Colon")
  val BackSlash: SymbolKey = new SymbolKey('\\', "Back Slash")
  val CloseBracket: SymbolKey = new SymbolKey(']', "Close Bracket")
  val OpenBracket: SymbolKey = new SymbolKey('[', "Open Bracket")
  val BackQuote: SymbolKey = new SymbolKey('`', "Back Quote")
  val ForwardSlash: SymbolKey = new SymbolKey('/', "Forward Slash")
  val Period: SymbolKey = new SymbolKey('.', "Period")
  val Comma: SymbolKey = new SymbolKey(',', "Comma")
  val CloseCurlyBracket: SymbolKey = new SymbolKey('}', "Close Curly Bracket")
  val OpenCurlyBracket: SymbolKey = new SymbolKey('{', "Open Curly Bracket")
  val Minus: SymbolKey = new SymbolKey('-', "Minus")
  val Pipe: SymbolKey = new SymbolKey('|', "Pipe")
  val SingleQuote: SymbolKey = new SymbolKey('\'', "Single Quote")
  val DoubleQuote: SymbolKey = new SymbolKey('"', "Double Quote")
  val Plus: SymbolKey = new SymbolKey('+', "Plus")
  val Underscore: SymbolKey = new SymbolKey('_', "Underscore")
  val CloseParen: SymbolKey = new SymbolKey(')', "Close Paren")
  val OpenParen: SymbolKey = new SymbolKey('(', "Open Paren")
  val Asterisk: SymbolKey = new SymbolKey('*', "Asterisk")
  val Ampersand: SymbolKey = new SymbolKey('&', "Ampersand")
  val Circumflex: SymbolKey = new SymbolKey('^', "Circumflex")
  val Percent: SymbolKey = new SymbolKey('%', "Percent")
  val Dollar: SymbolKey = new SymbolKey('$', "Dollar")
  val Hash: SymbolKey = new SymbolKey('#', "Hash")
  val At: SymbolKey = new SymbolKey('@', "At")
  val Exclamation: SymbolKey = new SymbolKey('!', "Exclamation")
  val Tilde: SymbolKey = new SymbolKey('~', "Tilde")

  // Numeric
  val Zero: Key = new Key("0", "Numeric value '0'.", KeyType.Numeric)
  val One: Key = new Key("1", "Numeric value '1'.", KeyType.Numeric)
  val Two: Key = new Key("2", "Numeric value '2'.", KeyType.Numeric)
  val Three: Key = new Key("3", "Numeric value '3'.", KeyType.Numeric)
  val Four: Key = new Key("4", "Numeric value '4'.", KeyType.Numeric)
  val Five: Key = new Key("5", "Numeric value '5'.", KeyType.Numeric)
  val Six: Key = new Key("6", "Numeric value '6'.", KeyType.Numeric)
  val Seven: Key = new Key("7", "Numeric value '7'.", KeyType.Numeric)
  val Eight: Key = new Key("8", "Numeric value '8'.", KeyType.Numeric)
  val Nine: Key = new Key("9", "Numeric value '9'.", KeyType.Numeric)
  val Decimal: Key = new Key("Decimal", "The decimal point key (typically . or , depending on the region. In newer browsers, this value to simply be the character generated by the decimal key (one of those two characters). [1]", KeyType.Numeric)
  val Key11: Key = new Key("Key11", "The 11 key found on certain media numeric keypads.", KeyType.Numeric)
  val Key12: Key = new Key("Key12", "The 12 key found on certain media numeric keypads.", KeyType.Numeric)
  val Multiply: Key = new Key("Multiply", "The numeric keypad's multiplication key, *.", KeyType.Numeric)
  val Add: Key = new Key("Add", "The numeric keypad's addition key, +.", KeyType.Numeric)
  val Divide: Key = new Key("Divide", "The numeric keypad's division key, /.", KeyType.Numeric)
  val Subtract: Key = new Key("Subtract", "The numeric keypad's subtraction key, -.", KeyType.Numeric)
  val Separator: Key = new Key("Separator", "The numeric keypad's places separator character (in the United States, this is a comma, but elsewhere it is frequently a period).", KeyType.Numeric)
         
  // Special
  val Unidentified: Key = new Key("Unidentified", "The user agent wasn't able to map the event's virtual keycode to a specific key value. This can happen due to hardware or software constraints, or because of constraints around the platform on which the user agent is running.", KeyType.Special)
         
  // Modifier
  val Alt: Key = new Key("Alt", "The Alt (Alternative) key.", KeyType.Modifier)
  val AltGraph: Key = new Key("AltGraph", "The AltGr or AltGraph (Alternate Graphics) key. Enables the ISO Level 3 shift modifier (where Shift is the level 2 modifier).", KeyType.Modifier)
  val CapsLock: Key = new Key("CapsLock", "The Caps Lock key. Toggles the capital character lock on and off for subsequent input.", KeyType.Modifier)
  val Control: Key = new Key("Control", "The Control, Ctrl, or Ctl key. Allows typing control characters.", KeyType.Modifier)
  val Fn: Key = new Key("Fn", "The Fn (Function modifier) key. Used to allow generating function key (F1-F15, for instance) characters on keyboards without a dedicated function key area. Often handled in hardware so that events aren't generated for this key.", KeyType.Modifier)
  val FnLock: Key = new Key("FnLock", "The FnLock or F-Lock (Function Lock) key.Toggles the function key mode described by \"Fn\" on and off. Often handled in hardware so that events aren't generated for this key.", KeyType.Modifier)
  val Hyper: Key = new Key("Hyper", "The Hyper key.", KeyType.Modifier)
  val Meta: Key = new Key("Meta", "The Meta key. Allows issuing special command inputs. This is the Windows logo key, or the Command or ⌘ key on Mac keyboards.", KeyType.Modifier)
  val NumLock: Key = new Key("NumLock", "The NumLock (Number Lock) key. Toggles the numeric keypad between number entry some other mode (often directional arrows).", KeyType.Modifier)
  val ScrollLock: Key = new Key("ScrollLock", "The Scroll Lock key. Toggles beteen scrolling and cursor movement modes.", KeyType.Modifier)
  val Shift: Key = new Key("Shift", "The Shift key. Modifies keystrokes to allow typing upper (or other) case letters, and to support typing punctuation and other special characters.", KeyType.Modifier)
  val Super: Key = new Key("Super", "The Super key.", KeyType.Modifier)
  val Symbol: Key = new Key("Symbol", "The Symbol modifier key (found on certain virtual keyboards).", KeyType.Modifier)
  val SymbolLock: Key = new Key("SymbolLock", "The Symbol Lock key.", KeyType.Modifier)
         
  // Whitespace
  val Enter: Key = new Key("Enter", "The Enter or ↵ key (sometimes labeled Return).", KeyType.Whitespace)
  val Tab: Key = new Key("Tab", "The Horizontal Tab key, Tab.", KeyType.Whitespace)
  val Space: Key = new Key(" ", "The space key, Space Bar.", KeyType.Whitespace)
         
  // Navigation
  val ArrowDown: Key = new Key("ArrowDown", "The down arrow key.", KeyType.Navigation)
  val ArrowLeft: Key = new Key("ArrowLeft", "The left arrow key.", KeyType.Navigation)
  val ArrowRight: Key = new Key("ArrowRight", "The right arrow key.", KeyType.Navigation)
  val ArrowUp: Key = new Key("ArrowUp", "The up arrow key.", KeyType.Navigation)
  val End: Key = new Key("End", "The End key. Moves to the end of content.", KeyType.Navigation)
  val Home: Key = new Key("Home", "The Home key. Moves to the start of content.", KeyType.Navigation)
  val PageDown: Key = new Key("PageDown", "The Page Down (or PgDn) key. Scrolls down or displays the next page of content.", KeyType.Navigation)
  val PageUp: Key = new Key("PageUp", "The Page Up (or PgUp) key. Scrolls up or displays the previous page of content.", KeyType.Navigation)
         
  // Editing
  val Backspace: Key = new Key("Backspace", "The Backspace key. This key is labeled Delete on Mac keyboards.", KeyType.Editing)
  val Clear: Key = new Key("Clear", "The Clear key. Removes the currently selected input.", KeyType.Editing)
  val Copy: Key = new Key("Copy", "The Copy key (on certain extended keyboards).", KeyType.Editing)
  val CrSel: Key = new Key("CrSel", "The Cursor Select key, CrSel.", KeyType.Editing)
  val Cut: Key = new Key("Cut", "The Cut key (on certain extended keyboards).", KeyType.Editing)
  val Delete: Key = new Key("Delete", "The Delete key, Del.", KeyType.Editing)
  val EraseEof: Key = new Key("EraseEof", "Erase to End of Field. Deletes all characters from the current cursor position to the end of the current field.", KeyType.Editing)
  val ExSel: Key = new Key("ExSel", "The ExSel (Extend Selection) key.", KeyType.Editing)
  val Insert: Key = new Key("Insert", "The Insert key, Ins. Toggles  between inserting and overwriting text.", KeyType.Editing)
  val Paste: Key = new Key("Paste", "Paste from the clipboard.", KeyType.Editing)
  val Redo: Key = new Key("Redo", "Redo the last action.", KeyType.Editing)
  val Undo: Key = new Key("Undo", "Undo the last action.", KeyType.Editing)
         
  // UI
  val Accept: Key = new Key("Accept", "The Accept, Commit, or OK key or button. Accepts the currently selected option or input method sequence conversion.", KeyType.UI)
  val Again: Key = new Key("Again", "The Again key. Redoes or repeats a previous action.", KeyType.UI)
  val Attn: Key = new Key("Attn", "The Attn (Attention) key.", KeyType.UI)
  val Cancel: Key = new Key("Cancel", "The Cancel key.", KeyType.UI)
  val ContextMenu: Key = new Key("ContextMenu", "Shows the context menu. Typically found between the Windows (or OS) key and the Control key on the right side of the keyboard.", KeyType.UI)
  val Escape: Key = new Key("Escape", "The Esc (Escape) key. Typically used as an exit, cancel, or \"escape this operation\" button. Historically, the Escape character was used to signal the start of a special control sequence of characters called an \"escape sequence.\"", KeyType.UI)
  val Execute: Key = new Key("Execute", "The Execute key.", KeyType.UI)
  val Find: Key = new Key("Find", "The Find key. Opens an interface (typically a dialog box) for performing a find/search operation.", KeyType.UI)
  val Finish: Key = new Key("Finish", "The Finish key.", KeyType.UI)
  val Help: Key = new Key("Help", "The Help key. Opens or toggles the display of help information.", KeyType.UI)
  val Pause: Key = new Key("Pause", "The Pause key. Pauses the current application or state, if applicable. This shouldn't be confused with the \"MediaPause\" key value, which is used for media controllers, rather than to control applications and processes.", KeyType.UI)
  val Play: Key = new Key("Play", "The Play key. Resumes a previously paused application, if applicable. This shouldn't be confused with the \"MediaPlay\" key value, which is used for media controllers, rather than to control applications and processes.", KeyType.UI)
  val Props: Key = new Key("Props", "The Props (Properties) key.", KeyType.UI)
  val Select: Key = new Key("Select", "The Select key.", KeyType.UI)
  val ZoomIn: Key = new Key("ZoomIn", "The ZoomIn key.", KeyType.UI)
  val ZoomOut: Key = new Key("ZoomOut", "The ZoomOut key.", KeyType.UI)
         
  // Device
  val BrightnessDown: Key = new Key("BrightnessDown", "The Brightness Down key. Typically used to reduce the brightness of the display.", KeyType.Device)
  val BrightnessUp: Key = new Key("BrightnessUp", "The Brightness Up key. Typically increases the brightness of the display.", KeyType.Device)
  val Eject: Key = new Key("Eject", "The Eject key. Ejects removable media (or toggles an optical storage device tray open and closed).", KeyType.Device)
  val LogOff: Key = new Key("LogOff", "The LogOff key.", KeyType.Device)
  val Power: Key = new Key("Power", "The Power button or key, to toggle power on and off. Not all systems pass this key through to to the user agent.", KeyType.Device)
  val PowerOff: Key = new Key("PowerOff", "The PowerOff or PowerDown key. Shuts off the system.", KeyType.Device)
  val PrintScreen: Key = new Key("PrintScreen", "The PrintScreen or PrtScr key. Sometimes SnapShot. Captures the screen and prints it or saves it to disk.", KeyType.Device)
  val Hibernate: Key = new Key("Hibernate", "The Hibernate key. This saves the state of the computer to disk and then shuts down; the computer can be returned to its previous state by restoring the saved state information.", KeyType.Device)
  val Standby: Key = new Key("Standby", "The Standby key; also known as Suspend or Sleep. This turns off the display and puts the computer in a low power consumption mode, without completely powering off.", KeyType.Device)
  val WakeUp: Key = new Key("WakeUp", "The WakeUp key; used to wake the computer from the hibernation or standby modes.", KeyType.Device)
         
  // IME
  val AllCandidates: Key = new Key("AllCandidates", "The All Candidates key, which starts multi-candidate mode, in which multiple candidates are displayed for the ongoing input.", KeyType.IME)
  val Alphanumeric: Key = new Key("Alphanumeric", "The Alphanumeric key.", KeyType.IME)
  val CodeInput: Key = new Key("CodeInput", "The Code Input key, which enables code input mode, which lets the user enter characters by typing their code points (their Unicode character numbers, typically).", KeyType.IME)
  val Compose: Key = new Key("Compose", "The Compose key.", KeyType.IME)
  val Convert: Key = new Key("Convert", "The Convert key, which instructs the IME to convert the current input method sequence into the resulting character.", KeyType.IME)
  val Dead: Key = new Key("Dead", "A dead \"combining\" key; that is, a key which is used in tandem with other keys to generate accented and other modified characters. If pressed by itself, it doesn't generate a character. If you wish to identify which specific dead key was pressed (in cases where more than one exists), you can do so by examining the KeyboardEvent's associated compositionupdate event's  data property.", KeyType.IME)
  val FinalMode: Key = new Key("FinalMode", "The Final (Final Mode) key is used on some Asian keyboards to enter final mode when using IMEs.", KeyType.IME)
  val GroupFirst: Key = new Key("GroupFirst", "Switches to the first character group on an ISO/IEC 9995 keyboard. Each key may have multiple groups of characters, each in its own column. Pressing this key instructs the device to interpret keypresses as coming from the first column on subsequent keystrokes.", KeyType.IME)
  val GroupLast: Key = new Key("GroupLast", "Switches to the last character group on an ISO/IEC 9995 keyboard.", KeyType.IME)
  val GroupNext: Key = new Key("GroupNext", "Switches to the next character group on an ISO/IEC 9995 keyboard.", KeyType.IME)
  val GroupPrevious: Key = new Key("GroupPrevious", "Switches to the previous character group on an ISO/IEC 9995 keyboard.", KeyType.IME)
  val ModeChange: Key = new Key("ModeChange", "The Mode Change key. Toggles or cycles among input modes of IMEs.", KeyType.IME)
  val NextCandidate: Key = new Key("NextCandidate", "The Next Candidate function key. Selects the next possible match for the ongoing input.", KeyType.IME)
  val NonConvert: Key = new Key("NonConvert", "The NonConvert (\"Don't convert\") key. This accepts the current input method sequence without running conversion when using an IME.", KeyType.IME)
  val PreviousCandidate: Key = new Key("PreviousCandidate", "The Previous Candidate key. Selects the previous possible match for the ongoing input.", KeyType.IME)
  val Process: Key = new Key("Process", "The Process key. Instructs the IME to process the conversion.", KeyType.IME)
  val SingleCandidate: Key = new Key("SingleCandidate", "The Single Candidate key. Enables single candidate mode (as opposed to multi-candidate mode); in this mode, only one candidate is displayed at a time.", KeyType.IME)
  val HangulMode: Key = new Key("HangulMode", "The Hangul (Korean character set) mode key, which toggles between Hangul and English entry modes.", KeyType.IME)
  val HanjaMode: Key = new Key("HanjaMode", "Selects the Hanja mode, for converting Hangul characters to the more specific Hanja characters.", KeyType.IME)
  val JunjaMode: Key = new Key("JunjaMode", "Selects the Junja mode, in which Korean is represented using single-byte Latin characters.", KeyType.IME)
  val Eisu: Key = new Key("Eisu", "The Eisu key. This key's purpose is defined by the IME, but may be used to close the IME.", KeyType.IME)
  val Hankaku: Key = new Key("Hankaku", "The Hankaku (half-width characters) key.", KeyType.IME)
  val Hiragana: Key = new Key("Hiragana", "The Hiragana key; selects Kana characters mode.", KeyType.IME)
  val HiraganaKatakana: Key = new Key("HiraganaKatakana", "Toggles between the Hiragana and Katakana writing systems.", KeyType.IME)
  val KanaMode: Key = new Key("KanaMode", "The Kana Mode (Kana Lock) key.", KeyType.IME)
  val KanjiMode: Key = new Key("KanjiMode", "The Kanji Mode key. Enables entering Japanese text using the ideographic characters of Chinese origin.", KeyType.IME)
  val Katakana: Key = new Key("Katakana", "The Katakana key.", KeyType.IME)
  val Romaji: Key = new Key("Romaji", "The Romaji key; selects the Roman character set.", KeyType.IME)
  val Zenkaku: Key = new Key("Zenkaku", "The Zenkaku (full width) characters key.", KeyType.IME)
  val ZenkakuHanaku: Key = new Key("ZenkakuHanaku", "The Zenkaku/Hankaku (full width/half width) toggle key.", KeyType.IME)
         
  // Function
  val F1: Key = new Key("F1", "The first general-purpose function key, F1.", KeyType.Function)
  val F2: Key = new Key("F2", "The F2 key.", KeyType.Function)
  val F3: Key = new Key("F3", "The F3 key.", KeyType.Function)
  val F4: Key = new Key("F4", "The F4 key.", KeyType.Function)
  val F5: Key = new Key("F5", "The F5 key.", KeyType.Function)
  val F6: Key = new Key("F6", "The F6 key.", KeyType.Function)
  val F7: Key = new Key("F7", "The F7 key.", KeyType.Function)
  val F8: Key = new Key("F8", "The F8 key.", KeyType.Function)
  val F9: Key = new Key("F9", "The F9 key.", KeyType.Function)
  val F10: Key = new Key("F10", "The F10 key.", KeyType.Function)
  val F11: Key = new Key("F11", "The F11 key.", KeyType.Function)
  val F12: Key = new Key("F12", "The F12 key.", KeyType.Function)
  val F13: Key = new Key("F13", "The F13 key.", KeyType.Function)
  val F14: Key = new Key("F14", "The F14 key.", KeyType.Function)
  val F15: Key = new Key("F15", "The F15 key.", KeyType.Function)
  val F16: Key = new Key("F16", "The F16 key.", KeyType.Function)
  val F17: Key = new Key("F17", "The F17 key.", KeyType.Function)
  val F18: Key = new Key("F18", "The F18 key.", KeyType.Function)
  val F19: Key = new Key("F19", "The F19 key.", KeyType.Function)
  val F20: Key = new Key("F20", "The F20 key.", KeyType.Function)
  val Soft1: Key = new Key("Soft1", "The first general-purpose virtual function key.", KeyType.Function)
  val Soft2: Key = new Key("Soft2", "The second general-purpose virtual function key.", KeyType.Function)
  val Soft3: Key = new Key("Soft3", "The third general-purpose virtual function key.", KeyType.Function)
  val Soft4: Key = new Key("Soft4", "The fourth general-purpose virtual function key.", KeyType.Function)
         
  // Phone
  val AppSwitch: Key = new Key("AppSwitch", "Presents a list of recently-used applications which lets the user change apps quickly.", KeyType.Phone)
  val Call: Key = new Key("Call", "The Call key; dials the number which has been entered.", KeyType.Phone)
  val Camera: Key = new Key("Camera", "The Camera key; activates the camera.", KeyType.Phone)
  val CameraFocus: Key = new Key("CameraFocus", "The Focus key; focuses the camera.", KeyType.Phone)
  val EndCall: Key = new Key("EndCall", "The End Call or Hang Up button.", KeyType.Phone)
  val GoBack: Key = new Key("GoBack", "The Back button.", KeyType.Phone)
  val GoHome: Key = new Key("GoHome", "The Home button, which takes the user to the phone's main screen (usually an application launcher).", KeyType.Phone)
  val HeadsetHook: Key = new Key("HeadsetHook", "The Headset Hook key. This is typically actually a button on the headset which is used to hang up calls and play or pause media.", KeyType.Phone)
  val LastNumberRedial: Key = new Key("LastNumberRedial", "The Redial button, which redials the last-called number.", KeyType.Phone)
  val Notification: Key = new Key("Notification", "The Notification key.", KeyType.Phone)
  val MannerMode: Key = new Key("MannerMode", "A button which cycles among the notification modes: silent, vibrate, ring, and so forth.", KeyType.Phone)
  val VoiceDial: Key = new Key("VoiceDial", "The Voice Dial key. Initiates voice dialing.", KeyType.Phone)
         
  // Multimedia
  val ChannelDown: Key = new Key("ChannelDown", "Switches to the previous channel.", KeyType.Multimedia)
  val ChannelUp: Key = new Key("ChannelUp", "Switches to the next channel.", KeyType.Multimedia)
  val MediaFastForward: Key = new Key("MediaFastForward", "Starts, continues, or increases the speed of fast forwarding the media.", KeyType.Multimedia)
  val MediaPause: Key = new Key("MediaPause", "Pauses the currently playing media. Some older applications use simply \"Pause\" but this is not correct.", KeyType.Multimedia)
  val MediaPlay: Key = new Key("MediaPlay", "Starts or continues playing media at normal speed, if not already doing so. Has no effect otherwise.", KeyType.Multimedia)
  val MediaPlayPause: Key = new Key("MediaPlayPause", "Toggles between playing and pausing the current media.", KeyType.Multimedia)
  val MediaRecord: Key = new Key("MediaRecord", "Starts or resumes recording media.", KeyType.Multimedia)
  val MediaRewind: Key = new Key("MediaRewind", "Starts, continues, or increases the speed of rewinding the media.", KeyType.Multimedia)
  val MediaStop: Key = new Key("MediaStop", "Stops the current media activity (such as playing, recording, pausing, forwarding, or rewinding). Has no effect if the media is currently stopped already.", KeyType.Multimedia)
  val MediaTrackNext: Key = new Key("MediaTrackNext", "Seeks to the next media or program track.", KeyType.Multimedia)
  val MediaTrackPrevious: Key = new Key("MediaTrackPrevious", "Seeks to the previous media or program track.", KeyType.Multimedia)
         
  // Audio
  val AudioBalanceLeft: Key = new Key("AudioBalanceLeft", "Adjusts audio balance toward the left.", KeyType.Audio)
  val AudioBalanceRight: Key = new Key("AudioBalanceRight", "Adjusts audio balance twoard the right.", KeyType.Audio)
  val AudioBassDown: Key = new Key("AudioBassDown", "Decreases the amount of bass.", KeyType.Audio)
  val AudioBassBoostDown: Key = new Key("AudioBassBoostDown", "Reduces bass boosting or cycles downward through bass boost modes or states.", KeyType.Audio)
  val AudioBassBoostToggle: Key = new Key("AudioBassBoostToggle", "Toggles bass boosting on and off.", KeyType.Audio)
  val AudioBassBoostUp: Key = new Key("AudioBassBoostUp", "Increases the amoung of bass boosting, or cycles upward through a set of bass boost modes or states.", KeyType.Audio)
  val AudioBassUp: Key = new Key("AudioBassUp", "Increases the amount of bass.", KeyType.Audio)
  val AudioFaderFront: Key = new Key("AudioFaderFront", "Adjusts the audio fader toward the front.", KeyType.Audio)
  val AudioFaderRear: Key = new Key("AudioFaderRear", "Adjustts the audio fader toward the rear.", KeyType.Audio)
  val AudioSurroundModeNext: Key = new Key("AudioSurroundModeNext", "Selects the next available surround sound mode.", KeyType.Audio)
  val AudioTrebleDown: Key = new Key("AudioTrebleDown", "Decreases the amount of treble.", KeyType.Audio)
  val AudioTrebleUp: Key = new Key("AudioTrebleUp", "Increases the amount of treble.", KeyType.Audio)
  val AudioVolumeDown: Key = new Key("AudioVolumeDown", "Decreases the audio volume.", KeyType.Audio)
  val AudioVolumeMute: Key = new Key("AudioVolumeMute", "Mutes the audio.", KeyType.Audio)
  val AudioVolumeUp: Key = new Key("AudioVolumeUp", "Increases the audio volume.", KeyType.Audio)
  val MicrophoneToggle: Key = new Key("MicrophoneToggle", "Toggles the microphone on and off.", KeyType.Audio)
  val MicrophoneVolumeDown: Key = new Key("MicrophoneVolumeDown", "Decreases the microphone's input volume.", KeyType.Audio)
  val MicrophoneVolumeMute: Key = new Key("MicrophoneVolumeMute", "Mutes the microphone input.", KeyType.Audio)
  val MicrophoneVolumeUp: Key = new Key("MicrophoneVolumeUp", "Increases the microphone's input volume.", KeyType.Audio)
         
  // TV
  val TV: Key = new Key("TV", "Switches into TV viewing mode.", KeyType.TV)
  val TV3DMode: Key = new Key("TV3DMode", "Toggles 3D TV mode on and off.", KeyType.TV)
  val TVAntennaCable: Key = new Key("TVAntennaCable", "Toggles between antenna and cable inputs.", KeyType.TV)
  val TVAudioDescription: Key = new Key("TVAudioDescription", "Toggles audio description mode on and off.", KeyType.TV)
  val TVAudioDescriptionMixDown: Key = new Key("TVAudioDescriptionMixDown", "Decreases trhe audio description's mixing volume; reduces the volume of the audio descriptions relative to the program sound.", KeyType.TV)
  val TVAudioDescriptionMixUp: Key = new Key("TVAudioDescriptionMixUp", "Increases the audio description's mixing volume; increases the volume of the audio descriptions relative to the program sound.", KeyType.TV)
  val TVContentsMenu: Key = new Key("TVContentsMenu", "Displays or hides the media contents available for playback (this may be a channel guide showing the currently airing programs, or a list of media files to play).", KeyType.TV)
  val TVDataService: Key = new Key("TVDataService", "Displays or hides the TV's data service menu.", KeyType.TV)
  val TVInput: Key = new Key("TVInput", "Cycles the input mode on an external TV.", KeyType.TV)
  val TVInputComponent1: Key = new Key("TVInputComponent1", "Switches to the input \"Component 1.\"", KeyType.TV)
  val TVInputComponent2: Key = new Key("TVInputComponent2", "Switches to the input \"Component 2.\"", KeyType.TV)
  val TVInputComposite1: Key = new Key("TVInputComposite1", "Switches to the input \"Composite 1.\"", KeyType.TV)
  val TVInputComposite2: Key = new Key("TVInputComposite2", "Switches to the input \"Composite 2.\"", KeyType.TV)
  val TVInputHDMI1: Key = new Key("TVInputHDMI1", "Switches to the input \"HDMI 1.\"", KeyType.TV)
  val TVInputHDMI2: Key = new Key("TVInputHDMI2", "Switches to the input \"HDMI 2.\"", KeyType.TV)
  val TVInputHDMI3: Key = new Key("TVInputHDMI3", "Switches to the input \"HDMI 3.\"", KeyType.TV)
  val TVInputHDMI4: Key = new Key("TVInputHDMI4", "Switches to the input \"HDMI 4.\"", KeyType.TV)
  val TVInputVGA1: Key = new Key("TVInputVGA1", "Switches to the input \"VGA 1.\"", KeyType.TV)
  val TVMediaContext: Key = new Key("TVMediaContext", "The Media Context menu key.", KeyType.TV)
  val TVNetwork: Key = new Key("TVNetwork", "Toggle the TV's network connection on and off.", KeyType.TV)
  val TVNumberEntry: Key = new Key("TVNumberEntry", "Put the TV into number entry mode.", KeyType.TV)
  val TVPower: Key = new Key("TVPower", "The device's power button.", KeyType.TV)
  val TVRadioService: Key = new Key("TVRadioService", "Radio button.", KeyType.TV)
  val TVSatellite: Key = new Key("TVSatellite", "Satellite button.", KeyType.TV)
  val TVSatelliteBS: Key = new Key("TVSatelliteBS", "Broadcast Satellite button.", KeyType.TV)
  val TVSatelliteCS: Key = new Key("TVSatelliteCS", "Communication Satellite button.", KeyType.TV)
  val TVSatelliteToggle: Key = new Key("TVSatelliteToggle", "Toggles among available satellites.", KeyType.TV)
  val TVTerrestrialAnalog: Key = new Key("TVTerrestrialAnalog", "Selects analog terrestrial television service (analog cable or antenna reception).", KeyType.TV)
  val TVTerrestrialDigital: Key = new Key("TVTerrestrialDigital", "Selects digital terrestrial television service (digital cable or antenna receiption).", KeyType.TV)
  val TVTimer: Key = new Key("TVTimer", "Timer programming button.", KeyType.TV)
         
  // Media
  val AVRInput: Key = new Key("AVRInput", "Changes the input mode on an external audio/video receiver (AVR) unit.", KeyType.Media)
  val AVRPower: Key = new Key("AVRPower", "Toggles the power on an external AVR unit.", KeyType.Media)
  val ColorF0Red: Key = new Key("ColorF0Red", "General-purpose media function key, color-coded red; this has index 0 among the colored keys.", KeyType.Media)
  val ColorF1Green: Key = new Key("ColorF1Green", "General-purpose media funciton key, color-coded green; this has index 1 among the colored keys.", KeyType.Media)
  val ColorF2Yellow: Key = new Key("ColorF2Yellow", "General-purpose media funciton key, color-coded yellow; this has index 2 among the colored keys.", KeyType.Media)
  val ColorF3Blue: Key = new Key("ColorF3Blue", "General-purpose media funciton key, color-coded blue; this has index 3 among the colored keys.", KeyType.Media)
  val ColorF4Grey: Key = new Key("ColorF4Grey", "General-purpose media funciton key, color-coded grey; this has index 4 among the colored keys.", KeyType.Media)
  val ColorF5Brown: Key = new Key("ColorF5Brown", "General-purpose media funciton key, color-coded brown; this has index 5 among the colored keys.", KeyType.Media)
  val ClosedCaptionToggle: Key = new Key("ClosedCaptionToggle", "Toggles closed captioning on and off.", KeyType.Media)
  val Dimmer: Key = new Key("Dimmer", "Adjusts the brightness of the device by toggling between two brightness levels or by cycling among multiple brightness levels.", KeyType.Media)
  val DisplaySwap: Key = new Key("DisplaySwap", "Cycles among video sources.", KeyType.Media)
  val DVR: Key = new Key("DVR", "Switches the input source to the Digital Video Recorder (DVR).", KeyType.Media)
  val Exit: Key = new Key("Exit", "The Exit button, which exits the curreent application or menu.", KeyType.Media)
  val FavoriteClear0: Key = new Key("FavoriteClear0", "Clears the program or content stored in the first favorites list slot.", KeyType.Media)
  val FavoriteClear1: Key = new Key("FavoriteClear1", "Clears the program or content stored in the second favorites list slot.", KeyType.Media)
  val FavoriteClear2: Key = new Key("FavoriteClear2", "Clears the program or content stored in the third favorites list slot.", KeyType.Media)
  val FavoriteClear3: Key = new Key("FavoriteClear3", "Clears the program or content stored in the fourth favorites list slot.", KeyType.Media)
  val FavoriteRecall0: Key = new Key("FavoriteRecall0", "Selects (recalls) the program or content stored in the first favorites list slot.", KeyType.Media)
  val FavoriteRecall1: Key = new Key("FavoriteRecall1", "Selects (recalls) the program or content stored in the second favorites list slot.", KeyType.Media)
  val FavoriteRecall2: Key = new Key("FavoriteRecall2", "Selects (recalls) the program or content stored in the third favorites list slot.", KeyType.Media)
  val FavoriteRecall3: Key = new Key("FavoriteRecall3", "Selects (recalls) the program or content stored in the fourth favorites list slot.", KeyType.Media)
  val FavoriteStore0: Key = new Key("FavoriteStore0", "Stores the current program or content into the first favorites list slot.", KeyType.Media)
  val FavoriteStore1: Key = new Key("FavoriteStore1", "Stores the current program or content into the second favorites list slot.", KeyType.Media)
  val FavoriteStore2: Key = new Key("FavoriteStore2", "Stores the current program or content into the third favorites list slot.", KeyType.Media)
  val FavoriteStore3: Key = new Key("FavoriteStore3", "Stores the current program or content into the fourth favorites list slot.", KeyType.Media)
  val Guide: Key = new Key("Guide", "Toggles the display of the program or content guide.", KeyType.Media)
  val GuideNextDay: Key = new Key("GuideNextDay", "If the guide is currently displayed, this button tells the guide to display the next day's content.", KeyType.Media)
  val GuidePreviousDay: Key = new Key("GuidePreviousDay", "If the guide is currently displayed, this button tells the guide to display the previous day's content.", KeyType.Media)
  val Info: Key = new Key("Info", "Toggles the display of information about the currently selected content, program, or media.", KeyType.Media)
  val InstantReplay: Key = new Key("InstantReplay", "Tellls the device to perform an instant replay (typically some form of jumping back a short amount of time then playing it again, possibly but not usually in slow motion).", KeyType.Media)
  val Link: Key = new Key("Link", "Opens content liniked to the current program, if available and possible.", KeyType.Media)
  val ListProgram: Key = new Key("ListProgram", "Lists the current program.", KeyType.Media)
  val LiveContent: Key = new Key("LiveContent", "Toggles a display listing currently available live content or programs.", KeyType.Media)
  val Lock: Key = new Key("Lock", "Locks or unlocks the currently selected content or pgoram.", KeyType.Media)
  val MediaApps: Key = new Key("MediaApps", "Presents a list of media applications, such as photo viewers, audio and video players, and games. [1]", KeyType.Media)
  val MediaAudioTrack: Key = new Key("MediaAudioTrack", "The Audio Track key.", KeyType.Media)
  val MediaLast: Key = new Key("MediaLast", "Jumps back to the last-viewed content, program, or other media.", KeyType.Media)
  val MediaSkipBackward: Key = new Key("MediaSkipBackward", "Skips backward to the previous content or program.", KeyType.Media)
  val MediaSkipForward: Key = new Key("MediaSkipForward", "Skips forward to the next content or program.", KeyType.Media)
  val MediaStepBackward: Key = new Key("MediaStepBackward", "Steps backward to the previous content or program.", KeyType.Media)
  val MediaStepForward: Key = new Key("MediaStepForward", "Steps forward to the next content or program.", KeyType.Media)
  val MediaTopMenu: Key = new Key("MediaTopMenu", "Top Menu button; opens the media's main menu, such as on a DVD or Blu-Ray disc.", KeyType.Media)
  val NavigateIn: Key = new Key("NavigateIn", "Navigates into a submenu or option.", KeyType.Media)
  val NavigateNext: Key = new Key("NavigateNext", "Navigates to the next item.", KeyType.Media)
  val NavigateOut: Key = new Key("NavigateOut", "Navigates out of the current screen or menu.", KeyType.Media)
  val NavigatePrevious: Key = new Key("NavigatePrevious", "Navigates to the previous item.", KeyType.Media)
  val NextFavoriteChannel: Key = new Key("NextFavoriteChannel", "Cycles to the next channel in the favorites list.", KeyType.Media)
  val NextUserProfile: Key = new Key("NextUserProfile", "Cycles to the next saved user profile, if this feature is supported and multiple profiles exist.", KeyType.Media)
  val OnDemand: Key = new Key("OnDemand", "Opens the user interface for selecting on demand content or programs to watch.", KeyType.Media)
  val Pairing: Key = new Key("Pairing", "Starts the process of pairing the remote with a device to be controlled.", KeyType.Media)
  val PinPDown: Key = new Key("PinPDown", "A button to move the picture-in-picture view downward.", KeyType.Media)
  val PinPMove: Key = new Key("PinPMove", "A button to control moving the picture-in-picture view.", KeyType.Media)
  val PinPToggle: Key = new Key("PinPToggle", "Toggles display of th epicture-in-picture view on and off.", KeyType.Media)
  val PinPUp: Key = new Key("PinPUp", "A button to move the picture-in-picture view upward.", KeyType.Media)
  val PlaySpeedDown: Key = new Key("PlaySpeedDown", "Decreases the media playback rate.", KeyType.Media)
  val PlaySpeedReset: Key = new Key("PlaySpeedReset", "Returns the media playback rate to normal.", KeyType.Media)
  val PlaySpeedUp: Key = new Key("PlaySpeedUp", "Increases the media playback rate.", KeyType.Media)
  val RandomToggle: Key = new Key("RandomToggle", "Toggles random media (also known as \"shuffle mode\") on and off.", KeyType.Media)
  val RcLowBattery: Key = new Key("RcLowBattery", "A code sent when the remote control's battery is low. This doesn't actually correspond to a physical key at all.", KeyType.Media)
  val RecordSpeedNext: Key = new Key("RecordSpeedNext", "Cycles among the available media recording speeds.", KeyType.Media)
  val RfBypass: Key = new Key("RfBypass", "Toggles radio frequency (RF) input bypass mode on and off. RF bypass mode passes RF input directly to the RF output without any processing or filtering.", KeyType.Media)
  val ScanChannelsToggle: Key = new Key("ScanChannelsToggle", "Toggles the channel scan mode on and off; this is a mode which flips through channels automatically until the user stops the scan.", KeyType.Media)
  val ScreenModeNext: Key = new Key("ScreenModeNext", "Cycles through the available screen display modes.", KeyType.Media)
  val Settings: Key = new Key("Settings", "Toggles display of the device's settings screen on and off.", KeyType.Media)
  val SplitScreenToggle: Key = new Key("SplitScreenToggle", "Toggles split screen display mode on and off.", KeyType.Media)
  val STBInput: Key = new Key("STBInput", "Cycles among input modes on an external set-top box (STB).", KeyType.Media)
  val STBPower: Key = new Key("STBPower", "Toggles on and off an external STB.", KeyType.Media)
  val Subtitle: Key = new Key("Subtitle", "Toggles the display of subtitles on and off if they're available.", KeyType.Media)
  val Teletext: Key = new Key("Teletext", "Toggles display of teletext, if available.", KeyType.Media)
  val VideoModeNext: Key = new Key("VideoModeNext", "Cycles through the available video modes.", KeyType.Media)
  val Wink: Key = new Key("Wink", "Causes the device to identify itself in some fashion, such as by flashing a light, briefly changing the brightness of indicator lights, or emitting a tone.", KeyType.Media)
  val ZoomToggle: Key = new Key("ZoomToggle", "Toggles between full-screen and scaled content display, or otherwise change the magnification level.", KeyType.Media)
         
  // Speech
  val SpeechCorrectionList: Key = new Key("SpeechCorrectionList", "Presents a list of possible corrections for a word which was incorrectly identified.", KeyType.Speech)
  val SpeechInputToggle: Key = new Key("SpeechInputToggle", "Toggles between dictation mode and command/control mode. This lets the speech engine know whether to interpret spoken words as input text or as commands.", KeyType.Speech)
         
  // Document
  val Close: Key = new Key("Close", "Closes the current document or message. Must not exit the application.", KeyType.Document)
  val New: Key = new Key("New", "Creates a new document or message.", KeyType.Document)
  val Open: Key = new Key("Open", "Opens an existing document or message.", KeyType.Document)
  val Print: Key = new Key("Print", "Prints the current document or message.", KeyType.Document)
  val Save: Key = new Key("Save", "Saves the current document or message.", KeyType.Document)
  val SpellCheck: Key = new Key("SpellCheck", "Starts spell checking the current document.", KeyType.Document)
  val MailForward: Key = new Key("MailForward", "Opens the user interface to forward a message.", KeyType.Document)
  val MailReply: Key = new Key("MailReply", "Opens the user interface to reply to a message.", KeyType.Document)
  val MailSend: Key = new Key("MailSend", "Sends the current message.", KeyType.Document)
         
  // Application
  val LaunchCalculator: Key = new Key("LaunchCalculator", "The Calculator key, often labeled with an icon such as . This is often used as a generic application launcher key (APPCOMMAND_LAUNCH_APP2).", KeyType.Application)
  val LaunchCalendar: Key = new Key("LaunchCalendar", "The Calendar key, often labeled with an icon like .", KeyType.Application)
  val LaunchContacts: Key = new Key("LaunchContacts", "The Contacts key.", KeyType.Application)
  val LaunchMail: Key = new Key("LaunchMail", "The Mail key. This is often displayed as .", KeyType.Application)
  val LaunchMediaPlayer: Key = new Key("LaunchMediaPlayer", "The Media Player key.", KeyType.Application)
  val LaunchMusicPlayer: Key = new Key("LaunchMusicPlayer", "The Music Player key, often labeled with an icon such as .", KeyType.Application)
  val LaunchMyComputer: Key = new Key("LaunchMyComputer", "The My Computer key on Windows keyboards. This is often used as a generic application launcher key (APPCOMMAND_LAUNCH_APP1).", KeyType.Application)
  val LaunchPhone: Key = new Key("LaunchPhone", "The Phone key, to open the phone dialer application if one is present.", KeyType.Application)
  val LaunchScreenSaver: Key = new Key("LaunchScreenSaver", "The Screen Saver key.", KeyType.Application)
  val LaunchSpreadsheet: Key = new Key("LaunchSpreadsheet", "The Spreadsheet key. This key may be labeled with an icon such as or that of a specific spreadsheet application.", KeyType.Application)
  val LaunchWebBrowser: Key = new Key("LaunchWebBrowser", "The Web Browser key. This key is frequently labeled with an icon such as or the icon of a specific browser, depending on the device manufacturer.", KeyType.Application)
  val LaunchWebCam: Key = new Key("LaunchWebCam", "The WebCam key. Opens the webcam application.", KeyType.Application)
  val LaunchWordProcessor: Key = new Key("LaunchWordProcessor", "The Word Processor key. This may be an icon of a specific word processor application, or a generic document icon.", KeyType.Application)
  val LaunchApplication1: Key = new Key("LaunchApplication1", "The first generic application launcher button.", KeyType.Application)
  val LaunchApplication2: Key = new Key("LaunchApplication2", "The second generic application launcher button.", KeyType.Application)
  val LaunchApplication3: Key = new Key("LaunchApplication3", "The third generic application launcher button.", KeyType.Application)
  val LaunchApplication4: Key = new Key("LaunchApplication4", "The fourth generic application launcher button.", KeyType.Application)
  val LaunchApplication5: Key = new Key("LaunchApplication5", "The fifth generic application launcher button.", KeyType.Application)
  val LaunchApplication6: Key = new Key("LaunchApplication6", "The sixth generic application launcher button.", KeyType.Application)
  val LaunchApplication7: Key = new Key("LaunchApplication7", "The seventh generic application launcher button.", KeyType.Application)
  val LaunchApplication8: Key = new Key("LaunchApplication8", "The eighth generic application launcher button.", KeyType.Application)
  val LaunchApplication9: Key = new Key("LaunchApplication9", "The ninth generic application launcher button.", KeyType.Application)
  val LaunchApplication10: Key = new Key("LaunchApplication10", "The 10th generic application launcher button.", KeyType.Application)
  val LaunchApplication11: Key = new Key("LaunchApplication11", "The 11th generic application launcher button.", KeyType.Application)
  val LaunchApplication12: Key = new Key("LaunchApplication12", "The 12th generic application launcher button.", KeyType.Application)
  val LaunchApplication13: Key = new Key("LaunchApplication13", "The 13th generic application launcher button.", KeyType.Application)
  val LaunchApplication14: Key = new Key("LaunchApplication14", "The 14th generic application launcher button.", KeyType.Application)
  val LaunchApplication15: Key = new Key("LaunchApplication15", "The 15th generic application launcher button.", KeyType.Application)
  val LaunchApplication16: Key = new Key("LaunchApplication16", "The 16th generic application launcher button.", KeyType.Application)
         
  // Browser
  val BrowserBack: Key = new Key("BrowserBack", "Navigates to the previous content or page in the current Web view's history.", KeyType.Browser)
  val BrowserFavorites: Key = new Key("BrowserFavorites", "Opens the user's list of bookmarks/favorites.", KeyType.Browser)
  val BrowserForward: Key = new Key("BrowserForward", "Navigates to the next content or page in the current Web view's history.", KeyType.Browser)
  val BrowserHome: Key = new Key("BrowserHome", "Navigates to the user's preferred home page.", KeyType.Browser)
  val BrowserRefresh: Key = new Key("BrowserRefresh", "Refreshes the current page or contentl.", KeyType.Browser)
  val BrowserSearch: Key = new Key("BrowserSearch", "Activates the user's preferred search engine or the search interface within their browser.", KeyType.Browser)
  val BrowserStop: Key = new Key("BrowserStop", "Stops loading the currently displayed Web view or content.", KeyType.Browser)
         

  private def register(key: Key): Unit = {
    map += key.value -> key
  }

  def get(value: String): Option[Key] = map.get(value)

  def apply(value: String): Key = get(value).getOrElse(throw new RuntimeException(s"Unable to find Key for '$value'."))
}

case class KeyType private(name: String)

object KeyType {
  val Lowercase = KeyType("Lowercase")
  val Uppercase = KeyType("Uppercase")
  val Numeric = KeyType("Numeric")
  val Symbols = KeyType("Symbols")
  val Special = KeyType("Special")
  val Modifier = KeyType("Modifier")
  val Whitespace = KeyType("Whitespace")
  val Navigation = KeyType("Navigation")
  val Editing = KeyType("Editing")
  val UI = KeyType("UI")
  val Device = KeyType("Device")
  val IME = KeyType("IME")
  val Function = KeyType("Function")
  val Phone = KeyType("Phone")
  val Multimedia = KeyType("Multimedia")
  val Audio = KeyType("Audio")
  val TV = KeyType("TV")
  val Media = KeyType("Media")
  val Speech = KeyType("Speech")
  val Document = KeyType("Document")
  val Application = KeyType("Application")
  val Browser = KeyType("Browser")
}