/*
 * Decompiled with CFR 0.152.
 */
package org.twitter.zipkin.storage.cassandra;

import com.datastax.driver.core.BoundStatement;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.DataType;
import com.datastax.driver.core.KeyspaceMetadata;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.ProtocolVersion;
import com.datastax.driver.core.RegularStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.querybuilder.BindMarker;
import com.datastax.driver.core.querybuilder.Ordering;
import com.datastax.driver.core.querybuilder.QueryBuilder;
import com.datastax.driver.core.utils.Bytes;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.io.CharStreams;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.IntFunction;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.StreamSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Repository
implements AutoCloseable {
    public static final String KEYSPACE = "zipkin";
    public static final short BUCKETS = 10;
    private static final Logger LOG = LoggerFactory.getLogger(Repository.class);
    private static final Random RAND = new Random();
    private static final List<Integer> ALL_BUCKETS = Collections.unmodifiableList(new ArrayList<Integer>(){
        {
            for (int i = 0; i < 10; ++i) {
                this.add(i);
            }
        }
    });
    private static final long WRITTEN_NAMES_TTL = Long.getLong("zipkin.store.cassandra.internal.writtenNamesTtl", 3600000L);
    private static final long DURATION_INDEX_BUCKET_WINDOW_SECONDS = Long.getLong("zipkin.store.cassandra.internal.durationIndexBucket", 3600L);
    private final Session session;
    private final PreparedStatement selectTraces;
    private final PreparedStatement insertSpan;
    private final PreparedStatement selectDependencies;
    private final PreparedStatement insertDependencies;
    private final PreparedStatement selectServiceNames;
    private final PreparedStatement insertServiceName;
    private final PreparedStatement selectSpanNames;
    private final PreparedStatement insertSpanName;
    private final PreparedStatement selectTraceIdsByServiceName;
    private final PreparedStatement insertTraceIdByServiceName;
    private final PreparedStatement selectTraceIdsBySpanName;
    private final PreparedStatement insertTraceIdBySpanName;
    private final PreparedStatement selectTraceIdsByAnnotations;
    private final PreparedStatement insertTraceIdByAnnotation;
    private final PreparedStatement selectTraceIdsBySpanDuration;
    private final PreparedStatement insertTraceIdBySpanDuration;
    private final Map<String, String> metadata;
    private final ProtocolVersion protocolVersion;
    private final ThreadLocal<Set<String>> writtenNames = new ThreadLocal<Set<String>>(){
        private long cacheInterval = this.toCacheInterval(System.currentTimeMillis());

        @Override
        protected Set<String> initialValue() {
            return new HashSet<String>();
        }

        @Override
        public Set<String> get() {
            long newCacheInterval = this.toCacheInterval(System.currentTimeMillis());
            if (this.cacheInterval != newCacheInterval) {
                this.cacheInterval = newCacheInterval;
                this.set(new HashSet());
            }
            return (Set)super.get();
        }

        private long toCacheInterval(long ms) {
            return ms / WRITTEN_NAMES_TTL;
        }
    };
    private Function<ResultSet, Void> resultSetToVoidFunction = input -> null;

    public Repository(String keyspace, Cluster cluster, Boolean ensureSchema) {
        if (ensureSchema.booleanValue()) {
            Schema.ensureExists(keyspace, cluster);
        }
        this.metadata = Schema.readMetadata(keyspace, cluster);
        this.session = cluster.connect(keyspace);
        this.protocolVersion = cluster.getConfiguration().getProtocolOptions().getProtocolVersionEnum();
        this.insertSpan = this.session.prepare((RegularStatement)QueryBuilder.insertInto((String)"traces").value("trace_id", (Object)QueryBuilder.bindMarker((String)"trace_id")).value("ts", (Object)QueryBuilder.bindMarker((String)"ts")).value("span_name", (Object)QueryBuilder.bindMarker((String)"span_name")).value("span", (Object)QueryBuilder.bindMarker((String)"span")).using(QueryBuilder.ttl((BindMarker)QueryBuilder.bindMarker((String)"ttl_"))));
        this.selectTraces = this.session.prepare((RegularStatement)QueryBuilder.select((String[])new String[]{"trace_id", "span"}).from("traces").where(QueryBuilder.in((String)"trace_id", (Object[])new Object[]{QueryBuilder.bindMarker((String)"trace_id")})).limit(QueryBuilder.bindMarker((String)"limit_")));
        this.selectDependencies = this.session.prepare((RegularStatement)QueryBuilder.select((String[])new String[]{"dependencies"}).from("dependencies").where(QueryBuilder.in((String)"day", (Object[])new Object[]{QueryBuilder.bindMarker((String)"days")})));
        this.insertDependencies = this.session.prepare((RegularStatement)QueryBuilder.insertInto((String)"dependencies").value("day", (Object)QueryBuilder.bindMarker((String)"day")).value("dependencies", (Object)QueryBuilder.bindMarker((String)"dependencies")));
        this.selectServiceNames = this.session.prepare((RegularStatement)QueryBuilder.select((String[])new String[]{"service_name"}).from("service_names"));
        this.insertServiceName = this.session.prepare((RegularStatement)QueryBuilder.insertInto((String)"service_names").value("service_name", (Object)QueryBuilder.bindMarker((String)"service_name")).using(QueryBuilder.ttl((BindMarker)QueryBuilder.bindMarker((String)"ttl_"))));
        this.selectSpanNames = this.session.prepare((RegularStatement)QueryBuilder.select((String[])new String[]{"span_name"}).from("span_names").where(QueryBuilder.eq((String)"service_name", (Object)QueryBuilder.bindMarker((String)"service_name"))).and(QueryBuilder.eq((String)"bucket", (Object)QueryBuilder.bindMarker((String)"bucket"))).limit(QueryBuilder.bindMarker((String)"limit_")));
        this.insertSpanName = this.session.prepare((RegularStatement)QueryBuilder.insertInto((String)"span_names").value("service_name", (Object)QueryBuilder.bindMarker((String)"service_name")).value("bucket", (Object)QueryBuilder.bindMarker((String)"bucket")).value("span_name", (Object)QueryBuilder.bindMarker((String)"span_name")).using(QueryBuilder.ttl((BindMarker)QueryBuilder.bindMarker((String)"ttl_"))));
        this.selectTraceIdsByServiceName = this.session.prepare((RegularStatement)QueryBuilder.select((String[])new String[]{"ts", "trace_id"}).from("service_name_index").where(QueryBuilder.eq((String)"service_name", (Object)QueryBuilder.bindMarker((String)"service_name"))).and(QueryBuilder.in((String)"bucket", (Object[])new Object[]{QueryBuilder.bindMarker((String)"bucket")})).and(QueryBuilder.gte((String)"ts", (Object)QueryBuilder.bindMarker((String)"start_ts"))).and(QueryBuilder.lte((String)"ts", (Object)QueryBuilder.bindMarker((String)"end_ts"))).limit(QueryBuilder.bindMarker((String)"limit_")).orderBy(new Ordering[]{QueryBuilder.desc((String)"ts")}));
        this.insertTraceIdByServiceName = this.session.prepare((RegularStatement)QueryBuilder.insertInto((String)"service_name_index").value("service_name", (Object)QueryBuilder.bindMarker((String)"service_name")).value("bucket", (Object)QueryBuilder.bindMarker((String)"bucket")).value("ts", (Object)QueryBuilder.bindMarker((String)"ts")).value("trace_id", (Object)QueryBuilder.bindMarker((String)"trace_id")).using(QueryBuilder.ttl((BindMarker)QueryBuilder.bindMarker((String)"ttl_"))));
        this.selectTraceIdsBySpanName = this.session.prepare((RegularStatement)QueryBuilder.select((String[])new String[]{"ts", "trace_id"}).from("service_span_name_index").where(QueryBuilder.eq((String)"service_span_name", (Object)QueryBuilder.bindMarker((String)"service_span_name"))).and(QueryBuilder.gte((String)"ts", (Object)QueryBuilder.bindMarker((String)"start_ts"))).and(QueryBuilder.lte((String)"ts", (Object)QueryBuilder.bindMarker((String)"end_ts"))).limit(QueryBuilder.bindMarker((String)"limit_")).orderBy(new Ordering[]{QueryBuilder.desc((String)"ts")}));
        this.insertTraceIdBySpanName = this.session.prepare((RegularStatement)QueryBuilder.insertInto((String)"service_span_name_index").value("service_span_name", (Object)QueryBuilder.bindMarker((String)"service_span_name")).value("ts", (Object)QueryBuilder.bindMarker((String)"ts")).value("trace_id", (Object)QueryBuilder.bindMarker((String)"trace_id")).using(QueryBuilder.ttl((BindMarker)QueryBuilder.bindMarker((String)"ttl_"))));
        this.selectTraceIdsByAnnotations = this.session.prepare((RegularStatement)QueryBuilder.select((String[])new String[]{"ts", "trace_id"}).from("annotations_index").where(QueryBuilder.eq((String)"annotation", (Object)QueryBuilder.bindMarker((String)"annotation"))).and(QueryBuilder.in((String)"bucket", (Object[])new Object[]{QueryBuilder.bindMarker((String)"bucket")})).and(QueryBuilder.gte((String)"ts", (Object)QueryBuilder.bindMarker((String)"start_ts"))).and(QueryBuilder.lte((String)"ts", (Object)QueryBuilder.bindMarker((String)"end_ts"))).limit(QueryBuilder.bindMarker((String)"limit_")).orderBy(new Ordering[]{QueryBuilder.desc((String)"ts")}));
        this.insertTraceIdByAnnotation = this.session.prepare((RegularStatement)QueryBuilder.insertInto((String)"annotations_index").value("annotation", (Object)QueryBuilder.bindMarker((String)"annotation")).value("bucket", (Object)QueryBuilder.bindMarker((String)"bucket")).value("ts", (Object)QueryBuilder.bindMarker((String)"ts")).value("trace_id", (Object)QueryBuilder.bindMarker((String)"trace_id")).using(QueryBuilder.ttl((BindMarker)QueryBuilder.bindMarker((String)"ttl_"))));
        this.selectTraceIdsBySpanDuration = this.session.prepare((RegularStatement)QueryBuilder.select((String[])new String[]{"duration", "ts", "trace_id"}).from("span_duration_index").where(QueryBuilder.eq((String)"service_name", (Object)QueryBuilder.bindMarker((String)"service_name"))).and(QueryBuilder.eq((String)"span_name", (Object)QueryBuilder.bindMarker((String)"span_name"))).and(QueryBuilder.eq((String)"bucket", (Object)QueryBuilder.bindMarker((String)"time_bucket"))).and(QueryBuilder.lte((String)"duration", (Object)QueryBuilder.bindMarker((String)"max_duration"))).and(QueryBuilder.gte((String)"duration", (Object)QueryBuilder.bindMarker((String)"min_duration"))).orderBy(new Ordering[]{QueryBuilder.desc((String)"duration")}));
        this.insertTraceIdBySpanDuration = this.session.prepare((RegularStatement)QueryBuilder.insertInto((String)"span_duration_index").value("service_name", (Object)QueryBuilder.bindMarker((String)"service_name")).value("span_name", (Object)QueryBuilder.bindMarker((String)"span_name")).value("bucket", (Object)QueryBuilder.bindMarker((String)"bucket")).value("duration", (Object)QueryBuilder.bindMarker((String)"duration")).value("ts", (Object)QueryBuilder.bindMarker((String)"ts")).value("trace_id", (Object)QueryBuilder.bindMarker((String)"trace_id")).using(QueryBuilder.ttl((BindMarker)QueryBuilder.bindMarker((String)"ttl_"))));
    }

    public ListenableFuture<Void> storeSpan(long traceId, long timestamp, String spanName, ByteBuffer span, int ttl) {
        Preconditions.checkNotNull((Object)spanName);
        Preconditions.checkArgument((!spanName.isEmpty() ? 1 : 0) != 0);
        try {
            if (0L == timestamp && this.metadata.get("traces.compaction.class").contains("DateTieredCompactionStrategy")) {
                LOG.warn("span with no first or last timestamp. if this happens a lot consider switching back to SizeTieredCompactionStrategy for zipkin.traces");
            }
            BoundStatement bound = this.insertSpan.bind().setLong("trace_id", traceId).setBytesUnsafe("ts", this.serializeTs(timestamp)).setString("span_name", spanName).setBytes("span", span).setInt("ttl_", ttl);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugInsertSpan(traceId, timestamp, spanName, span, ttl));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), this.resultSetToVoidFunction);
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugInsertSpan(traceId, timestamp, spanName, span, ttl), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugInsertSpan(long traceId, long timestamp, String spanName, ByteBuffer span, int ttl) {
        return this.insertSpan.getQueryString().replace(":trace_id", String.valueOf(traceId)).replace(":ts", String.valueOf(timestamp)).replace(":span_name", spanName).replace(":span", Bytes.toHexString((ByteBuffer)span)).replace(":ttl_", String.valueOf(ttl));
    }

    public ListenableFuture<Map<Long, List<ByteBuffer>>> getSpansByTraceIds(Long[] traceIds, int limit) {
        Preconditions.checkNotNull((Object)traceIds);
        try {
            if (0 < traceIds.length) {
                BoundStatement bound = this.selectTraces.bind().setList("trace_id", Arrays.asList(traceIds)).setInt("limit_", limit);
                bound.setFetchSize(Integer.MAX_VALUE);
                if (LOG.isDebugEnabled()) {
                    LOG.debug(this.debugSelectTraces(traceIds, limit));
                }
                return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), input -> {
                    LinkedHashMap spans = new LinkedHashMap();
                    for (Row row : input) {
                        long traceId = row.getLong("trace_id");
                        if (!spans.containsKey(traceId)) {
                            spans.put(traceId, new ArrayList());
                        }
                        ((List)spans.get(traceId)).add(row.getBytes("span"));
                    }
                    return spans;
                });
            }
            return Futures.immediateFuture(Collections.emptyMap());
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugSelectTraces(traceIds, limit), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugSelectTraces(Long[] traceIds, int limit) {
        return this.selectTraces.getQueryString().replace(":trace_id", Arrays.toString((Object[])traceIds)).replace(":limit_", String.valueOf(limit));
    }

    public ListenableFuture<Void> storeDependencies(long epochDayMillis, ByteBuffer dependencies) {
        Date startFlooredToDay = new Date(epochDayMillis);
        try {
            BoundStatement bound = this.insertDependencies.bind().setDate("day", startFlooredToDay).setBytes("dependencies", dependencies);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugInsertDependencies(startFlooredToDay, dependencies));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), this.resultSetToVoidFunction);
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugInsertDependencies(startFlooredToDay, dependencies), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugInsertDependencies(Date startFlooredToDay, ByteBuffer dependencies) {
        return this.insertDependencies.getQueryString().replace(":day", startFlooredToDay.toString()).replace(":dependencies", Bytes.toHexString((ByteBuffer)dependencies));
    }

    public ListenableFuture<List<ByteBuffer>> getDependencies(long startEpochDayMillis, long endEpochDayMillis) {
        List<Date> days = Repository.getDays(startEpochDayMillis, endEpochDayMillis);
        try {
            BoundStatement bound = this.selectDependencies.bind().setList("days", days);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugSelectDependencies(days));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), input -> {
                ArrayList<ByteBuffer> dependencies = new ArrayList<ByteBuffer>();
                for (Row row : input) {
                    dependencies.add(row.getBytes("dependencies"));
                }
                return dependencies;
            });
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugSelectDependencies(days), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugSelectDependencies(List<Date> days) {
        return this.selectDependencies.getQueryString().replace(":days", Arrays.toString(days.toArray()));
    }

    public ListenableFuture<Set<String>> getServiceNames() {
        try {
            BoundStatement bound = this.selectServiceNames.bind();
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.selectServiceNames.getQueryString());
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), input -> {
                HashSet<String> serviceNames = new HashSet<String>();
                for (Row row : input) {
                    serviceNames.add(row.getString("service_name"));
                }
                return serviceNames;
            });
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.selectServiceNames.getQueryString(), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    public ListenableFuture<Void> storeServiceName(String serviceName, int ttl) {
        Preconditions.checkNotNull((Object)serviceName);
        Preconditions.checkArgument((!serviceName.isEmpty() ? 1 : 0) != 0);
        if (this.writtenNames.get().add(serviceName)) {
            try {
                BoundStatement bound = this.insertServiceName.bind().setString("service_name", serviceName).setInt("ttl_", ttl);
                if (LOG.isDebugEnabled()) {
                    LOG.debug(this.debugInsertServiceName(serviceName, ttl));
                }
                return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), this.resultSetToVoidFunction);
            }
            catch (RuntimeException ex) {
                LOG.error("failed " + this.debugInsertServiceName(serviceName, ttl), (Throwable)ex);
                this.writtenNames.get().remove(serviceName);
                throw ex;
            }
        }
        return Futures.immediateFuture(null);
    }

    private String debugInsertServiceName(String serviceName, int ttl) {
        return this.insertServiceName.getQueryString().replace(":service_name", serviceName).replace(":ttl_", String.valueOf(ttl));
    }

    public ListenableFuture<Set<String>> getSpanNames(String serviceName) {
        Preconditions.checkNotNull((Object)serviceName);
        serviceName = serviceName.toLowerCase();
        try {
            if (!serviceName.isEmpty()) {
                BoundStatement bound = this.selectSpanNames.bind().setString("service_name", serviceName).setInt("bucket", 0).setInt("limit_", 1000);
                if (LOG.isDebugEnabled()) {
                    LOG.debug(this.debugSelectSpanNames(serviceName));
                }
                return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), input -> {
                    HashSet<String> spanNames = new HashSet<String>();
                    for (Row row : input) {
                        spanNames.add(row.getString("span_name"));
                    }
                    return spanNames.size() < 1000 ? spanNames : Collections.singleton("too many span names");
                });
            }
            return Futures.immediateFuture(Collections.emptySet());
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugSelectSpanNames(serviceName), (Throwable)ex);
            throw ex;
        }
    }

    private String debugSelectSpanNames(String serviceName) {
        return this.selectSpanNames.getQueryString().replace(":service_name", serviceName);
    }

    public ListenableFuture<Void> storeSpanName(String serviceName, String spanName, int ttl) {
        Preconditions.checkNotNull((Object)serviceName);
        Preconditions.checkArgument((!serviceName.isEmpty() ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)spanName);
        Preconditions.checkArgument((!spanName.isEmpty() ? 1 : 0) != 0);
        if (this.writtenNames.get().add(serviceName + "\u2013\u2013" + spanName)) {
            try {
                BoundStatement bound = this.insertSpanName.bind().setString("service_name", serviceName).setInt("bucket", 0).setString("span_name", spanName).setInt("ttl_", ttl);
                if (LOG.isDebugEnabled()) {
                    LOG.debug(this.debugInsertSpanName(serviceName, spanName, ttl));
                }
                return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), this.resultSetToVoidFunction);
            }
            catch (RuntimeException ex) {
                LOG.error("failed " + this.debugInsertSpanName(serviceName, spanName, ttl), (Throwable)ex);
                this.writtenNames.get().remove(serviceName + "\u2013\u2013" + spanName);
                return Futures.immediateFailedFuture((Throwable)ex);
            }
        }
        return Futures.immediateFuture(null);
    }

    private String debugInsertSpanName(String serviceName, String spanName, int ttl) {
        return this.insertSpanName.getQueryString().replace(":service_name", serviceName).replace(":span_name", spanName).replace(":ttl_", String.valueOf(ttl));
    }

    public ListenableFuture<Map<Long, Long>> getTraceIdsByServiceName(String serviceName, long endTs, long lookback, int limit) {
        Preconditions.checkNotNull((Object)serviceName);
        Preconditions.checkArgument((!serviceName.isEmpty() ? 1 : 0) != 0);
        long startTs = endTs - lookback;
        try {
            BoundStatement bound = this.selectTraceIdsByServiceName.bind().setString("service_name", serviceName).setList("bucket", ALL_BUCKETS).setBytesUnsafe("start_ts", this.serializeTs(startTs)).setBytesUnsafe("end_ts", this.serializeTs(endTs)).setInt("limit_", limit);
            bound.setFetchSize(Integer.MAX_VALUE);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugSelectTraceIdsByServiceName(serviceName, startTs, endTs, limit));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), input -> {
                LinkedHashMap<Long, Long> traceIdsToTimestamps = new LinkedHashMap<Long, Long>();
                for (Row row : input) {
                    traceIdsToTimestamps.put(row.getLong("trace_id"), this.deserializeTs(row, "ts"));
                }
                return traceIdsToTimestamps;
            });
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugSelectTraceIdsByServiceName(serviceName, startTs, endTs, limit), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugSelectTraceIdsByServiceName(String serviceName, long startTs, long endTs, int limit) {
        return this.selectTraceIdsByServiceName.getQueryString().replace(":service_name", serviceName).replace(":start_ts", new Date(startTs / 1000L).toString()).replace(":end_ts", new Date(endTs / 1000L).toString()).replace(":limit_", String.valueOf(limit));
    }

    public ListenableFuture<Void> storeTraceIdByServiceName(String serviceName, long timestamp, long traceId, int ttl) {
        Preconditions.checkNotNull((Object)serviceName);
        Preconditions.checkArgument((!serviceName.isEmpty() ? 1 : 0) != 0);
        try {
            BoundStatement bound = this.insertTraceIdByServiceName.bind().setString("service_name", serviceName).setInt("bucket", RAND.nextInt(10)).setBytesUnsafe("ts", this.serializeTs(timestamp)).setLong("trace_id", traceId).setInt("ttl_", ttl);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugInsertTraceIdByServiceName(serviceName, timestamp, traceId, ttl));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), this.resultSetToVoidFunction);
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugInsertTraceIdByServiceName(serviceName, timestamp, traceId, ttl), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugInsertTraceIdByServiceName(String serviceName, long timestamp, long traceId, int ttl) {
        return this.insertTraceIdByServiceName.getQueryString().replace(":service_name", serviceName).replace(":ts", new Date(timestamp / 1000L).toString()).replace(":trace_id", new Date(traceId).toString()).replace(":ttl_", String.valueOf(ttl));
    }

    public ListenableFuture<Map<Long, Long>> getTraceIdsBySpanName(String serviceName, String spanName, long endTs, long lookback, int limit) {
        Preconditions.checkNotNull((Object)serviceName);
        Preconditions.checkArgument((!serviceName.isEmpty() ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)spanName);
        Preconditions.checkArgument((!spanName.isEmpty() ? 1 : 0) != 0);
        String serviceSpanName = serviceName + "." + spanName;
        long startTs = endTs - lookback;
        try {
            BoundStatement bound = this.selectTraceIdsBySpanName.bind().setString("service_span_name", serviceSpanName).setBytesUnsafe("start_ts", this.serializeTs(startTs)).setBytesUnsafe("end_ts", this.serializeTs(endTs)).setInt("limit_", limit);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugSelectTraceIdsBySpanName(serviceSpanName, startTs, endTs, limit));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), input -> {
                LinkedHashMap<Long, Long> traceIdsToTimestamps = new LinkedHashMap<Long, Long>();
                for (Row row : input) {
                    traceIdsToTimestamps.put(row.getLong("trace_id"), this.deserializeTs(row, "ts"));
                }
                return traceIdsToTimestamps;
            });
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugSelectTraceIdsBySpanName(serviceSpanName, startTs, endTs, limit), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugSelectTraceIdsBySpanName(String serviceSpanName, long startTs, long endTs, int limit) {
        return this.selectTraceIdsByServiceName.getQueryString().replace(":service_span_name", serviceSpanName).replace(":start_ts", new Date(startTs / 1000L).toString()).replace(":end_ts", new Date(endTs / 1000L).toString()).replace(":limit_", String.valueOf(limit));
    }

    public ListenableFuture<Void> storeTraceIdBySpanName(String serviceName, String spanName, long timestamp, long traceId, int ttl) {
        Preconditions.checkNotNull((Object)serviceName);
        Preconditions.checkArgument((!serviceName.isEmpty() ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)spanName);
        Preconditions.checkArgument((!spanName.isEmpty() ? 1 : 0) != 0);
        try {
            String serviceSpanName = serviceName + "." + spanName;
            BoundStatement bound = this.insertTraceIdBySpanName.bind().setString("service_span_name", serviceSpanName).setBytesUnsafe("ts", this.serializeTs(timestamp)).setLong("trace_id", traceId).setInt("ttl_", ttl);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugInsertTraceIdBySpanName(serviceSpanName, timestamp, traceId, ttl));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), this.resultSetToVoidFunction);
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugInsertTraceIdBySpanName(serviceName, timestamp, traceId, ttl), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugInsertTraceIdBySpanName(String serviceSpanName, long timestamp, long traceId, int ttl) {
        return this.insertTraceIdBySpanName.getQueryString().replace(":service_span_name", serviceSpanName).replace(":ts", String.valueOf(timestamp)).replace(":trace_id", String.valueOf(traceId)).replace(":ttl_", String.valueOf(ttl));
    }

    public ListenableFuture<Map<Long, Long>> getTraceIdsByAnnotation(ByteBuffer annotationKey, long endTs, long lookback, int limit) {
        long startTs = endTs - lookback;
        try {
            BoundStatement bound = this.selectTraceIdsByAnnotations.bind().setBytes("annotation", annotationKey).setList("bucket", ALL_BUCKETS).setBytesUnsafe("start_ts", this.serializeTs(startTs)).setBytesUnsafe("end_ts", this.serializeTs(endTs)).setInt("limit_", limit);
            bound.setFetchSize(Integer.MAX_VALUE);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugSelectTraceIdsByAnnotations(annotationKey, startTs, endTs, limit));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), input -> {
                LinkedHashMap<Long, Long> traceIdsToTimestamps = new LinkedHashMap<Long, Long>();
                for (Row row : input) {
                    traceIdsToTimestamps.put(row.getLong("trace_id"), this.deserializeTs(row, "ts"));
                }
                return traceIdsToTimestamps;
            });
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugSelectTraceIdsByAnnotations(annotationKey, startTs, endTs, limit), (Throwable)ex);
            throw ex;
        }
    }

    private String debugSelectTraceIdsByAnnotations(ByteBuffer annotationKey, long startTs, long endTs, int limit) {
        return this.selectTraceIdsByAnnotations.getQueryString().replace(":annotation", new String(Bytes.getArray((ByteBuffer)annotationKey))).replace(":start_ts", new Date(startTs / 1000L).toString()).replace(":end_ts", new Date(endTs / 1000L).toString()).replace(":limit_", String.valueOf(limit));
    }

    public ListenableFuture<Void> storeTraceIdByAnnotation(ByteBuffer annotationKey, long timestamp, long traceId, int ttl) {
        try {
            BoundStatement bound = this.insertTraceIdByAnnotation.bind().setBytes("annotation", annotationKey).setInt("bucket", RAND.nextInt(10)).setBytesUnsafe("ts", this.serializeTs(timestamp)).setLong("trace_id", traceId).setInt("ttl_", ttl);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugInsertTraceIdByAnnotation(annotationKey, timestamp, traceId, ttl));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), this.resultSetToVoidFunction);
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugInsertTraceIdByAnnotation(annotationKey, timestamp, traceId, ttl), (Throwable)ex);
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugInsertTraceIdByAnnotation(ByteBuffer annotationKey, long timestamp, long traceId, int ttl) {
        return this.insertTraceIdByAnnotation.getQueryString().replace(":annotation", new String(Bytes.getArray((ByteBuffer)annotationKey))).replace(":ts", new Date(timestamp / 1000L).toString()).replace(":trace_id", String.valueOf(traceId)).replace(":ttl_", String.valueOf(ttl));
    }

    private static int compareDurationRowTimestamp(DurationRow d1, DurationRow d2) {
        return d1.timestamp.compareTo(d2.timestamp);
    }

    public ListenableFuture<Map<Long, Long>> getTraceIdsByDuration(String serviceName, String spanName, long minDuration, long maxDuration, long endTs, long startTs, int limit, int ttl) {
        long oldestData = TimeUnit.MILLISECONDS.toMicros(System.currentTimeMillis() - TimeUnit.SECONDS.toMillis(ttl));
        long safeStartTs = Math.max(startTs, oldestData);
        long safeEndTs = Math.max(endTs, oldestData);
        int startBucket = this.durationIndexBucket(safeStartTs);
        int endBucket = this.durationIndexBucket(safeEndTs);
        try {
            if (startBucket > endBucket) {
                throw new IllegalArgumentException("Start bucket (" + startBucket + ") > end bucket (" + endBucket + ")");
            }
            IntFunction<ListenableFuture> oneBucketQuery = bucket -> {
                BoundStatement bound = this.selectTraceIdsBySpanDuration.bind().setString("service_name", serviceName).setString("span_name", spanName == null ? "" : spanName).setInt("time_bucket", bucket).setLong("max_duration", maxDuration).setLong("min_duration", minDuration);
                bound.setFetchSize(limit);
                if (LOG.isDebugEnabled()) {
                    LOG.debug(this.debugSelectTraceIdsByDuration(serviceName, spanName, minDuration, maxDuration, limit));
                }
                return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), rs -> {
                    Iterable it = () -> ((ResultSet)rs).iterator();
                    return StreamSupport.stream(it.spliterator(), false).map(x$0 -> new DurationRow((Row)x$0)).filter(row -> row.timestamp >= safeStartTs && row.timestamp <= safeEndTs).limit(limit).collect(Collectors.toList());
                });
            };
            List futures = IntStream.rangeClosed(startBucket, endBucket).mapToObj(oneBucketQuery).collect(Collectors.toList());
            return Futures.transform((ListenableFuture)Futures.successfulAsList(futures), input -> input.stream().flatMap(Collection::stream).collect(Collectors.groupingBy(d -> d.trace_id, Collectors.collectingAndThen(Collectors.minBy(Repository::compareDurationRowTimestamp), d -> ((DurationRow)d.get()).timestamp))));
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugSelectTraceIdsByDuration(serviceName, spanName, minDuration, maxDuration, limit), (Throwable)ex);
            throw ex;
        }
    }

    private String debugSelectTraceIdsByDuration(String serviceName, String spanName, long minDuration, long maxDuration, int limit) {
        return this.selectTraceIdsBySpanDuration.getQueryString().replace(":service_name", serviceName).replace(":span_name", spanName).replace(":max_duration", String.valueOf(maxDuration)).replace(":min_duration", String.valueOf(minDuration)).replace(":limit_", String.valueOf(limit));
    }

    private int durationIndexBucket(long ts) {
        return (int)(ts / DURATION_INDEX_BUCKET_WINDOW_SECONDS / 1000000L);
    }

    public ListenableFuture<Void> storeTraceIdByDuration(String serviceName, String spanName, long timestamp, long duration, long traceId, int ttl) {
        try {
            BoundStatement bound = this.insertTraceIdBySpanDuration.bind().setString("service_name", serviceName).setString("span_name", spanName).setInt("bucket", this.durationIndexBucket(timestamp)).setBytesUnsafe("ts", this.serializeTs(timestamp)).setLong("duration", duration).setLong("trace_id", traceId).setInt("ttl_", ttl);
            if (LOG.isDebugEnabled()) {
                LOG.debug(this.debugInsertTraceIdBySpanDuration(serviceName, spanName, timestamp, duration, traceId, ttl));
            }
            return Futures.transform((ListenableFuture)this.session.executeAsync((Statement)bound), this.resultSetToVoidFunction);
        }
        catch (RuntimeException ex) {
            LOG.error("failed " + this.debugInsertTraceIdBySpanDuration(serviceName, spanName, timestamp, duration, traceId, ttl));
            return Futures.immediateFailedFuture((Throwable)ex);
        }
    }

    private String debugInsertTraceIdBySpanDuration(String serviceName, String spanName, long timestamp, long duration, long traceId, int ttl) {
        return this.insertTraceIdBySpanDuration.getQueryString().replace(":service_name", serviceName).replace(":span_name", spanName).replace(":bucket", String.valueOf(this.durationIndexBucket(timestamp))).replace(":ts", new Date(timestamp / 1000L).toString()).replace(":duration", String.valueOf(duration)).replace(":trace_id", String.valueOf(traceId)).replace(":ttl_", String.valueOf(ttl));
    }

    private static List<Date> getDays(long from, long to) {
        ArrayList<Date> days = new ArrayList<Date>();
        for (long time = from; time <= to; time += TimeUnit.DAYS.toMillis(1L)) {
            days.add(new Date(time));
        }
        return days;
    }

    @Override
    public void close() {
        this.session.close();
    }

    private ByteBuffer serializeTs(long timestamp) {
        return DataType.bigint().serialize((Object)(timestamp / 1000L), this.protocolVersion);
    }

    private long deserializeTs(Row row, String name) {
        return 1000L * (Long)DataType.bigint().deserialize(row.getBytesUnsafe(name), this.protocolVersion);
    }

    private static class Schema {
        private static final String SCHEMA = "/cassandra-schema-cql3.txt";

        static Map<String, String> readMetadata(String keyspace, Cluster cluster) {
            LinkedHashMap<String, String> metadata = new LinkedHashMap<String, String>();
            try (Session ignored = cluster.connect();){
                KeyspaceMetadata keyspaceMetadata = Schema.getKeyspaceMetadata(keyspace, cluster);
                Map replication = keyspaceMetadata.getReplication();
                if ("SimpleStrategy".equals(replication.get("class")) && "1".equals(replication.get("replication_factor"))) {
                    LOG.warn("running with RF=1, this is not suitable for production. Optimal is 3+");
                }
                Map tracesCompaction = keyspaceMetadata.getTable("traces").getOptions().getCompaction();
                metadata.put("traces.compaction.class", (String)tracesCompaction.get("class"));
            }
            return metadata;
        }

        private static KeyspaceMetadata getKeyspaceMetadata(String keyspace, Cluster cluster) {
            KeyspaceMetadata keyspaceMetadata = cluster.getMetadata().getKeyspace(keyspace);
            if (keyspaceMetadata == null) {
                throw new IllegalStateException(String.format("Cannot read keyspace metadata for give keyspace: %s and cluster: %s", keyspace, cluster.getClusterName()));
            }
            return keyspaceMetadata;
        }

        static void ensureExists(String keyspace, Cluster cluster) {
            try (Session session = cluster.connect();
                 InputStreamReader reader = new InputStreamReader(Schema.class.getResourceAsStream(SCHEMA));){
                for (String cmd : CharStreams.toString((Readable)reader).split(";")) {
                    if ((cmd = cmd.trim().replace(" zipkin", " " + keyspace)).isEmpty()) continue;
                    session.execute(cmd);
                }
            }
            catch (IOException ex) {
                LOG.error(ex.getMessage(), (Throwable)ex);
            }
        }

        private Schema() {
        }
    }

    private class DurationRow {
        Long trace_id;
        Long duration;
        Long timestamp;

        DurationRow(Row row) {
            this.trace_id = row.getLong("trace_id");
            this.duration = row.getLong("duration");
            this.timestamp = Repository.this.deserializeTs(row, "ts");
        }

        public String toString() {
            return String.format("trace_id=%d, duration=%d, timestamp=%d", this.trace_id, this.duration, this.timestamp);
        }
    }
}

