/**
 * 
 */
package it.micegroup.voila2runtime.specification;

import java.io.Serializable;
import java.util.Collection;
import java.util.function.Function;

import it.micegroup.voila2runtime.filter.Filter;
import it.micegroup.voila2runtime.filter.RangeFilter;
import it.micegroup.voila2runtime.filter.StringFilter;
import it.micegroup.voila2runtime.specification.FilterService;
import it.micegroup.voila2runtime.specification.SpecificationGetterService;

import javax.persistence.criteria.Path;
import javax.persistence.criteria.Root;
import javax.persistence.metamodel.CollectionAttribute;
import javax.persistence.metamodel.SingularAttribute;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
 *
 */
@Transactional
@Service
public class FilterServiceImpl<ENTITY> implements FilterService<ENTITY> {

	@Autowired
	SpecificationGetterService<ENTITY> specGetter;

	/****************************************************************************************************************
	 * Helper function che ritorna una specification su un singolo campo, che
	 * supporta equal, valueIn e null/non-null
	 *
	 * @param filter : L'oggetto filtro contenente i valori proveniente dal
	 *               frontend.
	 * @param field  : JPA static metamodel che rappresenta il campo.
	 * @param <T>    : Il tipo dell'attributo da filtrare.
	 * @return una Specification
	 *******************************************************************************************************************/
	@Override
	public <T extends Serializable> Specification<ENTITY> generateSpecification(Filter<T> filter,
			SingularAttribute<? super ENTITY, T> field) {
		if (filter.getEquals() != null) {
			return specGetter.equivalent(field, filter.getEquals());
		} else if (filter.getIn() != null) {
			return specGetter.valueIn(field, filter.getIn());
		} else if (filter.getSpecified() != null) {
			return specGetter.byFieldSpecified(field, filter.getSpecified());
		}
		return null;
	}

	/****************************************************************************************************************
	 * Helper function che ritorna una specification su un singolo campo, che
	 * supporta equal, valueIn e null/non-null
	 *
	 * @param filter : L'oggetto filtro (Stringa) contenente i valori proveniente
	 *               dal frontend.
	 * @param field  : JPA static metamodel che rappresenta il campo.
	 * @return una Specification
	 *******************************************************************************************************************/
	@Override
	public Specification<ENTITY> generateStringSpecification(StringFilter filter,
			SingularAttribute<? super ENTITY, String> field) {
		if (filter.getEquals() != null) {
			return specGetter.equivalent(field, filter.getEquals());
		} else if (filter.getIn() != null) {
			return specGetter.valueIn(field, filter.getIn());
		} else if (filter.getContains() != null) {
			return specGetter.likeUpperSpecification(field, filter.getContains());
		} else if (filter.getSpecified() != null) {
			return specGetter.byFieldSpecified(field, filter.getSpecified());
		}
		return null;
	}

	/****************************************************************************************************************
	 * Helper function che ritorna una specification su un singolo campo
	 * {@link Comparable}, che supporta equal, less than, greater than and
	 * less-than-or-equal-to and greater-than-or-equal-to and null/non-null
	 *
	 * @param filter : L'oggetto filtro (Stringa) contenente i valori proveniente
	 *               dal frontend.
	 * @param field  : JPA static metamodel che rappresenta il campo.
	 * @param <T>    : Il tipo dell'attributo da filtrare.
	 * @return una Specification
	 *******************************************************************************************************************/
	@Override
	public <T extends Serializable & Comparable<? super T>> Specification<ENTITY> generateRangeSpecification(
			RangeFilter<T> filter, SingularAttribute<? super ENTITY, T> field) {
		if (filter.getEquals() != null) {
			return specGetter.equivalent(field, filter.getEquals());
		} else if (filter.getIn() != null) {
			return specGetter.valueIn(field, filter.getIn());
		}

		Specification<ENTITY> result = Specification.where(null);
		if (filter.getSpecified() != null) {
			result = result.and(specGetter.byFieldSpecified(field, filter.getSpecified()));
		}
		if (filter.getGreaterThan() != null) {
			result = result.and(specGetter.greaterThan(field, filter.getGreaterThan()));
		}
		if (filter.getGreaterOrEqualThan() != null) {
			result = result.and(specGetter.greaterThanOrEqualTo(field, filter.getGreaterOrEqualThan()));
		}
		if (filter.getLessThan() != null) {
			result = result.and(specGetter.lessThan(field, filter.getLessThan()));
		}
		if (filter.getLessOrEqualThan() != null) {
			result = result.and(specGetter.lessThanOrEqualTo(field, filter.getLessOrEqualThan()));
		}
		return result;
	}

	/**************************************************************************************************************************
	 * Helper function che ritorna una specification su una one-to-one o many-to-one
	 * reference. Utilizzo:
	 * 
	 * <pre>
	 * Specification&lt;Employee&gt; specByProjectId = generateReferringEntitySpecification(criteria.getProjectId(),
	 * 		Employee_.project, Project_.id);
	 * Specification&lt;Employee&gt; specByProjectName = generateReferringEntitySpecification(criteria.getProjectName(),
	 * 		Employee_.project, Project_.name);
	 * </pre>
	 *
	 * @param filter     : L'oggetto filtro contenente i valori, che devono matchare
	 *                   oppure un flag.
	 * @param reference  : L'attributo del metamodello per la referring entity.
	 * @param valueField : L'attributo del metamodello per la referring entity, dove
	 *                   l'uguaglinza dovrebbe essere verificata
	 * @param <OTHER>    : Il tipo della referenced entity.
	 * @param <T>        : Il tipo dell'attributo da filtrare.
	 * @return a Specification
	 ***************************************************************************************************************************/
	@Override
	public <OTHER, T extends Serializable & Comparable<? super T>> Specification<ENTITY> generateReferringEntityRangeSpecification(
			RangeFilter<T> filter, SingularAttribute<? super ENTITY, ? extends OTHER> reference,
			SingularAttribute<OTHER, T> valueField) {
		if (filter.getEquals() != null) {
			return specGetter.equivalent(reference, valueField, filter.getEquals());
		} else if (filter.getGreaterThan() != null) {
			return specGetter.greaterThan(reference, valueField, filter.getGreaterThan());
		} else if (filter.getGreaterOrEqualThan() != null) {
			return specGetter.greaterOrEqualThan(reference, valueField, filter.getGreaterOrEqualThan());
		} else if (filter.getLessThan() != null) {
			return specGetter.lessThan(reference, valueField, filter.getLessThan());
		} else if (filter.getLessOrEqualThan() != null) {
			return specGetter.lessOrEqualThan(reference, valueField, filter.getLessOrEqualThan());
		}
		return null;
	}

	/**************************************************************************************************************************
	 * Helper function che ritorna una specification su una one-to-one o many-to-one
	 * reference. Utilizzo:
	 * 
	 * <pre>
	 * Specification&lt;Employee&gt; specByProjectId = generateReferringEntitySpecification(criteria.getProjectId(),
	 * 		Employee_.project, Project_.id);
	 * Specification&lt;Employee&gt; specByProjectName = generateReferringEntitySpecification(criteria.getProjectName(),
	 * 		Employee_.project, Project_.name);
	 * </pre>
	 *
	 * @param filter     : L'oggetto filtro contenente i valori, che devono matchare
	 *                   oppure un flag.
	 * @param reference  : L'attributo del metamodello per la referring entity.
	 * @param valueField : L'attributo del metamodello per la referring entity, dove
	 *                   l'uguaglinza dovrebbe essere verificata
	 * @param <OTHER>    : Il tipo della referenced entity.
	 * @param <T>        : Il tipo dell'attributo da filtrare.
	 * @return a Specification
	 ***************************************************************************************************************************/
	@Override
	public <OTHER, T extends Serializable & Comparable<? super T>> Specification<ENTITY> generateReferringEntitySpecification(
			Filter<T> filter, SingularAttribute<? super ENTITY, ? extends OTHER> reference,
			SingularAttribute<OTHER, T> valueField) {
		if (filter.getEquals() != null) {
			return specGetter.equivalent(reference, valueField, filter.getEquals());
		} else if (filter.getIn() != null) {
			return specGetter.valueIn(reference, valueField, filter.getIn());
		} else if (filter.getSpecified() != null) {
			return specGetter.byFieldSpecified(reference, filter.getSpecified());
		}
		return null;
	}

	/**************************************************************************************************************************
	 * Helper function che ritorna una specification su una one-to-one o many-to-one
	 * reference. Utilizzo:
	 * 
	 * <pre>
	 * Specification&lt;Employee&gt; specByProjectId = generateReferringEntitySpecification(criteria.getProjectId(),
	 * 		Employee_.project, Project_.id);
	 * Specification&lt;Employee&gt; specByProjectName = generateReferringEntitySpecification(criteria.getProjectName(),
	 * 		Employee_.project, Project_.name);
	 * </pre>
	 *
	 * @param filter     : L'oggetto filtro contenente i valori, che devono matchare
	 *                   oppure un flag.
	 * @param reference  : L'attributo del metamodello per la referring entity.
	 * @param valueField : L'attributo del metamodello per la referring entity, dove
	 *                   l'uguaglinza dovrebbe essere verificata
	 * @param <OTHER>    : Il tipo della referenced entity.
	 * @return a Specification
	 ***************************************************************************************************************************/
	@Override
	public <OTHER> Specification<ENTITY> generateReferringEntityStringSpecification(StringFilter filter,
			SingularAttribute<? super ENTITY, ? extends OTHER> reference, SingularAttribute<OTHER, String> valueField) {
		if (filter.getEquals() != null) {
			return specGetter.equivalent(reference, valueField, filter.getEquals());
		} else if (filter.getContains() != null) {
			return specGetter.likeUpperSpecification(reference, valueField, filter.getContains());
		} else if (filter.getSpecified() != null) {
			return specGetter.byFieldSpecified(reference, filter.getSpecified());
		}
		return null;
	}

	/**************************************************************************************************************************
	 * Helper function che ritorna una specification su una one-to-one o many-to-one
	 * reference. Utilizzo:
	 * 
	 * <pre>
	 * Specification&lt;Employee&gt; specByEmployeeId = buildReferringEntitySpecification(criteria.getEmployeId(),
	 * 		Project_.employees, Employee_.id);
	 * Specification&lt;Employee&gt; specByEmployeeName = buildReferringEntitySpecification(criteria.getEmployeName(),
	 * 		Project_.project, Project_.name);
	 * </pre>
	 *
	 * @param filter     : L'oggetto filtro contenente i valori, che devono matchare
	 *                   oppure un flag.
	 * @param reference  : L'attributo del metamodello per la referring entity.
	 * @param valueField : L'attributo del metamodello per la referring entity, dove
	 *                   l'uguaglinza dovrebbe essere verificata
	 * @param <OTHER>    : Il tipo della referenced entity.
	 * @param <T>        : Il tipo dell'attributo da filtrare.
	 * @return a Specification
	 ***************************************************************************************************************************/
	@Override
	public <OTHER, T extends Serializable> Specification<ENTITY> buildReferringEntitySpecification(Filter<T> filter,
			CollectionAttribute<ENTITY, OTHER> reference, SingularAttribute<OTHER, T> valueField) {
		if (filter.getEquals() != null) {
			return specGetter.equivalent(reference, valueField, filter.getEquals());
		} else if (filter.getSpecified() != null) {
			return specGetter.byFieldSpecified(reference, filter.getSpecified());
		}
		return null;
	}

	/***************************************************************************************
	 * Helper function che ritorna una specification per la ricerca del sub-child
	 * 
	 * @param filter:          L'oggetto filtro contenente i valori, che devono
	 *                         matchare oppure un flag.
	 * @param reference:       L'attributo del metamodello per la referring entity
	 *                         (classe associativa).
	 * @param referencedField: L'attributo del metamodello per la referring entity
	 *                         (child).
	 * @param valueField:      L'attributo del metamodello per la referring entity,
	 *                         dove l'uguaglinza dovrebbe essere verificata (campo
	 *                         del child).
	 * 
	 * @param <OTHER>          : tipo dell'entità referenziata (child)
	 * @param <Y>              : tipo del campo referenziato nell'entità
	 *                         referenziata (campo del child)
	 * 
	 * @return a Specification
	 */
	@Override
	public <OTHER, X extends Serializable, Y> Specification<ENTITY> buildReReferringEntitySpecification(
			Filter<X> filter, CollectionAttribute<ENTITY, OTHER> reference, SingularAttribute<OTHER, Y> referencedField,
			SingularAttribute<Y, X> valueField) {
		if (filter.getEquals() != null) {
			return specGetter.equivalent(reference, valueField, referencedField, filter.getEquals());
		} else if (filter.getSpecified() != null) {
			return specGetter.byFieldSpecified(reference, filter.getSpecified());
		}
		return null;
	}

	/**
	 * Metodo che ritorna una Specification per la ricerca in un range (utile per
	 * date, numeri etc...)
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 *
	 * @param <OTHER>      : The referenced type containing the represented
	 *                     attribute
	 * @param <T>          : The referenced type of the represented attribute
	 * @param filter       : L'oggetto filtro contenente i valori dei filtri da
	 *                     applicare
	 * @param pathFunction : Funzione per il calcolo del path a partire da un root
	 * @return a Specification
	 */
	@Override
	public <OTHER, T extends Serializable & Comparable<? super T>> Specification<ENTITY> generateRangeSpecification(
			RangeFilter<T> filter, Function<Root<ENTITY>, Path<? extends T>> pathFunction) {
		if (filter.getEquals() != null) {
			return specGetter.equivalent(filter.getEquals(), pathFunction);
		} else if (filter.getGreaterThan() != null) {
			return specGetter.greaterThan(filter.getGreaterThan(), pathFunction);
		} else if (filter.getGreaterOrEqualThan() != null) {
			return specGetter.greaterOrEqualThan(filter.getGreaterOrEqualThan(), pathFunction);
		} else if (filter.getLessThan() != null) {
			return specGetter.lessThan(filter.getLessThan(), pathFunction);
		} else if (filter.getLessOrEqualThan() != null) {
			return specGetter.lessOrEqualThan(filter.getLessOrEqualThan(), pathFunction);
		}
		return null;
	}

	/**
	 * Metodo che ritorna una Specification per la ricerca su Stringhe (utile per
	 * Like, contains etc...)
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 *
	 * @param filter       : L'oggetto filtro contenente i valori dei filtri da
	 *                     applicare
	 * @param pathFunction : Funzione per il calcolo del path a partire da un root
	 * @return Metodo che ritorna una Specification per la ricerca su Stringhe
	 *         (utile per Like, contains etc...)
	 */
	@Override
	public Specification<ENTITY> generateStringSpecification(StringFilter filter,
			Function<Root<ENTITY>, Path<String>> pathFunction) {
		if (filter.getEquals() != null) {
			return specGetter.stringEquivalent(filter.getEquals(), pathFunction);
		} else if (filter.getIn() != null) {
			return specGetter.stringIn(filter.getIn(), pathFunction);
		} else if (filter.getContains() != null) {
			return specGetter.likeUpperSpecification(filter.getContains(), pathFunction);
		} else if (filter.getSpecified() != null) {
			return specGetter.byFieldSpecified(filter.getSpecified(), pathFunction);
		}
		return null;
	}

	/**
	 * Composizione di una specification con una semplice stringa che deve essere
	 * equivalente nel path in input.
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 */
	@Override
	public <OTHER, T extends Serializable & Comparable<? super T>> Specification<ENTITY> generateEqualsSpecification(
			T filter, Function<Root<ENTITY>, Path<? extends T>> pathFunction) {
		if (filter != null) {
			return specGetter.equivalent(filter, pathFunction);
		}
		return null;
	}
}
