package it.micegroup.voila2runtime.specification;

import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import javax.persistence.criteria.CriteriaBuilder.In;
import javax.persistence.criteria.Path;
import javax.persistence.criteria.Root;
import javax.persistence.metamodel.CollectionAttribute;
import javax.persistence.metamodel.SingularAttribute;

import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;

/**
 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
 *
 * 
 *         Utility class needed to build a Specification used as criteria input.
 *
 * @param <ENTITY> Entity class of which Specification is need
 */
@Service
public class SpecificationGetterServiceImpl<ENTITY> implements SpecificationGetterService<ENTITY> {

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro di eguaglianza
	 * 
	 * @param value : il valore target con cui effettuare il confronto
	 * @param field : il valore presente nel metamodello generato dal plugin di
	 *              maven jpamodelgen (NomeClasse_) che risolve il nome del campo su
	 *              cui filtrare. Target Path: target/generated-sources/apt
	 * @return Specification per uguaglianza
	 */
	@Override
	public <T> Specification<ENTITY> equivalent(SingularAttribute<? super ENTITY, T> field, final T value) {
		return (root, query, cBuilder) -> cBuilder.equal(root.get(field), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro di eguaglianza
	 * 
	 * @param value     : il valore target con cui effettuare il confronto
	 * @param reference : la reference entity su cui effettuare l'uguaglianza
	 * @param idField   : il valore presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen (NomeClasse_) che risolve il nome del
	 *                  campo su cui filtrare. Target Path:
	 *                  target/generated-sources/apt
	 * @return Specification per uguaglianza
	 */
	@Override
	public <OTHER, T, X> Specification<ENTITY> equivalent(SingularAttribute<? super ENTITY, ? extends OTHER> reference,
			SingularAttribute<OTHER, T> idField, X value) {
		return (root, query, cBuilder) -> cBuilder.equal(root.get(reference).get(idField), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro di eguaglianza
	 * 
	 * @param value     : il valore target con cui effettuare il confronto
	 * @param reference : la reference entity su cui effettuare l'uguaglianza
	 * @param idField   : il valore presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen (NomeClasse_) che risolve il nome del
	 *                  campo su cui filtrare. Target Path:
	 *                  target/generated-sources/apt
	 * @return Specification per uguaglianza
	 */
	@Override
	public <OTHER, T> Specification<ENTITY> equivalent(CollectionAttribute<? super ENTITY, OTHER> reference,
			SingularAttribute<OTHER, T> idField, T value) {
		return (root, query, cBuilder) -> cBuilder.equal(root.join(reference).get(idField), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro di eguaglianza per la classe child referenziata
	 * indirettamente.
	 * 
	 * @param value           : il valore target con cui effettuare il confronto
	 * @param reference       : la reference entity (classe associativa)
	 * @param referencedField : la vera reference entity (child)
	 * @param idField         : il valore presente nel metamodello generato dal
	 *                        plugin di maven jpamodelgen (NomeClasse_) che risolve
	 *                        il nome del campo su cui filtrare. Target Path:
	 *                        target/generated-sources/apt
	 * 
	 * @param <OTHER>         : tipo dell'entità referenziata (child)
	 * @param <T>             : tipo del valore che stiamo passando sul quale
	 *                        effettuare il confronto
	 * @param <Y>             : tipo del campo referenziato nell'entità referenziata
	 *                        (campo del child)
	 * 
	 * @return Specification per uguaglianza
	 */
	@Override
	public <OTHER, T, Y> Specification<ENTITY> equivalent(CollectionAttribute<? super ENTITY, OTHER> reference,
			SingularAttribute<Y, T> idField, SingularAttribute<OTHER, Y> referencedField, T value) {
		return (root, query, cBuilder) -> cBuilder.equal(root.join(reference).join(referencedField).get(idField),
				value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro che verifica se il campo indicato è contenuto in una
	 * lista di valori passati in input.
	 * 
	 * @param values : lista di valori target in cui effettuare il confronto
	 * @param field  : il valore presente nel metamodello generato dal plugin di
	 *               maven jpamodelgen (NomeClasse_) che risolve il nome del campo
	 *               su cui filtrare. Target Path: target/generated-sources/apt
	 * @return Specification per uguaglianza
	 */
	@Override
	public <T> Specification<ENTITY> valueIn(SingularAttribute<? super ENTITY, T> field, final Collection<T> values) {
		return (root, query, cBuilder) -> {
			In<T> in = cBuilder.in(root.get(field));
			for (T value : values) {
				in = in.value(value);
			}
			return in;
		};
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro che verifica se il campo indicato è contenuto in una
	 * lista di valori passati in input.
	 * 
	 * @param values     : lista di valori target in cui effettuare il confronto
	 * @param reference  : la reference entity su cui effettuare l'uguaglianza
	 * @param valueField : il valore presente nel metamodello generato dal plugin di
	 *                   maven jpamodelgen (NomeClasse_) che risolve il nome del
	 *                   campo su cui filtrare. Target Path:
	 *                   target/generated-sources/apt
	 * @return Specification per uguaglianza
	 */
	@Override
	public <OTHER, T> Specification<ENTITY> valueIn(SingularAttribute<? super ENTITY, ? extends OTHER> reference,
			SingularAttribute<OTHER, T> valueField, final Collection<T> values) {
		return (root, query, builder) -> {
			In<T> in = builder.in(root.get(reference).get(valueField));
			for (T value : values) {
				in = in.value(value);
			}
			return in;
		};
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro maggiore di
	 * 
	 * @param value : il valore target con cui effettuare il confronto
	 * @param field : il valore presente nel metamodello generato dal plugin di
	 *              maven jpamodelgen (NomeClasse_) che risolve il nome del campo su
	 *              cui filtrare. Target Path: target/generated-sources/apt
	 * @param <T>   : tipo del valore che stiamo passando sul qual effettuare il
	 *              confronto
	 * @return Specification per filtro maggiore di
	 */
	@Override
	public <T extends Comparable<? super T>> Specification<ENTITY> greaterThan(
			SingularAttribute<? super ENTITY, T> field, final T value) {
		return (root, query, cBuilder) -> cBuilder.greaterThan(root.get(field), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro maggiore o uguale a
	 * 
	 * @param value : il valore target con cui effettuare il confronto
	 * @param field : il valore presente nel metamodello generato dal plugin di
	 *              maven jpamodelgen (NomeClasse_) che risolve il nome del campo su
	 *              cui filtrare. Target Path: target/generated-sources/apt
	 * @param <T>   : tipo del valore che stiamo passando sul qual effettuare il
	 *              confronto
	 * @return Specification per filtro maggiore o uguale a
	 */
	@Override
	public <T extends Comparable<? super T>> Specification<ENTITY> greaterThanOrEqualTo(
			SingularAttribute<? super ENTITY, T> field, final T value) {
		return (root, query, cBuilder) -> cBuilder.greaterThanOrEqualTo(root.get(field), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro minore di
	 * 
	 * @param value : il valore target con cui effettuare il confronto
	 * @param field : il valore presente nel metamodello generato dal plugin di
	 *              maven jpamodelgen (NomeClasse_) che risolve il nome del campo su
	 *              cui filtrare. Target Path: target/generated-sources/apt
	 * @param <T>   : tipo del valore che stiamo passando sul qual effettuare il
	 *              confronto
	 * @return Specification per filtro minore di
	 */
	@Override
	public <T extends Comparable<? super T>> Specification<ENTITY> lessThan(SingularAttribute<? super ENTITY, T> field,
			final T value) {
		return (root, query, cBuilder) -> cBuilder.lessThan(root.get(field), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro minore o uguale a
	 * 
	 * @param value : il valore target con cui effettuare il confronto
	 * @param field : il valore presente nel metamodello generato dal plugin di
	 *              maven jpamodelgen (NomeClasse_) che risolve il nome del campo su
	 *              cui filtrare. Target Path: target/generated-sources/apt
	 * @param <T>   : tipo del valore che stiamo passando sul qual effettuare il
	 *              confronto
	 * @return Specification per filtro minore o uguale a
	 */
	@Override
	public <T extends Comparable<? super T>> Specification<ENTITY> lessThanOrEqualTo(
			SingularAttribute<? super ENTITY, T> field, final T value) {
		return (root, query, cBuilder) -> cBuilder.lessThanOrEqualTo(root.get(field), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro uguale tra string case-insensitive
	 * 
	 * @param field : il valore presente nel metamodello generato dal plugin di
	 *              maven jpamodelgen
	 * @param value : il valore target con cui effettuare il confronto Target Path:
	 *              target/generated-sources/apt
	 * @return Specification per uguale tra string case-insensitive
	 */
	@Override
	public Specification<ENTITY> likeUpperSpecification(SingularAttribute<? super ENTITY, String> field,
			final String value) {
		return (root, query, cBuilder) -> cBuilder.like(cBuilder.upper(root.get(field)), wrapLikeQuery(value));
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad una Jpa Criteria
	 * ed esegue il controllo sulla presenza del value
	 * 
	 * @param field     : il valore presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen
	 * @param specified :il valore target di cui verificare la presenza in db
	 * @return Metodo che restituisce una Specification da dare in input ad una Jpa
	 *         Criteria ed esegue il controllo sulla presenza del value
	 */
	@Override
	public <X> Specification<ENTITY> byFieldSpecified(SingularAttribute<? super ENTITY, X> field,
			final boolean specified) {
		return specified ? (root, query, cBuilder) -> cBuilder.isTrue(root.get(field).as(Boolean.class))
				: (root, query, cBuilder) -> cBuilder.or(cBuilder.isFalse(root.<X>get(field).as(Boolean.class)),
						cBuilder.isNull(root.get(field)));
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad una Jpa Criteria
	 * ed esegue il controlla se il set è vuoto
	 * 
	 * @param field     : il valore presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen
	 * @param specified :il valore target di cui verificare la presenza in db
	 * @return Metodo che restituisce una Specification da dare in input ad una Jpa
	 *         Criteria ed esegue il controlla se il set è vuoto
	 */
	@Override
	public <X> Specification<ENTITY> byFieldSpecified(CollectionAttribute<ENTITY, X> field, final boolean specified) {
		return specified ? (root, query, cBuilder) -> cBuilder.isNotEmpty(root.get(field))
				: (root, query, cBuilder) -> cBuilder.isEmpty(root.get(field));
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro di eguaglianza sulla collection di oggetti
	 *
	 * @param reference : la reference entity su cui effettuare l'uguaglianza
	 * @param value     : il valore target con cui effettuare il confronto
	 * @param idField   : il set presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen (NomeClasse_) che risolve il nome del
	 *                  campo su cui filtrare. Target Path:
	 *                  target/generated-sources/apt
	 * @return Specification per uguaglianza su collection referenziata
	 */
	@Override
	public <OTHER, X> Specification<ENTITY> equalsSetSpecification(CollectionAttribute<? super ENTITY, OTHER> reference,
			SingularAttribute<OTHER, X> idField, X value) {
		return (root, query, cBuilder) -> cBuilder.equal(root.join(reference).get(idField), value);
	}

	/**
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 *
	 * @param value     : lista di valori target in cui effettuare il confronto
	 * @param reference : la reference entity su cui effettuare l'uguaglianza
	 * @param field     : il valore presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen (NomeClasse_) che risolve il nome del
	 *                  campo su cui filtrare. Target Path:
	 *                  target/generated-sources/apt
	 * @return Specification per maggiore di
	 */
	@Override
	public <OTHER, T extends Comparable<? super T>> Specification<ENTITY> greaterThan(
			SingularAttribute<? super ENTITY, ? extends OTHER> reference, SingularAttribute<OTHER, T> field,
			final T value) {
		return (root, query, cBuilder) -> cBuilder.greaterThan(root.get(reference).get(field), value);
	}

	/**
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 *
	 * @param value     : lista di valori target in cui effettuare il confronto
	 * @param reference : la reference entity su cui effettuare l'uguaglianza
	 * @param field     : il valore presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen (NomeClasse_) che risolve il nome del
	 *                  campo su cui filtrare. Target Path:
	 *                  target/generated-sources/apt
	 * @return Specification per maggiore o uguale a
	 */
	@Override
	public <OTHER, T extends Comparable<? super T>> Specification<ENTITY> greaterOrEqualThan(
			SingularAttribute<? super ENTITY, ? extends OTHER> reference, SingularAttribute<OTHER, T> field,
			final T value) {
		return (root, query, cBuilder) -> cBuilder.greaterThanOrEqualTo(root.get(reference).get(field), value);
	}

	/**
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 *
	 * @param value     : lista di valori target in cui effettuare il confronto
	 * @param reference : la reference entity su cui effettuare l'uguaglianza
	 * @param field     : il valore presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen (NomeClasse_) che risolve il nome del
	 *                  campo su cui filtrare. Target Path:
	 *                  target/generated-sources/apt
	 * @return Specification per minore di
	 */
	@Override
	public <OTHER, T extends Comparable<? super T>> Specification<ENTITY> lessThan(
			SingularAttribute<? super ENTITY, ? extends OTHER> reference, SingularAttribute<OTHER, T> field,
			final T value) {
		return (root, query, cBuilder) -> cBuilder.lessThan(root.get(reference).get(field), value);
	}

	/**
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 *
	 * @param value     : lista di valori target in cui effettuare il confronto
	 * @param reference : la reference entity su cui effettuare l'uguaglianza
	 * @param field     : il valore presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen (NomeClasse_) che risolve il nome del
	 *                  campo su cui filtrare. Target Path:
	 *                  target/generated-sources/apt
	 * @return Specification per minore o uguale a
	 */
	@Override
	public <OTHER, T extends Comparable<? super T>> Specification<ENTITY> lessOrEqualThan(
			SingularAttribute<? super ENTITY, ? extends OTHER> reference, SingularAttribute<OTHER, T> field,
			final T value) {
		return (root, query, cBuilder) -> cBuilder.lessThanOrEqualTo(root.get(reference).get(field), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro uguale tra string case-insensitive
	 * 
	 * @param field     : il valore presente nel metamodello generato dal plugin di
	 *                  maven jpamodelgen
	 * @param value     : il valore target con cui effettuare il confronto Target
	 *                  Path: target/generated-sources/apt
	 * @param reference : la reference entity su cui effettuare l'uguaglianza
	 * @return Specification per uguale tra string case-insensitive
	 */
	public <OTHER> Specification<ENTITY> likeUpperSpecification(
			SingularAttribute<? super ENTITY, ? extends OTHER> reference, SingularAttribute<OTHER, String> field,
			final String value) {
		return (root, query, cBuilder) -> cBuilder.like(cBuilder.upper(root.get(reference).get(field)),
				wrapLikeQuery(value));
	}

	/**
	 * Funzione di utility per la creazione di query per la ricerca di stringhe
	 * 
	 * @param txt testo contenuto nella stringa da ricercare
	 * @return
	 */
	private String wrapLikeQuery(String txt) {
		return "%" + txt.toUpperCase() + '%';
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro di eguaglianza
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * 
	 * @param value        : il valore target con cui effettuare il confronto
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @return Specification per uguaglianza
	 */
	@Override
	public <T extends Comparable<? super T>> Specification<ENTITY> equivalent(final T value,
			Function<Root<ENTITY>, Path<? extends T>> pathFunction) {
		return (root, query, cBuilder) -> cBuilder.equal(pathFunction.apply(root), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro minore o uguale di
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 *
	 * @param value        : il valore target con cui effettuare il confronto
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @return Specification per filtro minore di
	 */
	@Override
	public <T extends Comparable<? super T>> Specification<ENTITY> lessOrEqualThan(final T value,
			Function<Root<ENTITY>, Path<? extends T>> pathFunction) {
		return (root, query, cBuilder) -> cBuilder.lessThanOrEqualTo(pathFunction.apply(root), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro maggiore o uguale a
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * 
	 * @param value        : il valore target con cui effettuare il confronto
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @return Specification per filtro minore di
	 */
	@Override
	public <T extends Comparable<? super T>> Specification<ENTITY> greaterOrEqualThan(final T value,
			Function<Root<ENTITY>, Path<? extends T>> pathFunction) {
		return (root, query, cBuilder) -> cBuilder.greaterThanOrEqualTo(pathFunction.apply(root), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro minore di
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * 
	 * @param value        : il valore target con cui effettuare il confronto
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @return Specification per filtro minore di
	 */
	@Override
	public <T extends Comparable<? super T>> Specification<ENTITY> lessThan(final T value,
			Function<Root<ENTITY>, Path<? extends T>> pathFunction) {
		return (root, query, cBuilder) -> cBuilder.lessThan(pathFunction.apply(root), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro maggiore di
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * 
	 * @param value        : il valore target con cui effettuare il confronto
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @return Specification per filtro maggiore di
	 */
	@Override
	public <T extends Comparable<? super T>> Specification<ENTITY> greaterThan(final T value,
			Function<Root<ENTITY>, Path<? extends T>> pathFunction) {
		return (root, query, cBuilder) -> cBuilder.greaterThan(pathFunction.apply(root), value);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro di eguaglianza
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * @param equals       value to be equals to
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @return Specification per uguaglianza
	 */
	@Override
	public Specification<ENTITY> equivalent(String equals, Function<Root<ENTITY>, Path<?>> pathFunction) {
		return (root, query, cBuilder) -> cBuilder.equal(pathFunction.apply(root), equals);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro di eguaglianza
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * 
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @return Specification per uguaglianza
	 */
	@Override
	public Specification<ENTITY> stringEquivalent(String equals, Function<Root<ENTITY>, Path<String>> pathFunction) {
		return (root, query, cBuilder) -> cBuilder.equal(pathFunction.apply(root), equals);
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro che verifica se il campo indicato è contenuto in una
	 * lista di valori passati in input.
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * 
	 * @param values       : lista di valori target in cui effettuare il confronto
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @return Specification per uguaglianza
	 */
	@Override
	public <T> Specification<ENTITY> valueIn(List<T> values, Function<Root<ENTITY>, Path<? extends T>> pathFunction) {
		return (root, query, cBuilder) -> {
			In<T> in = cBuilder.in(pathFunction.apply(root));
			for (T value : values) {
				in = in.value(value);
			}
			return in;
		};
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue un filtro che verifica se il campo indicato è contenuto in una
	 * lista di valori passati in input.
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * 
	 * @param values       : lista di valori target in cui effettuare il confronto
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @return Specification per uguaglianza
	 */
	@Override
	public Specification<ENTITY> stringIn(List<String> values, Function<Root<ENTITY>, Path<String>> pathFunction) {
		return (root, query, cBuilder) -> {
			In<String> in = cBuilder.in(pathFunction.apply(root));
			for (String value : values) {
				in = in.value(value);
			}
			return in;
		};
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad un Jpa Criteria
	 * ed esegue il filtro uguale tra string case-insensitive
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * 
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @param contains     : il valore target con cui effettuare il confronto Target
	 *                     Path: target/generated-sources/apt
	 * @return Specification per uguale tra string case-insensitive
	 */
	@Override
	public Specification<ENTITY> likeUpperSpecification(String contains,
			Function<Root<ENTITY>, Path<String>> pathFunction) {
		return (root, query, cBuilder) -> cBuilder.like(cBuilder.upper(pathFunction.apply(root)),
				wrapLikeQuery(contains));
	}

	/**
	 * Metodo che restituisce una Specification da dare in input ad una Jpa Criteria
	 * ed esegue il controllo sulla presenza del value
	 * 
	 * @author Vittorio Niespolo vittorio.niespolo@micegroup.it
	 * 
	 * @param pathFunction funzione per il calcolo del path a aprtire da un root
	 * @param specified    :il valore target di cui verificare la presenza in db
	 * @return specification per exists
	 */
	@Override
	public Specification<ENTITY> byFieldSpecified(Boolean specified,
			Function<Root<ENTITY>, Path<String>> pathFunction) {
		return specified ? (root, query, cBuilder) -> cBuilder.isTrue(pathFunction.apply(root).as(Boolean.class))
				: (root, query, cBuilder) -> cBuilder.or(cBuilder.isFalse(pathFunction.apply(root).as(Boolean.class)),
						cBuilder.isNull(pathFunction.apply(root)));
	}
}
