package ltd.dolink.adapter.datatype;


import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView.ViewHolder;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

import ltd.dolink.adapter.datatype.ItemViewBinder.Lookup;

public abstract class DataTypeBinder implements Binder<Object, ViewHolder> {

    @NonNull
    private final Lookup itemViewLookup;
    @NonNull
    private DataTypeAdapter dataTypeAdapter;

    @NonNull
    private Notify itemViewNotify;

    public DataTypeBinder() {
        this(new DefaultLookup());
    }

    public DataTypeBinder(@NonNull Lookup itemViewBinderLookup) {
        Objects.requireNonNull(itemViewBinderLookup);
        this.itemViewLookup = itemViewBinderLookup;
    }

    @NonNull
    protected Lookup getItemViewLookup() {
        return itemViewLookup;
    }

    @NonNull
    public Notify getItemViewNotify() {
        return itemViewNotify;
    }

    @NonNull
    protected DataTypeAdapter getDataTypeAdapter() {
        return dataTypeAdapter;
    }

    void bind(@NonNull DataTypeAdapter dataTypeAdapter) {
        this.dataTypeAdapter = dataTypeAdapter;
        this.itemViewNotify = new ItemViewNotify(this.dataTypeAdapter);
    }

    public abstract int getItemCount();

    public <T> void link(@NonNull Class<T> dataType, @NonNull Linker<T> linker) {
        ItemViewLinker<T> itemViewLinker = new ItemViewLinker<>(dataType, linker, this);
        itemViewLookup.link(dataType, itemViewLinker);
    }

    protected ItemViewBinder<Object, ViewHolder> findItemViewBinderFromPosition(int position) {
        return itemViewLookup.findFromPosition(position, getItem(position));
    }

    protected ItemViewBinder<Object, ViewHolder> findItemViewBinderFromViewType(int viewType) {
        return itemViewLookup.findFromViewType(viewType);
    }


    @Override
    public int getItemViewType(int position) {
        return findItemViewBinderFromPosition(position).getItemViewType(position);
    }

    @Override
    public long getItemId(int position) {
        return findItemViewBinderFromPosition(position).getItemId(position);
    }

    @Override
    public void onViewRecycled(@NonNull ViewHolder holder) {
        findItemViewBinderFromViewType(holder.getItemViewType()).onViewRecycled(holder);
    }

    @Override
    public boolean onFailedToRecycleView(@NonNull ViewHolder holder) {
        return findItemViewBinderFromViewType(holder.getItemViewType()).onFailedToRecycleView(holder);
    }

    @Override
    public void onViewAttachedToWindow(@NonNull ViewHolder holder) {
        findItemViewBinderFromViewType(holder.getItemViewType()).onViewAttachedToWindow(holder);
    }

    @Override
    public void onViewDetachedFromWindow(@NonNull ViewHolder holder) {
        findItemViewBinderFromViewType(holder.getItemViewType()).onViewDetachedFromWindow(holder);
    }

    @NonNull
    @Override
    public ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        ItemViewBinder<Object, ViewHolder> itemViewBinder = findItemViewBinderFromViewType(viewType);
        ViewHolder viewHolder = itemViewBinder.onCreateViewHolder(parent, viewType);
        itemViewBinder.onBindViewEvents(viewHolder, viewType);
        return viewHolder;
    }

    @Override
    public void onBindViewHolder(@NonNull ViewHolder holder, int position) {
        findItemViewBinderFromViewType(holder.getItemViewType()).onBindViewHolder(holder, position);
    }

    @Override
    public void onBindViewHolder(@NonNull ViewHolder holder, int position, @NonNull List<Object> payloads) {
        findItemViewBinderFromViewType(holder.getItemViewType()).onBindViewHolder(holder, position, payloads);
    }
}


class DefaultLookup implements Lookup {
    private final Map<Integer, ItemViewBinder<Object, ViewHolder>> itemViewBinderMap = new ConcurrentHashMap<>();
    private final Map<Class<?>, ItemViewLinker<Object>> itemViewLinkerMap = new ConcurrentHashMap<>();


    @Override
    public <T> void link(@NonNull Class<T> dataType, @NonNull ItemViewLinker<T> itemViewLinker) {
        @SuppressWarnings("unchecked") ItemViewLinker<Object> linker = (ItemViewLinker<Object>) itemViewLinker;
        itemViewLinkerMap.put(dataType, linker);
    }

    @Override
    public ItemViewBinder<Object, ViewHolder> findFromPosition(int position, @NonNull Object data) {
        Class<?> dataType = data.getClass();
        ItemViewLinker<Object> itemViewLinker = itemViewLinkerMap.get(dataType);
        Objects.requireNonNull(itemViewLinker, String.format("Data type %s has not link to Linker yet.", dataType));

        int viewType = itemViewLinker.getViewType(position, data);
        return itemViewBinderMap.computeIfAbsent(viewType, integer -> {
            ItemViewBinder<Object, ViewHolder> itemViewBinder = itemViewLinker.getViewBinder(position, data);
            Objects.requireNonNull(itemViewBinder, String.format("Data type %s link to ItemViewBinder failed, Can not create an instance of ItemViewBinder", dataType));
            itemViewBinder.setItemViewLinker(itemViewLinker);
            return itemViewBinder;
        });
    }

    @Override
    public ItemViewBinder<Object, ViewHolder> findFromViewType(int viewType) {
        return itemViewBinderMap.get(viewType);
    }
}

