/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk;

import static ly.warp.sdk.utils.constants.WarpConstants.IN_APP_FILTER_ALL;

import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager.NameNotFoundException;
import android.database.Cursor;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

import androidx.appcompat.app.AlertDialog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import ly.warp.sdk.db.WarplyDBHelper;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.callbacks.CampaignsHook;
import ly.warp.sdk.io.callbacks.InboxStatsHook;
import ly.warp.sdk.io.callbacks.PostHook;
import ly.warp.sdk.io.callbacks.SimpleCallbackReceiver;
import ly.warp.sdk.io.callbacks.VolleyTransformer;
import ly.warp.sdk.io.callbacks.WarplyReadyCallback;
import ly.warp.sdk.io.models.Campaign;
import ly.warp.sdk.io.models.CampaignList;
import ly.warp.sdk.io.models.InboxStats;
import ly.warp.sdk.io.models.LoyaltyContextualOfferModel;
import ly.warp.sdk.io.request.WarplyInboxRequest;
import ly.warp.sdk.io.request.WarplyJsonArrayRequest;
import ly.warp.sdk.io.request.WarplyJsonObjectRequest;
import ly.warp.sdk.io.volley.Request.Method;
import ly.warp.sdk.io.volley.RequestQueue;
import ly.warp.sdk.io.volley.toolbox.Volley;
import ly.warp.sdk.receivers.WarplyBeaconsApplication;
import ly.warp.sdk.utils.GCMRegistrar;
import ly.warp.sdk.utils.ObjectSerializer;
import ly.warp.sdk.utils.WarpJSONParser;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.WarplyDeviceInfoCollector;
import ly.warp.sdk.utils.WarplyInitializer;
import ly.warp.sdk.utils.WarplyManagerHelper;
import ly.warp.sdk.utils.WarplyPreferences;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.utils.constants.WarpConstants.LocationModes;
import ly.warp.sdk.utils.constants.WarpConstants.ServiceRegistrationCallback;
import ly.warp.sdk.utils.managers.WarplyAnalyticsManager;
import ly.warp.sdk.utils.managers.WarplyLocationManager;
import ly.warp.sdk.utils.managers.WarplyServerPreferencesManager;
import ly.warp.sdk.utils.managers.WarplyUserManager;
import ly.warp.sdk.views.dialogs.InAppDialog;

public enum Warply {

    INSTANCE;

    Warply() {
    }

    public static Map<WarpConstants.LocationSetting, Integer> LOCATION_SETTINGS_MAP;
    private static final int MINIMUM_REQUESTS_FOR_SENDING = 10;

    public Context mContext;
    private CallbackReceiver<ServiceRegistrationCallback> mRegistrationListener;
    private RequestQueue mRequestQueue;

    /* Locks */
    private final AtomicBoolean registerLock = new AtomicBoolean(false);
    private final AtomicBoolean postLock = new AtomicBoolean(false);

    /* last received campaigns */
    private CampaignList mLastReceivedCampaigns;
    public CampaignList mInAppCampaigns;

    /* Initialization methods */

    /**
     * Initialize the Warply INSTANCE!
     *
     * @param context A context, used to handle systems resources
     *                The Warply instance, for easier use
     */
    public static WarplyInitializer getInitializer(Context context) {
        return getInitializer(context, null);
    }

    public static WarplyInitializer getInitializer(Context context, WarplyReadyCallback callback) {
        return new WarplyInitializer(context, callback, new WarplyInitializer.WarplyInitCallback() {

            @Override
            public void onInit(Context context) {
                if (context != null) {
                    INSTANCE.mContext = context.getApplicationContext();
                }
                initInternal(context);
            }
        });
    }

    private static void initInternal(Context context) {
        if (/*isInitialized()*/ context == null) {
            INSTANCE.check();
            return;
        }
        if (INSTANCE.mRequestQueue == null)
            INSTANCE.mRequestQueue = Volley.newRequestQueue(context);
        INSTANCE.mContext = context.getApplicationContext();
        WarpConstants.DEBUG = WarplyProperty.isDebugMode(INSTANCE.mContext);
        INSTANCE.isInitializedOrThrow();
        WarpConstants.GCM_SENDER_ID = WarpUtils.getLastGCMSenderId(INSTANCE.mContext);
        if (!WarpUtils.getLastApplicationUUID(context).equals(WarplyProperty.getAppUuid(context))) {
            resetWarplyWebId();
        }
        WarplyServerPreferencesManager.initiateMicroAppStatusesMap(context);
    }

    /**
     * Method used to check if the Warply.INSTANCE is correctly initialized. If
     * not, it throws a runtime exception with a message describing the reason
     */
    private void isInitializedOrThrow() {
        if (mContext == null) {
            WarpUtils.log("Warply has not been initialized, call init(Context) first");
//            return;
        }
        if (mContext != null) {
            WarpUtils.log("Warply has been initialized, all good");
//            return;
        }

        if (mContext != null) {
            String apiKey = WarplyProperty.getAppUuid(mContext);
            if (TextUtils.isEmpty(apiKey)) {
                WarpUtils.log("Warply application UUID has not been set in the Manifest");
//            return;
            }
            if (!TextUtils.isEmpty(apiKey) && (apiKey.length() != 32) && (apiKey.length() != 36)) {
                WarpUtils.log("Warply application UUID has not been set correctly in the Manifest, key got: " + apiKey);
//            return;
            }
        }
    }

    /**
     * Method used to check if the Warply INSTANCE is initialized
     *
     * @param printError True, if it should print the reason it is not correctly
     *                   initialized
     */
    public static boolean isInitialized(boolean printError) {
        return INSTANCE.isInitializedInternal(printError);
    }

    private boolean isInitializedInternal(boolean printError) {
        try {
            isInitializedOrThrow();
        } catch (RuntimeException e) {
            if (printError)
                WarpUtils
                        .log("************* WARPLY Initialisation Error ********************");
            WarpUtils.log("[WARPLY Trace] Error: " + e.getMessage());
            WarpUtils
                    .log("**************************************************************");
            return false;
        }
        return true;
    }

    /**
     * Method used to check if the Warply INSTANCE is initialized
     */
    public static boolean isInitialized() {
        return INSTANCE.isInitializedInternal();
    }

    private boolean isInitializedInternal() {
        return isInitialized(false);
    }

    /* Locking methods */

    private boolean acquirePostLockInternal() {
        synchronized (postLock) {
            if (postLock.get()) {
                return false;
            }
            postLock.set(true);
        }
        return true;
    }

    public static void releasePostLock() {
        INSTANCE.releasePostLockInternal();
    }

    private void releasePostLockInternal() {
        synchronized (postLock) {
            postLock.set(false);
        }
    }

    private boolean acquireRegisterLock() {
        synchronized (registerLock) {
            if (registerLock.get()) {
                return false;
            }
            registerLock.set(true);
        }
        return true;
    }

    private void releaseRegisterLock() {
        synchronized (registerLock) {
            registerLock.set(false);
        }
    }

    /* Generic microapp methods */

    /**
     * Method used to get the application context from the server
     *
     * @param receiver The receiver to get the callback when the context is received
     */
    public static void getContext(CallbackReceiver<JSONObject> receiver) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.getContextInternal(receiver);
    }

    private void getContextInternal(CallbackReceiver<JSONObject> receiver) {
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            if (receiver != null)
                receiver.onFailure(WarpConstants.RESULT_CODE_NOT_REGISTERED);
            return;
        }
        getFromServerInternal(null, null, receiver, null);
    }

    /**
     * Generic method used to get data from a Microapp
     *
     * @param microappName The name of the Microapp
     * @param receiver     The receiver to get the callback when the microapp data are
     *                     received
     */
    public static void getMicroappData(String microappName,
                                       CallbackReceiver<JSONObject> receiver) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.getMicroappDataInternal(microappName, receiver);
    }

    private void getMicroappDataInternal(String microappName,
                                         CallbackReceiver<JSONObject> receiver) {
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            if (receiver != null)
                receiver.onFailure(WarpConstants.RESULT_CODE_NOT_REGISTERED);
            return;
        }
        getFromServerInternal(null, microappName, receiver, null);
    }

    /**
     * Method used to get data from the server while sending data at the same
     * time
     *
     * @param microappName The name of the Microapp
     * @param jObj         The data to post
     * @param receiver     The receiver to get the callback when the microapp data are
     *                     received
     */
    public static void postReceiveMicroappData(String microappName,
                                               JSONObject jObj, CallbackReceiver<JSONObject> receiver) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.postReceiveMicroappDataInternal(microappName, jObj, receiver);
    }

    public static void postReceiveMicroappData(boolean hasAuthHeaders, String path, JSONObject jObj,
                                               CallbackReceiver<JSONObject> receiver) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.postReceiveMicroappDataInternal(hasAuthHeaders, path, jObj, receiver);
    }

    public static void postReceiveMicroappData(String microappName, boolean hasAuthHeaders, String path,
                                               JSONObject jObj, CallbackReceiver<JSONObject> receiver) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.postReceiveMicroappDataInternal(microappName, hasAuthHeaders, path, jObj, receiver);
    }

    private void postReceiveMicroappDataInternal(String microappName,
                                                 JSONObject jObj, CallbackReceiver<JSONObject> receiver) {
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            if (receiver != null)
                receiver.onFailure(WarpConstants.RESULT_CODE_NOT_REGISTERED);
            return;
        }
        JSONObject jObject = new JSONObject();
        try {
            jObject.putOpt(microappName, jObj);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        postToServerInternal(jObject, receiver, null);
    }

    private void postReceiveMicroappDataInternal(boolean hasAuthHeaders, String path, JSONObject jObj,
                                                 CallbackReceiver<JSONObject> receiver) {
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            if (receiver != null)
                receiver.onFailure(WarpConstants.RESULT_CODE_NOT_REGISTERED);
            return;
        }

        String tag = null;
        if (path.equals("cosuser")) {
            tag = "cosuser";
        }

        postToServerInternal(hasAuthHeaders, path, jObj, receiver, tag);
    }

    private void postReceiveMicroappDataInternal(String microappName, boolean hasAuthHeaders, String path,
                                                 JSONObject jObj, CallbackReceiver<JSONObject> receiver) {
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            if (receiver != null)
                receiver.onFailure(WarpConstants.RESULT_CODE_NOT_REGISTERED);
            return;
        }
        JSONObject jObject = new JSONObject();
        try {
            jObject.putOpt(microappName, jObj);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }

        postToServerInternal(hasAuthHeaders, path, jObject, receiver, null);
    }

    /**
     * Default method used to post data to context. Same as
     * postMicroappData(String,JSONObject,false)
     *
     * @param microappName The name of the Microapp
     * @param jObj         A JSON Object representing the data to be sent
     */
    public static void postMicroappData(String microappName, JSONObject jObj) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.postMicroappDataInternal(microappName, jObj);
    }

    private void postMicroappDataInternal(String microappName, JSONObject jObj) {
        postMicroappData(microappName, jObj, false);
    }

    /**
     * Method used to post data to context. If this method is forced, it might
     * consume a lot of battery. Use only for data that needs to be updated real
     * time
     *
     * @param microappName The name of the Microapp
     * @param jObj         A JSON Object representing the data to be sent
     * @param force        If the post should be forced and sent to server, or can be
     *                     queued. Do not force unimportant data to save battery.
     */
    public static void postMicroappData(String microappName, JSONObject jObj, boolean force) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.postMicroappDataInternal(microappName, jObj, force);
    }

    public static void postMicroappData(Context context, String microappName, JSONObject jObj, boolean force) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.postMicroappDataInternal(context, microappName, jObj, force);
    }

    private void postMicroappDataInternal(String microappName, JSONObject jObj, boolean force) {
        isInitializedOrThrow();
        long requestsInQueueCount = 0;
        if (jObj != null && mContext != null) {
            requestsInQueueCount = WarplyDBHelper.getInstance(mContext).addRequest(
                    microappName, jObj.toString(), force);
        }
        tryWakingSendingTaskInternal(requestsInQueueCount);
    }

    private void postMicroappDataInternal(Context context, String microappName, JSONObject jObj, boolean force) {
        isInitializedOrThrow();
        long requestsInQueueCount = 0;
        if (jObj != null && context != null) {
            requestsInQueueCount = WarplyDBHelper.getInstance(context).addRequest(
                    microappName, jObj.toString(), force);
        }
        tryWakingSendingTaskInternal(context, requestsInQueueCount);
    }

    public static void postMicroappPush(String microappName, JSONObject jObj, boolean force) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.postMicroappPushInternal(microappName, jObj, force);
    }

    private void postMicroappPushInternal(String microappName, JSONObject jObj, boolean force) {
        isInitializedOrThrow();
        long requestsInQueueCount = 0;
        if (jObj != null) {
            requestsInQueueCount = WarplyDBHelper.getInstance(mContext).addPushRequest(
                    microappName, jObj.toString(), force);
        }
        tryWakingSendingPushTaskInternal(requestsInQueueCount, false);
    }

    public static void postMicroappPushAck(String microappName, JSONObject jObj, boolean force) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.postMicroappPushAckInternal(microappName, jObj, force);
    }

    private void postMicroappPushAckInternal(String microappName, JSONObject jObj, boolean force) {
        isInitializedOrThrow();
        long requestsInQueueCount = 0;
        if (jObj != null) {
            requestsInQueueCount = WarplyDBHelper.getInstance(mContext).addPushAckRequest(
                    microappName, jObj.toString(), force);
        }
        tryWakingSendingPushAckTaskInternal(requestsInQueueCount, false);
    }

    /* Campaigns specific methods */

    /**
     * Method used to get the campaigns associated with this device
     *
     * @param receiver The receiver to get the Campaigns when the task returns
     */
    public static void getInbox(CallbackReceiver<CampaignList> receiver) {
        getInbox(null, receiver);
    }

    public static void getInbox(WarplyInboxRequest request, final CallbackReceiver<CampaignList> receiver) {

        if (request == null) {
            request = new WarplyInboxRequest();
        }

        final String requestSignature = request.getSignature();
        final WarplyPreferences warplyPreferences = new WarplyPreferences(INSTANCE.mContext);
        long elapsedTimeAfterLastUpdate = System.currentTimeMillis() - warplyPreferences.getInboxLastCachedTimeStamp(requestSignature);
        long updateInterval = request.getCacheUpdateInterval();
        boolean isNeedUpdateCampaignsByTimeStamp = elapsedTimeAfterLastUpdate > updateInterval;

        if (isNeedUpdateCampaignsByTimeStamp
                || INSTANCE.mLastReceivedCampaigns == null
                || INSTANCE.mLastReceivedCampaigns.size() == 0
                || !INSTANCE.mLastReceivedCampaigns.getRequestSignature().equals(requestSignature)) {

            final ObjectSerializer objectSerializer = new ObjectSerializer(INSTANCE.mContext);
            CampaignList campaignsCache = (CampaignList) objectSerializer.deserialize(File.separator + requestSignature);
            if (isNeedUpdateCampaignsByTimeStamp
                    || campaignsCache == null
                    || campaignsCache.size() == 0
                    || !campaignsCache.getRequestSignature().equals(requestSignature)) {

                INSTANCE.isInitializedOrThrow();
                INSTANCE.getInboxInternal(new CallbackReceiver<CampaignList>() {
                    @Override
                    public void onSuccess(CampaignList result) {
//                        WarplyManagerHelper.setCampaignList(result);


                        if (result.size() >= 0) {
                            objectSerializer.serialize(result, File.separator + requestSignature);
                            warplyPreferences.saveInboxLastCachedTimeStamp(requestSignature, System.currentTimeMillis());

                            CampaignList campaignLoyaltyList = new CampaignList();
                            campaignLoyaltyList.clear();
                            for (Campaign camp : result) {
                                try {
                                    JSONObject extraFields = WarpJSONParser.getJSONFromString(camp.getExtraFields());
                                    if (extraFields != null) {
                                        if (extraFields.length() == 0 || !(extraFields.has("ccms_offer") || extraFields.has("type"))) {
                                            campaignLoyaltyList.add(camp);
                                        }
                                    }
                                } catch (Exception exception) {
                                    campaignLoyaltyList.add(camp);
                                }
                            }

                            INSTANCE.mLastReceivedCampaigns = result;
                            if (receiver != null) {
                                receiver.onSuccess(result/*campaignLoyaltyList*/);
                            }
                        }
                    }

                    @Override
                    public void onFailure(int errorCode) {
                        if (receiver != null) {
                            receiver.onFailure(errorCode);
                        }
                    }
                }, request);

            } else {

                INSTANCE.mLastReceivedCampaigns = campaignsCache;
                if (receiver != null) {
                    receiver.onSuccess(INSTANCE.mLastReceivedCampaigns);
                }
            }

        } else {

            if (receiver != null) {
                receiver.onSuccess(INSTANCE.mLastReceivedCampaigns);
            }
        }
    }

    public static void getInboxInApp(WarplyInboxRequest request, final CallbackReceiver<CampaignList> receiver) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.getInboxInAppInternal(new CallbackReceiver<CampaignList>() {
            @Override
            public void onSuccess(CampaignList result) {
                receiver.onSuccess(result);
            }

            @Override
            public void onFailure(int errorCode) {
                receiver.onFailure(errorCode);
            }
        }, request);
    }

    public CampaignList getLastReceivedCampaigns() {
        return mLastReceivedCampaigns;
    }

    public CampaignList getInAppCampaigns() {
        return mInAppCampaigns;
    }

    public void getInboxUnreadCount(final CallbackReceiver<Integer> receiver, WarplyInboxRequest request) {

        getInbox(request, new CallbackReceiver<CampaignList>() {
            @Override
            public void onSuccess(CampaignList result) {
                receiver.onSuccess(result != null ?
                        result.getUnreadCampaignsCount() : 0);
            }

            @Override
            public void onFailure(int errorCode) {
                receiver.onSuccess(0);
            }
        });
    }

    /**
     * @deprecated Use {@link #getInbox(CallbackReceiver) getInbox} method
     * instead
     */
    @Deprecated
    public static void getCampaigns(CallbackReceiver<CampaignList> receiver) {
        getInbox(receiver);
    }

    /**
     * Method used to get the campaign stats associated with this device
     *
     * @param receiver The receiver to get the InboxStats when the task returns
     */
    public static void getInboxStats(CallbackReceiver<InboxStats> receiver) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.getInboxStatsInternal(receiver);
    }

    private void getInboxInternal(CallbackReceiver<CampaignList> receiver, WarplyInboxRequest request) {

        WarpUtils.log("************* WARPLY Microapp ********************");
        if (!WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.OFFERS)) {
            WarpUtils.log("[WARP Trace] Offers Microapp is not active");
            WarpUtils.log("**************************************************");
            receiver.onFailure(2);
            return;
        }
        WarpUtils.log("[WARP Trace] Offers Microapp is active");
        WarpUtils.log("**************************************************");

        request = request != null ? request : new WarplyInboxRequest();
        postReceiveMicroappDataInternal(WarpConstants.MICROAPP_CAMPAIGNS, request.toJson(),
                new CampaignsHook(receiver, request.getSignature()));
    }

    private void getInboxInAppInternal(CallbackReceiver<CampaignList> receiver, WarplyInboxRequest request) {

        WarpUtils.log("************* WARPLY Microapp ********************");
        if (!WarplyServerPreferencesManager.isMicroAppActive(WarpConstants.MicroApp.OFFERS)) {
            WarpUtils.log("[WARP Trace] Offers Microapp is not active");
            WarpUtils.log("**************************************************");
            receiver.onFailure(2);
            return;
        }
        WarpUtils.log("[WARP Trace] Offers Microapp is active");
        WarpUtils.log("**************************************************");

        request = request != null ? request : new WarplyInboxRequest();
        postReceiveMicroappDataInternal(WarpConstants.MICROAPP_CAMPAIGNS, request.toJsonInApp(),
                new CampaignsHook(receiver, request.getSignature()));
    }


    private void getInboxStatsInternal(CallbackReceiver<InboxStats> receiver) {

        WarpUtils.log("************* WARPLY Microapp ********************");
        if (!WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.OFFERS)) {
            WarpUtils.log("[WARP Trace] Offers Microapp is not active");
            WarpUtils.log("**************************************************");
            receiver.onFailure(2);
            return;
        }
        WarpUtils.log("[WARP Trace] Offers Microapp is active");
        WarpUtils.log("**************************************************");

        JSONObject jObj = new JSONObject();
        try {
            jObj.put("action", "offer_status");
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        postReceiveMicroappDataInternal(WarpConstants.MICROAPP_CAMPAIGNS, jObj,
                new InboxStatsHook(receiver));
    }

    /**
     * Method used to check if an update of app is available and show alert dialog
     *
     * @param activityContext The activity's context in order the method to be able to create alert dialog
     * @param callback        return false if user if user declined an update or update no need
     */
    public static void checkForAppUpdate(final Context activityContext,
                                         final SimpleCallbackReceiver<Boolean> callback) {

        INSTANCE.isInitializedOrThrow();
        final WarplyPreferences warplyPreferences = new WarplyPreferences(INSTANCE.mContext);
        if (warplyPreferences.getBoolean(WarpConstants.KEY_APP_VAR_UPDATE_HAS, false)
                && warplyPreferences.getInt(WarpConstants.KEY_APP_VAR_UPDATE_VERSION_CODE, 0) >
                WarplyProperty.getAppVersionCode(Warply.getWarplyContext())) {

            AlertDialog.Builder builder = new AlertDialog.Builder(activityContext);
            builder.setCancelable(false);
            builder.setTitle("Update");
            builder.setMessage("A new version of your app is available.");
            builder.setPositiveButton("Update", new DialogInterface.OnClickListener() {
                public void onClick(DialogInterface dialog, int id) {
                    callback.onResult(true, 0);
                    try {
                        activityContext.startActivity(new Intent(Intent.ACTION_VIEW,
                                Uri.parse("market://details?id="
                                        + activityContext.getApplicationContext().getPackageName())));
                    } catch (ActivityNotFoundException e) {
                        try {
                            activityContext.startActivity(new Intent(Intent.ACTION_VIEW,
                                    Uri.parse("https://play.google.com/store/apps/details?id="
                                            + activityContext.getApplicationContext().getPackageName())));
                        } catch (ActivityNotFoundException e1) {
                            if (WarpConstants.DEBUG) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
            });

            if (!warplyPreferences.getBoolean(WarpConstants.KEY_APP_VAR_UPDATE_FORCE, false)) {
                builder.setNegativeButton("Close", new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int id) {
                        dialog.dismiss();
                        callback.onResult(false, 0);
                    }
                });
            }
            builder.create().show();
        } else {
            callback.onResult(false, 0);
        }
    }


    /**
     * Add a listener to listen to registration updates
     *
     * @param receiver
     */
    public static void setRegistrationListener(
            CallbackReceiver<ServiceRegistrationCallback> receiver) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.listenGCMInternal(receiver);
    }

    private void listenGCMInternal(
            CallbackReceiver<ServiceRegistrationCallback> receiver) {
        mRegistrationListener = receiver;
    }

    /**
     * Stop listening to registration updates
     */
    public static void stopRegistrationListening() {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.stopGCMListeningInternal();
    }

    private void stopGCMListeningInternal() {
        mRegistrationListener = null;
    }

    /**
     * Method used to register with the Warply service. In order to receive push
     * notifications, you also need to register with GCM using registerGCm or
     * registerGCMAuto methods
     */
    public static void registerWarply() {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.registerWarplyInternal();
    }

    private void registerWarplyInternal() {
        isInitializedOrThrow();
        if (!WarpUtils.isRegisteredWarply(mContext))
            wakeRegistrationTask();
    }

    /**
     * Method used to unregister from the Warply service. You also need to
     * unregister from the GCM service, in order to stop receiving push
     * notifications. Better is to unregister from the GCM service first and
     * then unregister from Warply.
     */
    public static void resetWarplyWebId() {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.resetWarplyWebIdInternal();
    }

    private void resetWarplyWebIdInternal() {
        isInitializedOrThrow();
        WarpUtils.setWebId(mContext, "");
        if (mRegistrationListener != null)
            mRegistrationListener
                    .onSuccess(ServiceRegistrationCallback.UNREGISTERED_WARPLY);
        registerWarply();
    }

    /**
     * Method used to enable or disable GCM push notifications. By default, this
     * is enabled
     */
    public void setGCMEnabled(boolean enabled) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.setGCMEnabledInternal(enabled);
    }

    private void setGCMEnabledInternal(boolean enabled) {
        WarpUtils.setGCMEnabled(mContext, enabled);
    }

    /**
     * Returns the Warply INSTANCE - identical to using Warply.INSTANCE
     */
    public static Warply getWarplyInstance() {
        return INSTANCE;
    }

    /**
     * @return the context that the Warply INSTANCE has
     */
    public static Context getWarplyContext() {
        INSTANCE.isInitializedOrThrow();
        return INSTANCE.getWarplyContextInternal();
    }

    private Context getWarplyContextInternal() {
        return mContext;
    }

    /**
     * Method used to post the device info data to the server, id data have not
     * changed, nothing happens
     */
    public void postDeviceInfoData() {

        isInitializedOrThrow();

        WarpUtils.log("************* WARPLY Microapp ********************");
        if (!WarplyServerPreferencesManager.isMicroAppActive(WarpConstants.MicroApp.APPLICATION_DATA)) {
            WarpUtils.log("[WARP Trace] Application Info Microapp is not active");
            return;
        } else {
            WarpUtils.log("[WARP Trace] Application Info Microapp is active");
        }
        WarpUtils.log("**************************************************");


        final WarplyDeviceInfoCollector deviceInfoCollector = new WarplyDeviceInfoCollector(mContext);
        deviceInfoCollector.collectToJson(new SimpleCallbackReceiver<JSONObject>() {
            @Override
            public void onResult(JSONObject json, int errorCode) {
                super.onResult(json, errorCode);

                if (json != null) {
                    if (!WarpUtils.getIsDeviceInfoSaved(mContext)) {
                        WarpUtils.setIsDeviceInfoSaved(mContext, true);
                        WarpUtils.setDeviceInfoObject(mContext, json);
                        postMicroappData(WarpConstants.MICROAPP_DEVICE_INFO, json, true);
                    } else {
                        if (hasDeviceInfoDifference(WarpUtils.getDeviceInfoObject(mContext), json)) {
                            WarpUtils.setDeviceInfoObject(mContext, json);
                            postMicroappData(WarpConstants.MICROAPP_DEVICE_INFO, json, true);
                        }
                    }
                }
            }
        });
    }

    private boolean hasDeviceInfoDifference(JSONObject savedDeviceInfo, JSONObject toSendDeviceInfo) {
        if (savedDeviceInfo.length() != toSendDeviceInfo.length()) {
            return true;
        }

        for (Iterator<String> iterToSend = toSendDeviceInfo.keys(); iterToSend.hasNext(); ) {
            String keyToSend = iterToSend.next();
            for (Iterator<String> iterSaved = savedDeviceInfo.keys(); iterSaved.hasNext(); ) {
                String keySaved = iterSaved.next();
                if (keyToSend.equals(keySaved)) {
                    try {
                        if (!toSendDeviceInfo.get(keyToSend).equals(savedDeviceInfo.get(keySaved))) {
                            return true;
                        }
                    } catch (JSONException e) {
                        return false;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Method used to post the application data to the server, if data have not
     * changed, nothing happens
     */
    private void postApplicationData(boolean force) {

        isInitializedOrThrow();
        WarpUtils.log("************* WARPLY Microapp ********************");
        if (!WarplyServerPreferencesManager
                .isMicroAppActive(WarpConstants.MicroApp.APPLICATION_DATA)) {
            WarpUtils
                    .log("[WARP Trace] Application Info Microapp is not active");
            return;
        } else {
            WarpUtils.log("[WARP Trace] Application Info Microapp is active");
        }
        WarpUtils.log("**************************************************");
        JSONObject object = new JSONObject();
        try {
            PackageInfo info = mContext.getPackageManager().getPackageInfo(mContext.getApplicationContext().getPackageName(), 0);

            if (WarpUtils.getTrackersEnabled(mContext)) {
                object.putOpt("app_version", info.versionName);
                object.putOpt("sdk_version", WarpConstants.SDK_VERSION);
                object.putOpt("app_build", info.versionCode);
            }
//            object.putOpt("bundle_identifier", mContext.get().getApplicationContext().getPackageName());
            if (!WarpUtils.getHasApplicationInfo(mContext)) {
                WarpUtils.setHasApplicationInfo(mContext, true);
                WarpUtils.setAppDataObject(mContext, object);
                postMicroappData(WarpConstants.MICROAPP_APPLICATION_DATA, object, force);
            } else {
                if (hasApplicationDataDifference(WarpUtils.getAppDataObject(mContext), object)) {
                    WarpUtils.setAppDataObject(mContext, object);
                    postMicroappData(WarpConstants.MICROAPP_APPLICATION_DATA, object, force);
                }
            }
        } catch (JSONException e) {
            WarpUtils.warn("Problem when creating Device Info JSON", e);
        } catch (NameNotFoundException e) {
            WarpUtils.warn(
                    "Problem when creating Application Data JSON, in package",
                    e);
        }
    }

    private boolean hasApplicationDataDifference(JSONObject savedAppData, JSONObject toSendAppData) {
        for (Iterator<String> iterToSend = toSendAppData.keys(); iterToSend.hasNext(); ) {
            String keyToSend = iterToSend.next();
            for (Iterator<String> iterSaved = savedAppData.keys(); iterSaved.hasNext(); ) {
                String keySaved = iterSaved.next();
                if (keyToSend.equals(keySaved)) {
                    try {
                        if (!toSendAppData.get(keyToSend).equals(savedAppData.get(keySaved))) {
                            return true;
                        }
                    } catch (JSONException e) {
                        return false;
                    }
                }
            }
        }
        return false;
    }

    public static void resetApplicationInfo() {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.resetApplicationInfoInternal();
    }

    private void resetApplicationInfoInternal() {
        WarpUtils.log("************* WARPLY Info ********************");
        WarpUtils.log("[WARP Trace] Resetting application info");
        WarpUtils.log("**********************************************");
        WarpUtils.setLastApplicationData(mContext, "");
        if (WarpUtils.getIsAPPDATAENABLED(mContext))
            postApplicationData(true);
    }

    public static void resetDeviceInfo() {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.resetDeviceInfoInternal();
    }

    private void resetDeviceInfoInternal() {
        WarpUtils.setLastDeviceInfo(mContext, "");
//        postDeviceInfoData();
    }

    public static void resetIsWarped() {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.resetIsWarpedInternal();
    }

    private void resetIsWarpedInternal() {
        String lastGCMSenderId = WarpUtils.getLastGCMSenderId(mContext);
        if (lastGCMSenderId == null)
            return;
        if (lastGCMSenderId.equals(""))
            return;
    }

    public void onGCMRegistrationSuccess() {
        if (mRegistrationListener != null)
            mRegistrationListener
                    .onSuccess(ServiceRegistrationCallback.REGISTERED_GCM);
        resetDeviceInfoInternal();
    }

    public void onGCMUnregistrationSuccess() {
        if (mRegistrationListener != null)
            mRegistrationListener
                    .onSuccess(ServiceRegistrationCallback.UNREGISTERED_GCM);
    }

    public void onGCMRegistrationFail(int errorCode) {
        if (mRegistrationListener != null)
            mRegistrationListener.onFailure(errorCode);
    }

    private void check() {
        if (mContext == null) {
            return;
        }
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            registerWarply();
            return;
        }
        if (WarpUtils.getIsDEVICEINFOENABLED(mContext))
            postDeviceInfoData();
        if (WarpUtils.getIsAPPDATAENABLED(mContext))
            postApplicationData(true);
        tryWakingSendingTaskInternal(WarplyDBHelper.getInstance(mContext)
                .getRequestsInQueueCount());
    }

    private void wakeSendingTask() {

        if (!acquirePostLockInternal())
            return;

        ArrayList<Long> ids = new ArrayList<>();
        JSONArray jArray = new JSONArray();
        Cursor c = WarplyDBHelper.getInstance(mContext).getAllRequests();
        while (c.moveToNext()) {
            JSONObject jObj = new JSONObject();
            try {
                jObj.putOpt(
                        c.getString(c
                                .getColumnIndex(WarplyDBHelper.KEY_REQUESTS_MICROAPP)),
                        new JSONObject(c.getString(c
                                .getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ENTITY))));
                jArray.put(jObj);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    WarpUtils
                            .warn("[WARP Trace] Exception thrown when creating the JSON from DB with id: "
                                            + c.getLong(c
                                            .getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ID)),
                                    e);
                }
            } finally {
                ids.add(c.getLong(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ID)));
            }
        }
        c.close();
        PostHook ph = new PostHook(getWarplyContext(), ids);
        postToServerInternal(null, jArray, ph, null);
        getFromServerInternal(null, null, null, null);/** get context **/
    }

    private void wakeSendingPushTask() {

        if (!acquirePostLockInternal())
            return;

        ArrayList<Long> ids = new ArrayList<>();
        JSONArray jArray = new JSONArray();
        Cursor c = WarplyDBHelper.getInstance(mContext).getAllPushRequests();
        while (c.moveToNext()) {
            JSONObject jObj = new JSONObject();
            try {
                jObj.putOpt(
                        c.getString(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_MICROAPP)),
                        new JSONObject(c.getString(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ENTITY))));
                jArray.put(jObj);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    WarpUtils
                            .warn("[WARP Trace] Exception thrown when creating the JSON from DB with id: "
                                            + c.getLong(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ID)),
                                    e);
                }
            } finally {
                ids.add(c.getLong(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ID)));
            }
        }
        c.close();
        PostHook ph = new PostHook(getWarplyContext(), ids, true);
        postToServerInternal(null, jArray, ph, null);
    }

    private void wakeSendingPushAckTask() {

        if (!acquirePostLockInternal())
            return;

        ArrayList<Long> ids = new ArrayList<>();
        JSONArray jArray = new JSONArray();
        Cursor c = WarplyDBHelper.getInstance(mContext).getAllPushAckRequests();
        while (c.moveToNext()) {
            JSONObject jObj = new JSONObject();
            try {
                jObj.putOpt(
                        c.getString(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_MICROAPP)),
                        new JSONObject(c.getString(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ENTITY))));
                jArray.put(jObj);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    WarpUtils.warn("[WARP Trace] Exception thrown when creating the JSON from DB with id: "
                                    + c.getLong(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ID)),
                            e);
                }
            } finally {
                ids.add(c.getLong(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ID)));
            }
        }
        c.close();
        PostHook ph = new PostHook(getWarplyContext(), ids, false, true);
        postToServerInternal(null, jArray, ph, null);
    }

    /**
     * Method used to wake the post task, for sending events to the Warply
     * server. This will actually start, only if forced events are present in
     * the queue or the queue has reached the 10 items in size.
     */
    public static void tryWakingSendingTask() {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.tryWakingSendingTaskInternal(WarplyDBHelper.getInstance(
                INSTANCE.mContext).getRequestsInQueueCount());
    }

    private void tryWakingSendingTaskInternal(long requestsInQueue) {
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            WarpUtils
                    .log("************* WARPLY Registration ********************");
            WarpUtils
                    .log("[WARP TRace] Not registered yet, not waking post task!");
            WarpUtils
                    .log("******************************************************");
            return;
        }

//        if (WarplyDBHelper.getInstance(mContext.get()).isForceRequestsExist()
//                || (requestsInQueue >= MINIMUM_REQUESTS_FOR_SENDING)) {

        if (WarplyDBHelper.getInstance(mContext).isForceRequestsExist()
                || (requestsInQueue > 0)) {

            WarpUtils.log("Waking post task!");
            wakeSendingTask();
        } else {
            WarpUtils.log("Criteria not matched, not waking post task!");
        }
    }

    private void tryWakingSendingTaskInternal(Context context, long requestsInQueue) {
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            WarpUtils
                    .log("************* WARPLY Registration ********************");
            WarpUtils
                    .log("[WARP TRace] Not registered yet, not waking post task!");
            WarpUtils
                    .log("******************************************************");
            return;
        }

//        if (WarplyDBHelper.getInstance(mContext.get()).isForceRequestsExist()
//                || (requestsInQueue >= MINIMUM_REQUESTS_FOR_SENDING)) {

        if (WarplyDBHelper.getInstance(mContext).isForceRequestsExist()
                || (requestsInQueue > 0)) {

            WarpUtils.log("Waking post task!");
            wakeSendingTask(context);
        } else {
            WarpUtils.log("Criteria not matched, not waking post task!");
        }
    }

    private void wakeSendingTask(Context context) {

        if (!acquirePostLockInternal())
            return;

        ArrayList<Long> ids = new ArrayList<>();
        JSONArray jArray = new JSONArray();
        Cursor c = WarplyDBHelper.getInstance(mContext).getAllRequests();
        while (c.moveToNext()) {
            JSONObject jObj = new JSONObject();
            try {
                jObj.putOpt(
                        c.getString(c
                                .getColumnIndex(WarplyDBHelper.KEY_REQUESTS_MICROAPP)),
                        new JSONObject(c.getString(c
                                .getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ENTITY))));
                jArray.put(jObj);
            } catch (JSONException e) {
                if (WarpConstants.DEBUG) {
                    WarpUtils
                            .warn("[WARP Trace] Exception thrown when creating the JSON from DB with id: "
                                            + c.getLong(c
                                            .getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ID)),
                                    e);
                }
            } finally {
                ids.add(c.getLong(c.getColumnIndex(WarplyDBHelper.KEY_REQUESTS_ID)));
            }
        }
        c.close();
        PostHook ph = new PostHook(getWarplyContext(), ids);
        postToServerInternal(context, null, jArray, ph, null);
        getFromServerInternal(null, null, null, null);/** get context **/
    }

    private void postToServerInternal(Context context, String warplyPath, JSONArray data,
                                      CallbackReceiver<JSONObject> listener, Object tag) {
        String url = buildWarplyRequestUrl(context, warplyPath);

        JSONArray tempAnalytics = new JSONArray();
        JSONArray tempDeviceInfo = new JSONArray();
        JSONArray tempOther = new JSONArray();
        if (data.length() > 0) {
            for (int i = 0; i < data.length(); i++) {
                JSONObject microappItem = data.optJSONObject(i);
                if (microappItem != null) {
                    if (microappItem.has("inapp_analytics")) {
                        tempAnalytics.put(microappItem);
                    } else if (microappItem.has("device_info") || microappItem.has("application_data")) {
                        tempDeviceInfo.put(microappItem);
                    } else {
                        tempOther.put(microappItem);
                    }
                }
            }
        }

        if (tempAnalytics != null && tempAnalytics.length() > 0) {
            url = WarplyProperty.getBaseUrl(mContext) + WarpConstants.WARPLY_ASYNC + WarpConstants.WARPLY_ANALYTICS + WarplyProperty.getAppUuid(mContext) + "/";
        }
        if (tempDeviceInfo != null && tempDeviceInfo.length() > 0) {
            url = WarplyProperty.getBaseUrl(mContext) + WarpConstants.WARPLY_ASYNC + WarpConstants.WARPLY_DEVICE_INFO + WarplyProperty.getAppUuid(mContext) + "/";
        }
        if (tempOther != null && tempOther.length() > 0) {
            url = buildWarplyRequestUrl(warplyPath);
        }

        requestToServerInternal(Method.POST, url, data, listener, tag);
    }

    private String buildWarplyRequestUrl(Context context, String warplyPath) {
        StringBuilder sb = new StringBuilder(WarplyProperty.getBaseUrl(context) + WarpConstants.BASE_URL_MOBILE);
        sb.append(WarplyProperty.getAppUuid(mContext)).append("/")
                .append(WarpConstants.PATH_CONTEXT).append("/");
        if (warplyPath != null)
            sb.append("?path=").append(warplyPath);
        return sb.toString();
    }

    public static void tryWakingSendingPushTask(boolean force) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.tryWakingSendingPushTaskInternal(WarplyDBHelper.getInstance(
                INSTANCE.mContext).getPushRequestsInQueueCount(), force);
    }

    private void tryWakingSendingPushTaskInternal(long requestsInQueue, boolean force) {
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            WarpUtils
                    .log("************* WARPLY Registration ********************");
            WarpUtils
                    .log("[WARP TRace] Not registered yet, not waking post task!");
            WarpUtils
                    .log("******************************************************");
            return;
        }
        if (force || (requestsInQueue >= MINIMUM_REQUESTS_FOR_SENDING)) {

            WarpUtils.log("Waking post task!");
            wakeSendingPushTask();
        } else {
            WarpUtils.log("Criteria not matched, not waking post task!");
        }
    }

    public static void tryWakingSendingPushAckTask(boolean force) {
        INSTANCE.isInitializedOrThrow();
        INSTANCE.tryWakingSendingPushAckTaskInternal(WarplyDBHelper.getInstance(
                INSTANCE.mContext).getPushAckRequestsInQueueCount(), force);
    }

    private void tryWakingSendingPushAckTaskInternal(long requestsInQueue, boolean force) {
        if (!WarpUtils.isRegisteredWarply(mContext)) {
            WarpUtils
                    .log("************* WARPLY Registration ********************");
            WarpUtils
                    .log("[WARP TRace] Not registered yet, not waking post task!");
            WarpUtils
                    .log("******************************************************");
            return;
        }
        if (force || (requestsInQueue >= MINIMUM_REQUESTS_FOR_SENDING)) {

            WarpUtils.log("Waking post task!");
            wakeSendingPushAckTask();
        } else {
            WarpUtils.log("Criteria not matched, not waking post task!");
        }
    }

    private void wakeRegistrationTask() {
        if (!acquireRegisterLock())
            return;
        registerToServerInternal();
    }

    /**
     * This method starts location reporting. It is called when application
     * becomes active.
     */
    public static void onApplicationStarted() {
        INSTANCE.isInitializedOrThrow();
//        WarplyLocationManager.startDefaultReportingLocation(getWarplyContext());
        WarplyAnalyticsManager.logAppLifeCycleEvent();
    }


    public static void onApplicationEnterForeground() {
        INSTANCE.isInitializedOrThrow();
        new WarplyPreferences(getWarplyContext()).saveAppStatus("foreground");
        changeLocationSettings(true);
        WarplyServerPreferencesManager.checkServerPreferences(mServerPreferencesReceiveListener);
    }

    public static void onApplicationEnterBackground() {
        INSTANCE.isInitializedOrThrow();
        new WarplyPreferences(getWarplyContext()).saveAppStatus("background");
        changeLocationSettings(false);
    }

    private static void changeLocationSettings(boolean isForeground) {
        WarplyAnalyticsManager.logAppLifeCycleEvent(isForeground);
        if (Warply.LOCATION_SETTINGS_MAP != null) {

            String provider;

            int mode = Warply.LOCATION_SETTINGS_MAP
                    .get(isForeground ? WarpConstants.LocationSetting.LOCATION_FOREGROUND_MODE
                            : WarpConstants.LocationSetting.LOCATION_BACKGROUND_MODE);

            switch (mode) {
                case 0:
                    provider = LocationModes.OFF.getJsonKey();
                    break;
                case 1:
                    provider = LocationModes.PASSIVE.getJsonKey();
                    break;
                case 2:
                    provider = LocationModes.NETWORK.getJsonKey();
                    break;
                case 3:
                    provider = LocationModes.GPS.getJsonKey();
                    break;
                default:
                    provider = LocationModes.OFF.getJsonKey();
            }

            long minTime = Warply.LOCATION_SETTINGS_MAP
                    .get(isForeground ? WarpConstants.LocationSetting.FOREGROUND_MIN_TIME :
                            WarpConstants.LocationSetting.BACKGROUND_MIN_TIME);
            float minDistance = Warply.LOCATION_SETTINGS_MAP
                    .get(isForeground ? WarpConstants.LocationSetting.FOREGROUND_MIN_DISTANCE :
                            WarpConstants.LocationSetting.BACKGROUND_MIN_DISTANCE);

            WarplyLocationManager.regulateLocationWithNewSettings(getWarplyContext(),
                    provider, minTime * 1000, minDistance);
        }
    }

    public static void initBeaconsApplicationIfNeed() {

        try {
            WarplyBeaconsApplication beaconsApplication = WarplyBeaconsApplication.getInstance();
            if (beaconsApplication != null) {

                if (WarplyBeaconsApplication.isMicroAppEnabled()) {
                    beaconsApplication.enable();
                    beaconsApplication.addBeaconLayouts(WarplyBeaconsApplication.getBeaconLayouts());
                } else {
                    beaconsApplication.disable();
                }
            }
        } catch (NoClassDefFoundError e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Fundamental method, for doing requests to server, using JSON Object. This
     * should be kept private and used by every other method in the SDK.
     *
     * @param method   The method to use when requesting, like POST, GET, etc.
     * @param data     The JSON Object that should be sent to server as data
     * @param listener The listener, if any, to listen to request results
     * @param tag      Tag is used for massively canceling requests, should be
     *                 exported to SDK users somehow
     */
    private void requestToServerInternal(int method, String url, JSONObject data,
                                         CallbackReceiver<JSONObject> listener, Object tag) {

        VolleyTransformer vt = new VolleyTransformer(listener);
        String methodName = null;
        switch (method) {
            case 0:
                methodName = "GET";
                break;
            case 1:
                methodName = "POST";
                break;
            case 2:
                methodName = "PUT";
                break;
            case 3:
                methodName = "DELETE";
                break;
        }

        WarpUtils.log("************* WARPLY " + methodName + " Context ********************");
        WarpUtils.log("[WARP Trace] HTTP Web Id: " + WarpUtils.getWebId(mContext));
        WarpUtils.log("[WARP Trace] HTTP API Key: " + WarplyProperty.getAppUuid(mContext));
        WarpUtils.verbose("[WARP Trace] HTTP " + methodName + " Request URL: " + url);
        if (data != null) {
            try {
                WarpUtils.verbose("[WARP Trace] Request: " + data.toString(2));
            } catch (JSONException e) {
                WarpUtils.warn(
                        "[WARP Trace] Failed conversting JSON to string", e);
            }
        }
        WarpUtils.log("**********************************************************");
        WarplyJsonObjectRequest request = new WarplyJsonObjectRequest(method, url, data, vt, vt);
        request.setTag(tag);
        mRequestQueue.add(request);
    }

    public void requestToServerInternalNew(int method, String url, JSONObject data,
                                           CallbackReceiver<JSONObject> listener, Object tag) {

        VolleyTransformer vt = new VolleyTransformer(listener);
        String methodName = null;
        switch (method) {
            case 0:
                methodName = "GET";
                break;
            case 1:
                methodName = "POST";
                break;
            case 2:
                methodName = "PUT";
                break;
            case 3:
                methodName = "DELETE";
                break;
        }

        WarpUtils.log("************* WARPLY " + methodName + " Context ********************");
        WarpUtils.log("[WARP Trace] HTTP Web Id: " + WarpUtils.getWebId(getWarplyContext()));
        WarpUtils.log("[WARP Trace] HTTP API Key: " + WarplyProperty.getAppUuid(getWarplyContext()));
        WarpUtils.verbose("[WARP Trace] HTTP " + methodName + " Request URL: " + url);
        if (data != null) {
            try {
                WarpUtils.verbose("[WARP Trace] Request: " + data.toString(2));
            } catch (JSONException e) {
                WarpUtils.warn(
                        "[WARP Trace] Failed conversting JSON to string", e);
            }
        }
        WarpUtils.log("**********************************************************");
        WarplyJsonObjectRequest request = new WarplyJsonObjectRequest(method, url, data, vt, vt);
        request.setTag(tag);
        mRequestQueue.add(request);
    }

    private void requestToServerInternal(int method, String url, JSONObject data,
                                         CallbackReceiver<JSONObject> listener,
                                         Object tag, boolean hasAuthHeaders) {

        VolleyTransformer vt = new VolleyTransformer(listener);
        String methodName = null;
        switch (method) {
            case 0:
                methodName = "GET";
                break;
            case 1:
                methodName = "POST";
                break;
            case 2:
                methodName = "PUT";
                break;
            case 3:
                methodName = "DELETE";
                break;
        }

        WarpUtils.log("************* WARPLY " + methodName + " Context ********************");
        WarpUtils.log("[WARP Trace] HTTP Web Id: " + WarpUtils.getWebId(mContext));
        WarpUtils.log("[WARP Trace] HTTP API Key: " + WarplyProperty.getAppUuid(mContext));
        WarpUtils.verbose("[WARP Trace] HTTP " + methodName + " Request URL: " + url);
        if (data != null) {
            try {
                WarpUtils.verbose("[WARP Trace] Request: " + data.toString(2));
            } catch (JSONException e) {
                WarpUtils.warn("[WARP Trace] Failed conversting JSON to string", e);
            }
        }
        WarpUtils.log("**********************************************************");
        WarplyJsonObjectRequest request = new WarplyJsonObjectRequest(method, url, data, vt, vt);
        request.setTag(hasAuthHeaders ? "true" : tag);
        mRequestQueue.add(request);
    }

    /**
     * Fundamental method, for doing requests to server, using a JSON Array.
     * This should be kept private and used by every other method in the SDK.
     *
     * @param method   The method to use when requesting, like POST, GET, etc.
     * @param data     The JSON Object that should be sent to server as data
     * @param listener The listener, if any, to listen to request results
     * @param tag      Tag is used for massively canceling requests, should be
     *                 exported to SDK users somehow
     */
    private void requestToServerInternal(int method, String url, JSONArray data,
                                         CallbackReceiver<JSONObject> listener, Object tag) {

        VolleyTransformer vt = new VolleyTransformer(listener);
        String methodName = null;
        switch (method) {
            case 0:
                methodName = "GET";
                break;
            case 1:
                methodName = "POST";
                break;
            case 2:
                methodName = "PUT";
                break;
            case 3:
                methodName = "DELETE";
                break;
        }

        WarpUtils.log("************* WARPLY " + methodName + " Context ********************");
        WarpUtils.log("[WARP Trace] HTTP Web Id: " + WarpUtils.getWebId(mContext));
        WarpUtils.log("[WARP Trace] HTTP API Key: " + WarplyProperty.getAppUuid(mContext));
        WarpUtils.verbose("[WARP Trace] HTTP " + methodName + " Request URL: " + url);
        if (data != null) {
            try {
                WarpUtils.verbose("[WARP Trace] Request: " + data.toString(2));
            } catch (JSONException e) {
                WarpUtils.warn(
                        "[WARP Trace] Failed conversting JSON to string", e);
            }
        }

        /*========================== NEW, REPLACE WITH OLD IF NEED ==========================*/
        if (data != null) {
            JSONArray tempAnalytics = new JSONArray();
            JSONArray tempDeviceInfo = new JSONArray();
            JSONArray tempOther = new JSONArray();
            if (data.length() > 0) {
                for (int i = 0; i < data.length(); i++) {
                    JSONObject microappItem = data.optJSONObject(i);
                    if (microappItem != null) {
                        if (microappItem.has("inapp_analytics")) {
                            tempAnalytics.put(microappItem);
                        } else if (microappItem.has("device_info") || microappItem.has("application_data")) {
                            tempDeviceInfo.put(microappItem);
                        } else {
                            tempOther.put(microappItem);
                        }
                    }
                }
            }

            if (INSTANCE.mRequestQueue == null)
                INSTANCE.mRequestQueue = Volley.newRequestQueue(mContext);

            if (tempAnalytics != null && tempAnalytics.length() > 0) {
                String urlAnalytics = WarplyProperty.getBaseUrl(mContext) + WarpConstants.WARPLY_ASYNC + WarpConstants.WARPLY_ANALYTICS + WarplyProperty.getAppUuid(mContext) + "/";
                WarplyJsonArrayRequest requestAnalytics = new WarplyJsonArrayRequest(method, urlAnalytics, tempAnalytics, vt, vt);
                requestAnalytics.setTag(tag);
                mRequestQueue.add(requestAnalytics);
            }
            if (tempDeviceInfo != null && tempDeviceInfo.length() > 0) {
                String urlDeviceInfo = WarplyProperty.getBaseUrl(mContext) + WarpConstants.WARPLY_ASYNC + WarpConstants.WARPLY_DEVICE_INFO + WarplyProperty.getAppUuid(mContext) + "/";
                WarplyJsonArrayRequest requestDeviceInfo = new WarplyJsonArrayRequest(method, urlDeviceInfo, tempDeviceInfo, vt, vt);
                requestDeviceInfo.setTag(tag);
                mRequestQueue.add(requestDeviceInfo);
            }
            if (tempOther != null && tempOther.length() > 0) {
                WarplyJsonArrayRequest request = new WarplyJsonArrayRequest(method, url, tempOther, vt, vt);
                request.setTag(tag);
                mRequestQueue.add(request);
            }
        }
        /*========================== NEW, REPLACE WITH OLD IF NEED ==========================*/

        /*========================== OLD, REVERT IF NEED ==========================*/
//        WarplyJsonArrayRequest request = new WarplyJsonArrayRequest(method, url, data, vt, vt);
//        request.setTag(tag);
//        mRequestQueue.add(request);
        /*========================== OLD, REVERT IF NEED ==========================*/
    }

    private void postToServerInternal(JSONObject data,
                                      CallbackReceiver<JSONObject> listener, Object tag) {
        String url = buildWarplyRequestUrl(null);
        requestToServerInternal(Method.POST, url, data, listener, tag);
    }

    private void postToServerInternal(boolean hasAuthHeaders, String path, JSONObject data,
                                      CallbackReceiver<JSONObject> listener, Object tag) {
        String url = "";
        if (path.equals("register") || path.equals("change_password") || path.equals("password_reset")
                || path.equals("generate"))
            url = buildWarplyAuthRequestUrl(path, true);
        else if (path.equals("campaigns"))
            url = buildWarplyRequestUrl(null);
        else if (path.equals("campaignsPersonalized"))
            url = buildWarplyRequestUrl(null, true);
        else
            url = buildWarplyAuthRequestUrl(path, false);
        requestToServerInternal(Method.POST, url, data, listener, tag, hasAuthHeaders);
    }

    private void postToServerInternal(String warplyPath, JSONArray data,
                                      CallbackReceiver<JSONObject> listener, Object tag) {
        String url = buildWarplyRequestUrl(warplyPath);

        JSONArray tempAnalytics = new JSONArray();
        JSONArray tempDeviceInfo = new JSONArray();
        JSONArray tempOther = new JSONArray();
        if (data.length() > 0) {
            for (int i = 0; i < data.length(); i++) {
                JSONObject microappItem = data.optJSONObject(i);
                if (microappItem != null) {
                    if (microappItem.has("inapp_analytics")) {
                        tempAnalytics.put(microappItem);
                    } else if (microappItem.has("device_info") || microappItem.has("application_data")) {
                        tempDeviceInfo.put(microappItem);
                    } else {
                        tempOther.put(microappItem);
                    }
                }
            }
        }

        if (tempAnalytics != null && tempAnalytics.length() > 0) {
            url = WarplyProperty.getBaseUrl(mContext) + WarpConstants.WARPLY_ASYNC + WarpConstants.WARPLY_ANALYTICS + WarplyProperty.getAppUuid(mContext) + "/";
        }
        if (tempDeviceInfo != null && tempDeviceInfo.length() > 0) {
            url = WarplyProperty.getBaseUrl(mContext) + WarpConstants.WARPLY_ASYNC + WarpConstants.WARPLY_DEVICE_INFO + WarplyProperty.getAppUuid(mContext) + "/";
        }
        if (tempOther != null && tempOther.length() > 0) {
            url = buildWarplyRequestUrl(warplyPath);
        }

        requestToServerInternal(Method.POST, url, data, listener, tag);
    }

    private void getFromServerInternal(JSONObject data, String warplyPath,
                                       CallbackReceiver<JSONObject> listener, Object tag) {
        String url = buildWarplyRequestUrl(warplyPath);
        if (WarpUtils.getCurrentTimeMillis(mContext) == 0 || (WarpUtils.getCurrentTimeMillis(mContext) > 0) && (System.currentTimeMillis() > WarpUtils.getCurrentTimeMillis(mContext) + (3600000))) {
            WarpUtils.setCurrentTimeMillis(mContext, System.currentTimeMillis());
            requestToServerInternal(Method.GET, url, data, listener, tag);
        }
    }

    public void getSingleCampaign(JSONObject data, String url,
                                  CallbackReceiver<JSONObject> listener, Object tag) {
        requestToServerInternalNew(Method.GET, url, data, listener, tag);
    }

    private void registerToServerInternal() {
        final String url = WarplyProperty.getBaseUrl(mContext) + WarpConstants.BASE_URL_MOBILE + WarplyProperty.getAppUuid(mContext)
                + "/" + WarpConstants.PATH_REGISTER + "/";

        new WarplyDeviceInfoCollector(mContext).getRegistrationParams(new SimpleCallbackReceiver<JSONObject>() {
            @Override
            public void onResult(JSONObject result, int errorCode) {
                super.onResult(result, errorCode);
                requestToServerInternal(Method.POST, url, result, mRegistrationCallBackReceiver, null);
            }
        });
    }

    private String buildWarplyRequestUrl(String warplyPath) {
        StringBuilder sb = new StringBuilder(WarplyProperty.getBaseUrl(mContext) + WarpConstants.BASE_URL_MOBILE);
        sb.append(WarplyProperty.getAppUuid(mContext)).append("/")
                .append(WarpConstants.PATH_CONTEXT).append("/");
        if (warplyPath != null)
            sb.append("?path=").append(warplyPath);
        return sb.toString();
    }

    private String buildWarplyRequestUrl(String warplyPath, boolean personalized) {
        StringBuilder sb = new StringBuilder(WarplyProperty.getBaseUrl(mContext) + WarpConstants.BASE_URL_AUTH);
        sb.append(WarplyProperty.getAppUuid(mContext)).append("/")
                .append(WarpConstants.PATH_CONTEXT);
        if (warplyPath != null)
            sb.append("?path=").append(warplyPath);
        return sb.toString();
    }

    private String buildWarplyAuthRequestUrl(String warplyPath, boolean isUserPath) {
        StringBuilder sb;
        if (isUserPath)
            sb = new StringBuilder(WarplyProperty.getBaseUrl(mContext) + WarpConstants.BASE_URL_USER_AUTH);
        else {
            if (warplyPath.equals("handle_image"))
                sb = new StringBuilder(WarplyProperty.getBaseUrl(mContext) + WarpConstants.BASE_URL_API);
            else if (warplyPath.equals("verify"))
                sb = new StringBuilder(WarplyProperty.getBaseUrl(mContext) + WarplyProperty.getVerifyUrl(mContext));
            else if (warplyPath.equals("cosuser"))
                sb = new StringBuilder(WarplyProperty.getBaseUrl(mContext) + "/partners/oauth/" + WarplyProperty.getAppUuid(mContext) + "/token");
            else
                sb = new StringBuilder(WarplyProperty.getBaseUrl(mContext) + WarpConstants.BASE_URL_AUTH);
        }

        if (warplyPath != null) {
            if (!warplyPath.equals("verify")) {
                if (warplyPath.equals("cosuser")) {
                    return sb.toString();
                }
                sb.append(WarplyProperty.getAppUuid(mContext)).append("/");
                if (warplyPath.equals("generate")) {
                    sb.append(WarpConstants.BASE_URL_OTP);
                }
                sb.append(warplyPath);
            }
        }
        return sb.toString();
    }

    public static void showInAppCampaign(Context context, String eventId) {
        String tempOfferCategoryName = eventId.split(":")[1];
        CampaignList tempCampaignList = new CampaignList();

        if (INSTANCE.getInAppCampaigns() != null && INSTANCE.getInAppCampaigns().size() > 0) {
            for (Campaign campaign : INSTANCE.getInAppCampaigns()) {
                if ((campaign.getOfferCategory().equals(IN_APP_FILTER_ALL) || campaign.getOfferCategory().equalsIgnoreCase(tempOfferCategoryName))
                        && campaign.isShow()) {
                    tempCampaignList.add(campaign);
                }
            }

            Collections.sort(tempCampaignList, new Comparator<Campaign>() {
                @Override
                public int compare(Campaign o1, Campaign o2) {
                    return Integer.valueOf(o2.getSorting()).compareTo(o1.getSorting());
                }
            });

            Log.v("SORTED ", "SUCCESS");

            if (tempCampaignList.size() > 0) {
                Campaign campaignToShow = tempCampaignList.get(1);

                if (campaignToShow.getActions() == null) {
                    if (campaignToShow.getAction() == 0) {
                        InAppDialog.showDefaultInAppDialog(context, campaignToShow, true, 0);
                    } else if (campaignToShow.getAction() == 1) {
                        InAppDialog.showDefaultInAppDialog(context, campaignToShow, false, 0);
                    }
                } else {
                    // TODO: handle custom buttons except Rate and Share
                    // TODO: remove cardView from gradle
                }
            }
        }
    }

    // callback receivers
    private CallbackReceiver<JSONObject> mRegistrationCallBackReceiver = new CallbackReceiver<JSONObject>() {
        @Override
        public void onSuccess(JSONObject result) {
            result = result.optJSONObject("context");
            if (result != null) {
                WarpUtils.setLastDeviceInfo(mContext, "");
                WarpUtils.setLastApplicationData(mContext, "");
                WarpUtils.setLastApplicationUUID(mContext, WarplyProperty.getAppUuid(mContext));
                WarplyDBHelper.getInstance(mContext).deleteAllRequests();
                WarplyDBHelper.getInstance(mContext).deleteAllPushRequests();
                WarplyDBHelper.getInstance(mContext).deleteAllPushAckRequests();
                String webId = result.optString("web_id", "");
                WarpUtils.setWebId(mContext, webId);
                String apiKey = result.optString("api_key", "");
                WarpUtils.setApiKey(mContext, apiKey);
                WarpUtils.log("************* WARPLY Registration ********************");
                WarpUtils.log("[WARPLY Registration] URL: " + WarplyProperty.getBaseUrl(mContext));
                WarpUtils.log("[WARPLY Registration] WEB_ID: " + webId);
                try {
                    WarpUtils.log("[WARPLY Registration] Response: " + result.toString(2));
                } catch (JSONException e) {
                    WarpUtils.warn("[WARP Trace] Failed converting JSON to string", e);
                }
                WarpUtils.log("******************************************************");
                if (mRegistrationListener != null)
                    mRegistrationListener.onSuccess(ServiceRegistrationCallback.REGISTERED_WARPLY);
//                postDeviceInfoData();
                postApplicationData(true);
            }
            releaseRegisterLock();
        }

        @Override
        public void onFailure(int errorCode) {
            WarpUtils.log("************* WARPLY Registration ********************");
            WarpUtils.log("[WARP Registration] Error: " + String.valueOf(errorCode));
            WarpUtils.log("******************************************************");
            if (mRegistrationListener != null)
                mRegistrationListener.onFailure(errorCode);

            if (errorCode == 3) {
                final String url = WarplyProperty.getBaseUrl(mContext) + WarpConstants.BASE_URL_MOBILE + WarplyProperty.getAppUuid(mContext)
                        + "/" + WarpConstants.PATH_REGISTER + "/";
                requestToServerInternal(Method.GET, url, (JSONObject) null, mRegistrationCallBackReceiver, null);
            } else
                releaseRegisterLock();
        }
    };

    private static WarplyServerPreferencesManager.ServerPreferencesReceivedListener mServerPreferencesReceiveListener =
            new WarplyServerPreferencesManager.ServerPreferencesReceivedListener() {
                @Override
                public void onServerPreferencesReceived() {
//                    changeLocationSettings(true);
                    initBeaconsApplicationIfNeed();
                    WarplyUserManager.rewriteTags();
                }
            };

}