package ly.warp.sdk.activities;

import android.Manifest;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AlertDialog;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.FragmentActivity;

import com.bumptech.glide.Glide;
import com.bumptech.glide.load.engine.DiskCacheStrategy;
import com.google.android.gms.maps.CameraUpdate;
import com.google.android.gms.maps.CameraUpdateFactory;
import com.google.android.gms.maps.GoogleMap;
import com.google.android.gms.maps.OnMapReadyCallback;
import com.google.android.gms.maps.SupportMapFragment;
import com.google.android.gms.maps.model.LatLng;
import com.google.android.gms.maps.model.Marker;
import com.google.android.gms.maps.model.MarkerOptions;
import com.google.android.material.bottomsheet.BottomSheetDialog;
import com.google.maps.android.clustering.Cluster;
import com.google.maps.android.clustering.ClusterManager;

import java.util.ArrayList;

import io.github.inflationx.viewpump.ViewPumpContextWrapper;
import ly.warp.sdk.R;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.models.Couponset;
import ly.warp.sdk.io.models.Merchant;
import ly.warp.sdk.io.models.MerchantList;
import ly.warp.sdk.io.request.WarplyMerchantsRequest;
import ly.warp.sdk.utils.WarplyManagerHelper;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.utils.managers.WarplyAnalyticsManager;
import ly.warp.sdk.utils.managers.WarplyManager;


public class ShopsActivity extends FragmentActivity implements View.OnClickListener,
        OnMapReadyCallback, GoogleMap.OnMapLoadedCallback, ClusterManager.OnClusterClickListener<Merchant>,
        GoogleMap.OnMarkerClickListener {

    // ===========================================================
    // Constants
    // ===========================================================

    public final static float DEFAULT_LOCATION_ZOOM = 6.2f;

    // ===========================================================
    // Fields
    // ===========================================================

    private ImageView mIvBack;
    private Couponset mCouponset;
    private GoogleMap mMap;
    private ClusterManager<Merchant> mClusterManager;
    private SupportMapFragment mMapView;
    private MerchantList mMerchantList = new MerchantList();
    private AlertDialog mAlertDialogNoShopsAvailable;
    private Merchant mMerchant;

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_shops);

        mCouponset = (Couponset) getIntent().getSerializableExtra("couponset");
        if (mCouponset != null) {
            for (Merchant merchant : WarplyManagerHelper.getMerchantList()) {
                if (merchant.getUuid().equals(mCouponset.getMerchantUuid())) {
                    mMerchant = merchant;
                    break;
                }
            }
        }

        mIvBack = findViewById(R.id.iv_shops_back);
        mMapView = (SupportMapFragment) getSupportFragmentManager().findFragmentById(R.id.mv_shops);
        mMapView.getMapAsync(this);
        mIvBack.setOnClickListener(this);
    }

    @Override
    public void onResume() {
        super.onResume();
        WarplyAnalyticsManager.logTrackersEvent(this, "screen", "ShopsScreen");
    }

    @Override
    public void onClick(View view) {
        if (view.getId() == R.id.iv_shops_back) {
            onBackPressed();
        }
    }

    @Override
    protected void attachBaseContext(Context newBase) {
        super.attachBaseContext(ViewPumpContextWrapper.wrap(newBase));
    }

//    @Override
//    public void onRequestPermissionsResult(
//            int requestCode,
//            String permissions[],
//            int[] grantResults) {
//        switch (requestCode) {
//            case REQUEST_PERMISSION_PHONE_STATE:
//                if (grantResults.length > 0
//                        && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
//                    Toast.makeText(MainActivity.this, "Permission Granted!", Toast.LENGTH_SHORT).show();
//                } else {
//                    Toast.makeText(MainActivity.this, "Permission Denied!", Toast.LENGTH_SHORT).show();
//                }
//        }
//    }

    @Override
    public void onMapReady(@NonNull GoogleMap googleMap) {
        mMap = googleMap;

        if (ActivityCompat.checkSelfPermission(this,
                Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED
                && ActivityCompat.checkSelfPermission(this,
                Manifest.permission.ACCESS_COARSE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
            if (ContextCompat.checkSelfPermission(this, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
                mMap.setMyLocationEnabled(true);
            }
        } else {
            mMap.setMyLocationEnabled(false);
//            ActivityCompat.requestPermissions(this,
//                    new String[]{Manifest.permission.ACCESS_FINE_LOCATION,
//                            Manifest.permission.ACCESS_COARSE_LOCATION}, 501);
        }

        mMap.setMapType(GoogleMap.MAP_TYPE_NORMAL);
        mMap.setMinZoomPreference(DEFAULT_LOCATION_ZOOM);
        mMap.setLatLngBoundsForCameraTarget(WarpConstants.GREECE_BOUNDS);
        mMap.getUiSettings().setZoomControlsEnabled(true);
        mMap.getUiSettings().setZoomGesturesEnabled(true);
        mMap.getUiSettings().setCompassEnabled(false);
        mMap.getUiSettings().setMyLocationButtonEnabled(false);
        mMap.getUiSettings().setMapToolbarEnabled(false);
        mMap.getUiSettings().setTiltGesturesEnabled(false);
        mMap.getUiSettings().setRotateGesturesEnabled(false);
        mMap.setOnMapLoadedCallback(this);
        mMap.setOnMarkerClickListener(/*mClusterManager*/this);
    }

    @Override
    public void onMapLoaded() {
        if (mMap != null) {
            CameraUpdate camUpdate;
            camUpdate = CameraUpdateFactory.newLatLngBounds(WarpConstants.GREECE_BOUNDS, 48);
            mMap.animateCamera(camUpdate);
        }

//        if (mClusterManager == null) {
//            mClusterManager = new ClusterManager<>(this, mMap);
//            mClusterManager.setOnClusterClickListener(this);
////            mMap.setOnMarkerClickListener(/*mClusterManager*/this);
//        }

        if (mCouponset != null) {
            ArrayList<String> uuids = new ArrayList<>();
            uuids.add(mCouponset.getMerchantUuid());
            new Thread(() -> {
                if (!Thread.currentThread().isInterrupted()) {
                    WarplyManager.getMerchantsMultilingual(new WarplyMerchantsRequest()
                                    .setIsMultilingual(true)
                                    .setParentUuids(uuids)
                            , mMerchantsCallback);
                }
                Thread.currentThread().interrupt();
            }).start();
        } else {
            ArrayList<String> catuuids = new ArrayList<>();
            catuuids.add(WarplyProperty.getAppUuid(this).equals("0086a2088301440792091b9f814c2267") ? "" : "adcace6cab6049c7b7271bc85bc2b26d"); //TODO: add live cat uuid
            new Thread(() -> {
                if (!Thread.currentThread().isInterrupted()) {
                    WarplyManager.getMerchantsMultilingual(new WarplyMerchantsRequest()
                                    .setIsMultilingual(true)
                                    .setCategories(catuuids)
                            , mMerchantsCallback);
                }
                Thread.currentThread().interrupt();
            }).start();
        }
    }

    @Override
    public boolean onClusterClick(Cluster<Merchant> cluster) {
        if (mMap != null) {
            mMap.animateCamera(CameraUpdateFactory.newLatLngZoom(cluster.getPosition(),
                    (float) Math.floor(mMap.getCameraPosition().zoom + 1)), 300, null);

//            mMap.clear();
//            mClusterManager.clearItems();
//            mClusterManager.addItems(cluster.getItems());
//            mClusterManager.cluster();
        }
        return true;
    }

    @Override
    public boolean onMarkerClick(@NonNull Marker marker) {
        for (Merchant merch : mMerchantList) {
            if (merch.getUuid().equals(marker.getSnippet())) {
                WarplyAnalyticsManager.logTrackersEvent(this, "click", ("ShopsScreenMarker").concat(":").concat(marker.getSnippet()));
                final BottomSheetDialog bottomSheetDialog = new BottomSheetDialog(this);
                bottomSheetDialog.setContentView(R.layout.dl_map_pin);

                ImageView dialogClose = (ImageView) bottomSheetDialog.findViewById(R.id.iv_map_pin_close);
                dialogClose.setOnClickListener(view -> bottomSheetDialog.dismiss());

                TextView pinTitle = (TextView) bottomSheetDialog.findViewById(R.id.tv_pin_title);
                pinTitle.setText(merch.getAdminName());

                ImageView pinLogo = (ImageView) bottomSheetDialog.findViewById(R.id.iv_pin_logo);
                if (!TextUtils.isEmpty(merch.getImgPreview())) {
                    Glide.with(this)
//                        .setDefaultRequestOptions(
//                                RequestOptions
//                                        .placeholderOf(R.drawable.ic_default_contact_photo)
//                                        .error(R.drawable.ic_default_contact_photo))
                            .load(merch.getImgPreview())
                            .diskCacheStrategy(DiskCacheStrategy.DATA)
                            .into(pinLogo);
                }

                TextView pinName = (TextView) bottomSheetDialog.findViewById(R.id.tv_pin_name);
                pinName.setText(merch.getName());

                TextView pinDays = (TextView) bottomSheetDialog.findViewById(R.id.tv_pin_days);
                pinDays.setText(merch.getSnippet()); //TODO: wrong getter

                TextView pinTel = (TextView) bottomSheetDialog.findViewById(R.id.tv_pin_tel);
                pinTel.setText(merch.getTelephone());

                TextView pinAddress = (TextView) bottomSheetDialog.findViewById(R.id.tv_pin_address);
                pinAddress.setText(merch.getAddress());

                LinearLayout pinDirections = (LinearLayout) bottomSheetDialog.findViewById(R.id.ll_directions);
                pinDirections.setOnClickListener(view -> {
                    Uri gmmIntentUri = Uri.parse("google.navigation:q=" + merch.getLatitude() + "," + merch.getLongitude()/* + "&mode=w"*/);
                    Intent mapIntent = new Intent(Intent.ACTION_VIEW, gmmIntentUri);
                    mapIntent.setPackage("com.google.android.apps.maps");
                    startActivity(mapIntent);
                });

                bottomSheetDialog.show();

                return false;
            }
        }

        return false;
    }

    // ===========================================================
    // Methods
    // ===========================================================

    private void showNoShopsAvailableDialog() {
        if (!isFinishing()) {
            mAlertDialogNoShopsAvailable = new AlertDialog.Builder(this)
                    .setTitle(R.string.cos_dlg_no_shops_title)
                    .setMessage(R.string.cos_dlg_no_shops)
                    .setCancelable(false)
                    .setPositiveButton(R.string.cos_dlg_no_shops_positive, (dialogPositive, whichPositive) -> {
                        dialogPositive.dismiss();
                        openWebsite();

                    })
                    .setNegativeButton(R.string.cos_dlg_negative_button2, (dialogNegative, whichNegative) -> {
                        dialogNegative.dismiss();
                        onBackPressed();
                    })
                    .show();
        }
    }

    private void openWebsite() {
        WarplyAnalyticsManager.logTrackersEvent(this, "click", "SeeShopWebsite");

        if (mMerchant != null && !TextUtils.isEmpty(mMerchant.getWebsite())) {
            try {
                Intent intent = new Intent(Intent.ACTION_VIEW);
                intent.setData(Uri.parse(mMerchant.getWebsite()));
                startActivity(intent);
            } catch (ActivityNotFoundException e) {
                e.printStackTrace();
            }
        }
    }

    private final CallbackReceiver<MerchantList> mMerchantsCallback = new CallbackReceiver<MerchantList>() {
        @Override
        public void onSuccess(MerchantList result) {
            mMerchantList.clear();
            mMerchantList.addAll(result);

            new Handler(Looper.getMainLooper()).post(() -> {
//                if (mClusterManager != null) {
//                    mMap.clear();
//                    mClusterManager.clearItems();
//                    if (result != null) {
//                        mClusterManager.addItems(result);
//                    }
//                    mClusterManager.cluster();
//                }

                int nonNullCoords = 0;
                for (Merchant merchCoords : result) {
                    if (merchCoords.getLatitude() != 0.0 && merchCoords.getLongitude() != 0.0) {
                        nonNullCoords++;
                    }
                }

                if (result.size() == 0 || nonNullCoords == 0) {
                    showNoShopsAvailableDialog();
                    return;
                }

                for (Merchant merch : result) {
                    mMap.addMarker(new MarkerOptions()
                            .position(new LatLng(merch.getLatitude(), merch.getLongitude()))
                            .anchor(0.5f, 0.5f)
                            .title(merch.getTitle())
                            .snippet(merch.getUuid()));
                }
            });
        }

        @Override
        public void onFailure(int errorCode) {
            Log.v("SHOPS_ACTIVITY", String.valueOf(errorCode));
        }
    };

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

}
