package ly.warp.sdk.activities;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.Environment;
import android.os.Handler;
import android.text.TextUtils;
import android.view.MotionEvent;
import android.view.View;
import android.view.WindowManager;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import com.google.android.material.snackbar.Snackbar;

import org.jetbrains.annotations.NotNull;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;

import io.github.inflationx.viewpump.ViewPumpContextWrapper;
import ly.warp.sdk.R;

/**
 * Created by Panagiotis Triantafyllou on 26/June/2023.
 */
public class TelematicsActivity extends Activity implements View.OnClickListener,
        SensorEventListener, LocationListener {

    // ===========================================================
    // Constants
    // ===========================================================

    // ===========================================================
    // Fields
    // ===========================================================

    private ImageView mIvBack;
    private boolean mIsTripStarted = false;
    private LinearLayout mLlTripButton, mLlTelematicsMain;
    private TextView mTvTripButton, mTvSensorData, mTvVelocity, mTvAvgVelocity, mTvRecordsSaved,
            mTvOrientationCount, mTvTouchCount;
    private SensorManager mSensorManager;
    private Sensor mSensor;
    private Handler mHandler, mLocationHandler, mTouchHandler;
    private Runnable mRunnable, mLocationRunnable, mTouchRunnable;
    private long lastUpdate = 0;
    private float lastX, lastY, lastZ;
    private float velocity = 0;
    private ArrayList<JSONObject> mAccelerationTimestamps = new ArrayList<>();
    private float mAcceleration = 0;
    private final float ALPHA = 0.8f; // Filter factor
    private final float STOP_THRESHOLD = 8.0f; // Stop threshold in m/s²
    private final int PERMISSION_REQUEST_WRITE_EXTERNAL_STORAGE = 4000;
    private final int PERMISSION_REQUEST_ACCESS_FINE_LOCATION = 4001;
    private JSONArray jsonArray = new JSONArray();
    private LocationManager locationManager;
    private final int LOCATION_UPDATE_INTERVAL = 1000;
    private double mLatitude = 0;
    private double mLongitude = 0;
    // Radius of the Earth in meters
    private final double EARTH_RADIUS = 6371000.0;
    private Location previousLocation;
    private double mSpeed = 0;
    private int orientationCount = 0, touchCount = 0;
    final long REFRESH_TIME = 100; // miliseconds
    private String mStartTimestamp = "", mStopTimestamp = "";
    private final int RECORDS_INTERVAL = 5000;
    private EditText mEtLimit, mEtSampleTime;
    private RelativeLayout mRlMainScroll;


    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public void onCreate(Bundle savedInstanceState) {
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_telematics);

        mIvBack = findViewById(R.id.iv_telematics_close);
        mLlTripButton = findViewById(R.id.ll_activate_button);
        mLlTripButton.setOnClickListener(this);
        mLlTripButton.setOnTouchListener(mTripTouchListener);
        mTvTripButton = findViewById(R.id.tv_trip_button);
        mLlTelematicsMain = findViewById(R.id.ll_telematics_main);
        mTvSensorData = findViewById(R.id.tv_sensor_data);
        mTvVelocity = findViewById(R.id.tv_velocity);
        mTvAvgVelocity = findViewById(R.id.tv_avg);
        mTvRecordsSaved = findViewById(R.id.tv_records);
        mIvBack.setOnClickListener(this);
        mTvOrientationCount = findViewById(R.id.tv_orientation);
        mTvTouchCount = findViewById(R.id.tv_touch);
        mEtLimit = findViewById(R.id.et_acceleration);
        mEtSampleTime = findViewById(R.id.et_save);
        mRlMainScroll = findViewById(R.id.rl_main_scroll);
        mRlMainScroll.setOnTouchListener(mScrollTouchListener);

        locationManager = (LocationManager) getSystemService(Context.LOCATION_SERVICE);
        previousLocation = null;

        initViews();
    }

    @Override
    public void onResume() {
        super.onResume();
//        WarplyAnalyticsManager.logTrackersEvent(this, "screen", "TelematicsActivity");
    }

    @Override
    protected void onPause() {
        super.onPause();
    }

    @Override
    public void onBackPressed() {
        super.onBackPressed();
        if (mIsTripStarted) {
            unregisterSensor();
            stopLocationUpdates();
        }
    }

    @Override
    public void onSensorChanged(SensorEvent sensorEvent) {
        mTvSensorData.setText(Arrays.toString(sensorEvent.values));

        float currentX = sensorEvent.values[0];
        float currentY = sensorEvent.values[1];
        float currentZ = sensorEvent.values[2];

        long currentTime = System.currentTimeMillis();
        long timeDifference = currentTime - lastUpdate;
        lastUpdate = currentTime;

        float time = timeDifference / 1000.0f; // Convert time difference to seconds

        // Apply low-pass filter
        float filteredX = ALPHA * lastX + (1 - ALPHA) * currentX;
        float filteredY = ALPHA * lastY + (1 - ALPHA) * currentY;
        float filteredZ = ALPHA * lastZ + (1 - ALPHA) * currentZ;

        // Calculate acceleration in m/s² using filtered values
        float accelerationX = (filteredX - lastX) / time;
        float accelerationY = (filteredY - lastY) / time;
        float accelerationZ = (filteredZ - lastZ) / time;

        // Calculate total acceleration
        float acceleration = (float) Math.sqrt(accelerationX * accelerationX + accelerationY * accelerationY + accelerationZ * accelerationZ);

        // If acceleration is below the stop threshold, assume we are in a stop
        if (acceleration < STOP_THRESHOLD) {
            velocity = 0;
        } else {
            // Update velocity
            velocity = acceleration * time;
        }
        // Convert velocity to km/h
//        mAcceleration = velocity * 3.6f; // Convert to km/h
        mAcceleration = velocity; // Convert to km/h
        mTvVelocity.setText(String.format("%.1f", velocity) + " m/s^2");
        // Update last values
        lastX = filteredX;
        lastY = filteredY;
        lastZ = filteredZ;
    }

    @Override
    public void onAccuracyChanged(Sensor sensor, int i) {

    }

    @Override
    public void onClick(View view) {
        if (view.getId() == R.id.iv_telematics_close) {
            onBackPressed();
            return;
        }
        if (view.getId() == R.id.ll_activate_button) {
            if (mIsTripStarted) {
                mEtLimit.setEnabled(true);
                mEtSampleTime.setEnabled(true);
                unregisterSensor();
                stopLocationUpdates();
                initViews();
                mIsTripStarted = false;
                mTvTripButton.setText(R.string.cos_dlg_start_trip);
            } else {
                if (mEtLimit.getText().length() == 0) {
                    Snackbar.make(mLlTelematicsMain, "Please fill the Cut off field", Snackbar.LENGTH_SHORT).show();
                    return;
                }
                mEtLimit.setEnabled(false);
                mEtSampleTime.setEnabled(false);
                mTouchHandler = new Handler();
                mTouchRunnable = new Runnable() {
                    @Override
                    public void run() {
                        mTouchHandler.postDelayed(this, REFRESH_TIME);
                    }
                };
                requestLocationUpdates();
                registerSensor();
                mIsTripStarted = true;
                mTvTripButton.setText(R.string.cos_dlg_stop_trip);
            }
        }
    }

    @Override
    protected void attachBaseContext(Context newBase) {
        super.attachBaseContext(ViewPumpContextWrapper.wrap(newBase));
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);

        if (requestCode == PERMISSION_REQUEST_WRITE_EXTERNAL_STORAGE) {
            if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                saveAccelerationDataToExternalStorage(jsonArray);
            } else {
                Snackbar.make(mLlTelematicsMain, "Storage Permission Denied", Snackbar.LENGTH_SHORT).show();
            }
            return;
        }
        if (requestCode == PERMISSION_REQUEST_ACCESS_FINE_LOCATION) {
            if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                startLocationUpdates();
            } else {
                Snackbar.make(mLlTelematicsMain, "Location Permission Denied", Snackbar.LENGTH_SHORT).show();
            }
        }
    }

    @Override
    public void onLocationChanged(Location location) {
        //TODO: comment the first block and uncomment the second block if needs revert to handler implementation
        if (mLatitude != 0 && mLongitude != 0) {
            mSpeed = calculateSpeed(mLatitude, mLongitude, location.getLatitude(), location.getLongitude(), (LOCATION_UPDATE_INTERVAL / 1000));
            mTvAvgVelocity.setText(String.format("%.1f", Math.floor(mSpeed)) + " km/h");
        }


//        if (mLatitude != 0 && mLongitude != 0)
//            requestLocationUpdatePeriodically(location);
        mLatitude = location.getLatitude();
        mLongitude = location.getLongitude();
    }

    @Override
    public void onStatusChanged(String provider, int status, Bundle extras) {
        // Handle status changes if needed
    }

    @Override
    public void onProviderEnabled(String provider) {
        // Handle provider enabled if needed
    }

    @Override
    public void onProviderDisabled(String provider) {
        // Handle provider disabled if needed
    }

    @Override
    public void onConfigurationChanged(@NotNull Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        if (mIsTripStarted) {
            if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE || newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) {
                orientationCount++;
                mTvOrientationCount.setText(String.valueOf(orientationCount));
            }
        }
    }

//    @Override
//    public boolean dispatchTouchEvent(MotionEvent event) {
//        if (mIsTripStarted) {
//            switch (event.getAction()) {
//                case MotionEvent.ACTION_DOWN:
//                    touchCount++;
//                    mTvTouchCount.setText(String.valueOf(touchCount));
//                    mTouchHandler.post(mTouchRunnable);
//                    return true;
//                case MotionEvent.ACTION_UP:
//                    mTouchHandler.removeCallbacks(mTouchRunnable);
//                    return true;
//            }
//        }
//        return super.dispatchTouchEvent(event);
//    }

//    @Override
//    public boolean onTouchEvent(MotionEvent event) {
////        return super.onTouchEvent(event);
//        if (mIsTripStarted) {
////            touchCount++;
////            mTvTouchCount.setText(String.valueOf(touchCount));
////            return true;
//
//            switch (event.getAction()) {
//                case MotionEvent.ACTION_DOWN:
//                    touchCount++;
//                    mTvTouchCount.setText(String.valueOf(touchCount));
//                    mTouchHandler.post(mTouchRunnable);
//                    return true;
//                case MotionEvent.ACTION_UP:
//                    mTouchHandler.removeCallbacks(mTouchRunnable);
//                    return true;
//            }
//            return false;
//        }
//        return false;
//    }

    // ===========================================================
    // Methods
    // ===========================================================

    private void initViews() {
        mTvVelocity.setText("0.0 m/s^2");
        mTvAvgVelocity.setText("0.0 km/h");
    }

    private void requestLocationUpdates() {
        if (ContextCompat.checkSelfPermission(this, Manifest.permission.ACCESS_FINE_LOCATION)
                != PackageManager.PERMISSION_GRANTED) {
            ActivityCompat.requestPermissions(this,
                    new String[]{Manifest.permission.ACCESS_FINE_LOCATION},
                    PERMISSION_REQUEST_ACCESS_FINE_LOCATION);
        } else {
            startLocationUpdates();
        }
    }

    private void startLocationUpdates() {
        try {
            locationManager.requestLocationUpdates(LocationManager.GPS_PROVIDER,
                    LOCATION_UPDATE_INTERVAL, // minimum time interval between location updates (in milliseconds)
                    0, // minimum distance between location updates (in meters)
                    this);
        } catch (SecurityException e) {
            e.printStackTrace();
        }
    }

    private void requestLocationUpdatePeriodically(Location location) {
        mLocationHandler = new Handler();
        mLocationRunnable = new Runnable() {
            @Override
            public void run() {
                double speed = calculateSpeed(mLatitude, mLongitude, location.getLatitude(), location.getLongitude(), (LOCATION_UPDATE_INTERVAL / 1000));
                mTvAvgVelocity.setText(String.format("%.1f", Math.floor(speed)) + " km/h");
                mLocationHandler.postDelayed(this, LOCATION_UPDATE_INTERVAL);
            }
        };
        mLocationHandler.postDelayed(mLocationRunnable, LOCATION_UPDATE_INTERVAL);
    }

    private double calculateDistance(double lat1, double lon1, double lat2, double lon2) {
        double x = Math.toRadians(lon2 - lon1) * Math.cos(Math.toRadians((lat1 + lat2) / 2));
        double y = Math.toRadians(lat2 - lat1);
        return Math.sqrt(x * x + y * y) * EARTH_RADIUS;
    }

    // Function to calculate speed in meters per second
    private double calculateSpeed(double lat1, double lon1, double lat2, double lon2, double timeDifferenceInSeconds) {
        double distance = calculateDistance(lat1, lon1, lat2, lon2);
        return (distance / timeDifferenceInSeconds) * 3.6f; // Convert to km/h;
    }

    private void requestSingleLocationUpdate() {
        try {
            locationManager.requestSingleUpdate(LocationManager.GPS_PROVIDER, this, null);
        } catch (SecurityException e) {
            e.printStackTrace();
            Snackbar.make(mLlTelematicsMain, "requestSingleLocationUpdate Exception", Snackbar.LENGTH_SHORT).show();
        }
    }

    private void stopLocationUpdates() {
        locationManager.removeUpdates(this);
        if (mLocationHandler != null)
            mLocationHandler.removeCallbacks(mLocationRunnable);
    }

    private String getCutOffLimit() {
        if (mEtLimit.getText().length() != 0) {
            if (Float.valueOf(mEtLimit.getText().toString()) > mAcceleration) {
                return "red";
            } else {
                return "green";
            }
        }

        return "";
    }

    private void registerSensor() {
        mStartTimestamp = String.valueOf(System.currentTimeMillis());
        mSensorManager = (SensorManager) getSystemService(Context.SENSOR_SERVICE);
        mSensor = mSensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);
        mSensorManager.registerListener(this, mSensor, SensorManager.SENSOR_DELAY_NORMAL);
        Snackbar.make(mLlTelematicsMain, "Sensor Registered", Snackbar.LENGTH_SHORT).show();

        final int[] recordsCount = {0};
        mTvRecordsSaved.setText(String.valueOf(recordsCount[0]));
        mHandler = new Handler();
        mRunnable = new Runnable() {
            @Override
            public void run() {
                try {
                    JSONObject jobj = new JSONObject();
                    JSONObject jobjData = new JSONObject();
                    String timestamp = String.valueOf(System.currentTimeMillis());
                    jobjData.putOpt("acceleration", mAcceleration);
                    jobjData.putOpt("speed", mSpeed);
                    jobjData.putOpt("orientation_count", orientationCount);
                    jobjData.putOpt("touch_count", touchCount);
                    jobjData.putOpt("timestamp", timestamp);
                    jobjData.putOpt("start_time", mStartTimestamp);
                    jobjData.putOpt("stop_time", mStopTimestamp);
                    jobjData.putOpt("latitude", mLatitude);
                    jobjData.putOpt("longitude", mLongitude);
                    jobjData.putOpt("limit", getCutOffLimit());
                    jobj.putOpt(timestamp, jobjData);
                    mAccelerationTimestamps.add(jobj);
                    recordsCount[0]++;
                    mTvRecordsSaved.setText(String.valueOf(recordsCount[0]));
                } catch (JSONException e) {
                    e.printStackTrace();
                    Snackbar.make(mLlTelematicsMain, "Runnable Failed", Snackbar.LENGTH_SHORT).show();
                }
                mHandler.postDelayed(this, TextUtils.isEmpty(mEtSampleTime.getText()) ? RECORDS_INTERVAL : Integer.valueOf(mEtSampleTime.getText().toString()));
            }
        };
        mHandler.postDelayed(mRunnable, TextUtils.isEmpty(mEtSampleTime.getText()) ? RECORDS_INTERVAL : Integer.valueOf(mEtSampleTime.getText().toString()));
    }

    private void unregisterSensor() {
        mStopTimestamp = String.valueOf(System.currentTimeMillis());
        orientationCount = 0;
        touchCount = 0;
        mSensorManager.unregisterListener(this);
        mTvVelocity.setText("0.0 m/s^2");
        mTvAvgVelocity.setText("0.0 km/h");
        Snackbar.make(mLlTelematicsMain, "Sensor Unregistered", Snackbar.LENGTH_SHORT).show();
        if (mHandler != null)
            mHandler.removeCallbacks(mRunnable);
        if (mTouchHandler != null)
            mTouchHandler.removeCallbacks(mTouchRunnable);
        saveAccelerationDataToFile();
    }

    private void saveAccelerationDataToFile() {
        jsonArray = new JSONArray();

        for (JSONObject jsonObject : mAccelerationTimestamps) {
            jsonArray.put(jsonObject);
        }

        //TODO: uncomment if needed to write to file
        if (ContextCompat.checkSelfPermission(this, Manifest.permission.WRITE_EXTERNAL_STORAGE)
                != PackageManager.PERMISSION_GRANTED) {
            ActivityCompat.requestPermissions(this,
                    new String[]{Manifest.permission.WRITE_EXTERNAL_STORAGE},
                    PERMISSION_REQUEST_WRITE_EXTERNAL_STORAGE);
        } else {
            saveAccelerationDataToExternalStorage(jsonArray);
        }
    }

    private void saveAccelerationDataToExternalStorage(JSONArray jsonArray) {
        //TODO: comment if needed to write to file
//        WarplyDBHelper.getInstance(this).saveTelematics(jsonArray);

        //TODO: uncomment if needed to write to file
        String state = Environment.getExternalStorageState();
        if (Environment.MEDIA_MOUNTED.equals(state)) {
            File documentsDir = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOCUMENTS);
            File file = new File(documentsDir, "telematics_data" + String.valueOf(System.currentTimeMillis()) + ".json");
            try {
                FileOutputStream fileOutputStream = new FileOutputStream(file);
                fileOutputStream.write(jsonArray.toString().getBytes());
                fileOutputStream.close();
                Snackbar.make(mLlTelematicsMain, "Success saving data to file", Snackbar.LENGTH_SHORT).show();
            } catch (IOException e) {
                e.printStackTrace();
                Snackbar.make(mLlTelematicsMain, "Error saving acceleration data to file", Snackbar.LENGTH_SHORT).show();
            }
        } else {
            Snackbar.make(mLlTelematicsMain, "External storage is not accessible", Snackbar.LENGTH_SHORT).show();
        }
    }

    private final View.OnTouchListener mTripTouchListener = new View.OnTouchListener() {
        @Override
        public boolean onTouch(View view, MotionEvent motionEvent) {
            if (mIsTripStarted) {
                if (motionEvent.getAction() == MotionEvent.ACTION_UP) {
                    mTouchHandler.removeCallbacks(mTouchRunnable);
                    onClick(view);
                    return true;
                }
            }
            return false;
        }
    };

    private final View.OnTouchListener mScrollTouchListener = new View.OnTouchListener() {
        @Override
        public boolean onTouch(View view, MotionEvent motionEvent) {
            if (mIsTripStarted) {
                switch (motionEvent.getAction()) {
                    case MotionEvent.ACTION_DOWN:
                        touchCount++;
                        mTvTouchCount.setText(String.valueOf(touchCount));
                        mTouchHandler.post(mTouchRunnable);
                        return true;
                    case MotionEvent.ACTION_UP:
                        mTouchHandler.removeCallbacks(mTouchRunnable);
                        return true;
                }
            }
            return false;
        }
    };

//    // Low-pass filter function using Exponential Moving Average (EMA)
//    private float lowPassFilter(float currentValue) {
//        float filteredValue = alpha * currentValue + (1 - alpha) * previousFilteredValue;
//        previousFilteredValue = filteredValue;
//        return filteredValue;
//    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

}
