/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.activities;

import static ly.warp.sdk.utils.constants.WarpConstants.RANDOM_MAX;
import static ly.warp.sdk.utils.constants.WarpConstants.RANDOM_MIN;

import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.text.TextUtils;
import android.view.KeyEvent;
import android.view.View;
import android.webkit.WebView;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;

import androidx.annotation.NonNull;
import androidx.lifecycle.ProcessLifecycleOwner;
import androidx.work.Constraints;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.WorkManager;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;
import org.json.JSONException;
import org.json.JSONObject;

import java.security.SecureRandom;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import ly.warp.sdk.Warply;
import ly.warp.sdk.db.WarplyDBHelper;
import ly.warp.sdk.io.callbacks.CallbackReceiver;
import ly.warp.sdk.io.models.CouponList;
import ly.warp.sdk.io.models.WarplyWebviewActivityCallbackEventModel;
import ly.warp.sdk.io.models.WarplyWebviewCallbackEventModel;
import ly.warp.sdk.io.request.PacingCalculateRequest;
import ly.warp.sdk.services.EventCampaignCouponService;
import ly.warp.sdk.services.EventQuestionnaireService;
import ly.warp.sdk.services.PushEventsClickedWorkerService;
import ly.warp.sdk.utils.WarpJSONParser;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.WarplyManagerHelper;
import ly.warp.sdk.utils.WarplyPreferences;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.managers.WarplyAnalyticsManager;
import ly.warp.sdk.utils.managers.WarplyEventBusManager;
import ly.warp.sdk.utils.managers.WarplyManager;
import ly.warp.sdk.utils.managers.WarplySessionManager;
import ly.warp.sdk.views.WarpView;

public class WarpViewActivity extends WarpBaseActivity {

    // ===========================================================
    // Fields
    // ===========================================================

    private WarpView mWarpView;
    private static Handler metersHandler;

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        metersHandler = new Handler();
        WarplySessionManager.onCreateActivity(this);
        initViews();
        Warply.getInitializer(this).init();
        setPageAccordingToIntent();
    }

    @Override
    protected void onNewIntent(Intent intent) {
        super.onNewIntent(intent);
        setIntent(intent);
        setPageAccordingToIntent();
    }

    @Override
    protected void onStart() {
        super.onStart();
        WarplySessionManager.onStartActivity(this);
        EventBus.getDefault().register(this);
    }

    @Override
    protected void onStop() {
        super.onStop();
        WarplySessionManager.onStopActivity(this);
        EventBus.getDefault().unregister(this);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (metersHandler != null) metersHandler.removeCallbacksAndMessages(null);
        sendSteps();
        WarpUtils.setWebviewParams(this, new JSONObject());
    }

    @Override
    public boolean onKeyDown(int keyCode, @NonNull KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_BACK) {
            if (mWarpView.canGoBack()) {
                mWarpView.goBack();
                return true;
            }
        }

        return super.onKeyDown(keyCode, event);
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == 3001) {
            WarplyWebviewCallbackEventModel webviewCallbackEventModel = new WarplyWebviewCallbackEventModel();
            webviewCallbackEventModel.setRequestId(requestCode);
            webviewCallbackEventModel.setResponseCode(grantResults[0] == PackageManager.PERMISSION_GRANTED ? "allow" : "deny");
            EventBus.getDefault().post(new WarplyEventBusManager(webviewCallbackEventModel));
        }
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (requestCode == 3002 || requestCode == 3003) {
            WarplyWebviewActivityCallbackEventModel webviewCallbackEventModel = new WarplyWebviewActivityCallbackEventModel();
            webviewCallbackEventModel.setRequestId(requestCode);
            webviewCallbackEventModel.setResponseCode(resultCode == RESULT_OK ? "enabled" : "disabled");
            EventBus.getDefault().post(new WarplyEventBusManager(webviewCallbackEventModel));
        }
    }

    @Subscribe()
    public void onMessageEvent(WarplyEventBusManager event) {
//        EventBus.getDefault().unregister(this);
        if (event.getQuestionnaire() != null) {
            OneTimeWorkRequest mywork = new OneTimeWorkRequest.Builder(EventQuestionnaireService.class).build();
            WorkManager.getInstance(WarpViewActivity.this).enqueue(mywork);
            setResult(RESULT_OK, new Intent());
            finish();
        }
        if (event.getCoupon() != null) {
            OneTimeWorkRequest mywork = new OneTimeWorkRequest.Builder(EventCampaignCouponService.class).build();
            WorkManager.getInstance(WarpViewActivity.this).enqueue(mywork);
        }
//        if (event.getPacingCard() != null)
//            finish();
//        if (event.getPacingService() != null)
//            finish();
        if (event.getPacing() != null && !event.getPacing().isVisible()) finish();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    private void initViews() {

        RelativeLayout root = new RelativeLayout(this);
        root.setBackgroundColor(Color.WHITE);

        final ImageView ivLogo = new ImageView(this);
        RelativeLayout.LayoutParams ivLogoParams = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        ivLogoParams.addRule(RelativeLayout.CENTER_IN_PARENT);
        root.addView(ivLogo, ivLogoParams);

        ivLogo.setImageDrawable(WarplyProperty.getProgressDrawable(this));
        ivLogo.startAnimation(WarpUtils.getPulseAnimation());

        mWarpView = new WarpView(this, WarpViewActivity.this, true);
        RelativeLayout.LayoutParams warpViewParams = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT);
        root.setFitsSystemWindows(true);
        root.addView(mWarpView, warpViewParams);


        int progressHeightDp = 7;
        final ProgressBar progressBar = new ProgressBar(this, null, android.R.attr.progressBarStyleHorizontal);
        progressBar.setProgressDrawable(WarpUtils.getHorizontalProgressDrawable(WarplyProperty.getProgressColor(this)));
        RelativeLayout.LayoutParams progressBarParams = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, (int) (progressHeightDp * getResources().getDisplayMetrics().density + 0.5f));
        mWarpView.setProgressChangeListener(new WarpView.ProgressChangeListener() {
            @Override
            public void onProgressChanged(WebView view, int newProgress) {

                progressBar.setProgress(newProgress);
                if (progressBar.getProgress() == 100) {
                    WarpUtils.animateVisibility(ivLogo, View.INVISIBLE);
                    WarpUtils.animateVisibility(progressBar, View.INVISIBLE);
                } else {
                    WarpUtils.animateVisibility(ivLogo, View.VISIBLE);
                    WarpUtils.animateVisibility(progressBar, View.VISIBLE);
                }
            }
        });
//        root.addView(progressBar, progressBarParams);
        setContentView(root);
    }

    private void setPageAccordingToIntent() {

        Intent intent = getIntent();
        String sessionUUID = intent.getStringExtra("sessionUUID");

        // user viewed campaign through notification
        if (intent.hasExtra("source") && intent.getStringExtra("source").equalsIgnoreCase("from_notification_status")) {
            JSONObject params = new JSONObject();
            try {
                params.putOpt("web_id", WarpUtils.getWebId(Warply.getWarplyContext()));
                params.putOpt("app_uuid", WarplyProperty.getAppUuid(Warply.getWarplyContext()));
                params.putOpt("api_key", WarpUtils.getApiKey(Warply.getWarplyContext()));
                params.putOpt("session_uuid", sessionUUID);
                params.putOpt("access_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token"));
                params.putOpt("refresh_token", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("refresh_token"));
                params.putOpt("client_id", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_id"));
                params.putOpt("client_secret", WarplyDBHelper.getInstance(Warply.getWarplyContext()).getClientValue("client_secret"));
            } catch (JSONException e) {
                e.printStackTrace();
            }

            WarpUtils.setWebviewParams(Warply.getWarplyContext(), params);

            setUpPushEvents(sessionUUID);
        }

        // ensure update campaigns and count if was used cache
        if (Warply.INSTANCE.getLastReceivedCampaigns() != null && Warply.INSTANCE.getLastReceivedCampaigns().containsUuid(sessionUUID)) {
            new WarplyPreferences(this).clearInboxLastCachedTimeStamps();
        }

        if (!TextUtils.isEmpty(sessionUUID)) {
            mWarpView.loadWarpSessionUUID(sessionUUID);
        } else {
            mWarpView.loadWarpUrl(getIntent().getStringExtra("sessionUrl"));
        }
    }

    private void setUpPushEvents(String sessionUUID) {
        WarplyAnalyticsManager.logUserEngagedPush(WarpJSONParser.createJSONObjectFromString("session_uuid", sessionUUID));

//        if (WorkManager.getInstance(this).getWorkInfosByTag(PushEventsWorkerService.TAG).isDone() || WorkManager.getInstance(this).getWorkInfosByTag(PushEventsWorkerService.TAG).isCancelled()) {
        Constraints constraints = new Constraints.Builder().setRequiredNetworkType(NetworkType.CONNECTED).build();
        OneTimeWorkRequest sendEvent = new OneTimeWorkRequest.Builder(PushEventsClickedWorkerService.class).setConstraints(constraints).setInitialDelay(defineRandomStart(), TimeUnit.MINUTES).build();

        WorkManager.getInstance(this).enqueue(sendEvent);
//        }
    }

    private int defineRandomStart() {
        if (Build.VERSION.SDK_INT >= 21) {
            return ThreadLocalRandom.current().nextInt(RANDOM_MIN, RANDOM_MAX + 1);
        } else {
            SecureRandom rand = new SecureRandom();
            return rand.nextInt((RANDOM_MAX - RANDOM_MIN) + 1) + RANDOM_MIN;
        }
    }

    public static Intent createIntentFromSessionUUID(Context context, String sessionUUID, String source) {

        Intent intent = new Intent(context, WarpViewActivity.class);
        intent.putExtra("sessionUUID", sessionUUID);
        if (!TextUtils.isEmpty(source)) {
            intent.putExtra("source", source);
        }
        return intent;
    }

    public static Intent createIntentFromSessionUUID(Context context, String sessionUUID) {
        return createIntentFromSessionUUID(context, sessionUUID, null);
    }

    public static Intent createIntentFromURL(Context context, String url) {
        return new Intent(context, WarpViewActivity.class).putExtra("sessionUrl", url);
    }

    private final CallbackReceiver<CouponList> mUserCouponsReceiver = new CallbackReceiver<CouponList>() {
        @Override
        public void onSuccess(CouponList result) {

        }

        @Override
        public void onFailure(int errorCode) {

        }
    };

    public static Handler getMetersHandler() {
        return metersHandler;
    }

    private void sendSteps() {
        String date = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss", Locale.getDefault()).format(new Date());
        WarpUtils.log("SEND_STEPS: " + String.valueOf(WarpUtils.getStepsCounter(this)));
        if (WarpUtils.getStepsCounter(this) > 0) {
            WarplyManager.setPacingDetails(new PacingCalculateRequest().setCounter(WarpUtils.getStepsCounter(this)).setDate(date), new CallbackReceiver<JSONObject>() {
                @Override
                public void onSuccess(JSONObject result) {
                    int status = result.optInt("status", 2);
                    if (status == 1) {
                        WarplyManagerHelper.mSteps = 0;
                        WarpUtils.setStepsCounter(WarpViewActivity.this, 0);
                    }
                }

                @Override
                public void onFailure(int errorCode) {
                    WarpUtils.log("Warply Health Service error: " + String.valueOf(errorCode));
                }
            });
        }
    }
}