
package ly.warp.sdk.db;


import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.text.TextUtils;
import android.util.Log;

import androidx.annotation.Nullable;

import com.getkeepsafe.relinker.ReLinker;

import net.sqlcipher.DatabaseUtils;
import net.sqlcipher.database.SQLiteDatabase;
import net.sqlcipher.database.SQLiteOpenHelper;
import net.sqlcipher.database.SQLiteStatement;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import ly.warp.sdk.utils.constants.WarpConstants;

public class WarplyDBHelper extends SQLiteOpenHelper {

    // ===========================================================
    // Constants
    // ===========================================================

    private enum State {
        DOES_NOT_EXIST, UNENCRYPTED, ENCRYPTED
    }

    private static final String DB_NAME = "warply.db";
    private static final int DB_VERSION = 10;
    private static final String KEY_CIPHER = "tn#mpOl3v3Dy1pr@W";

    //------------------------------ Fields -----------------------------//
    private static String TABLE_REQUESTS = "requests";
    private static String TABLE_PUSH_REQUESTS = "push_requests";
    private static String TABLE_PUSH_ACK_REQUESTS = "push_ack_requests";
    public static final String KEY_REQUESTS_ID = "_id";
    public static final String KEY_REQUESTS_MICROAPP = "microapp";
    public static final String KEY_REQUESTS_ENTITY = "entity";
    public static final String KEY_REQUESTS_DATE_ADDED = "date_added";
    public static final String KEY_REQUESTS_FORCE = "force_post";
    private static String TABLE_CLIENT = "client";
    private static String TABLE_AUTH = "auth";
    private static String TABLE_TAGS = "tags";
    private static String TABLE_TELEMATICS = "telematics";
    public static final String KEY_TAG = "tag";
    public static final String KEY_TAG_LAST_ADD_DATE = "last_add_date";
    public static final String KEY_CLIENT_ID = "client_id";
    public static final String KEY_CLIENT_SECRET = "client_secret";
    public static final String KEY_ACCESS_TOKEN = "access_token";
    public static final String KEY_REFRESH_TOKEN = "refresh_token";
    public static final String KEY_TIMESTAMP = "timestamp";
    public static final String KEY_ACCELERATION = "acceleration";
    public static final String KEY_SPEED = "speed";

    //------------------------------ Tables -----------------------------//
    public static final String CREATE_TABLE_REQUESTS = "create table if not exists "
            + TABLE_REQUESTS + " ("
            + KEY_REQUESTS_ID + " integer primary key autoincrement, "
            + KEY_REQUESTS_MICROAPP + " text not null, "
            + KEY_REQUESTS_ENTITY + " text not null, "
            + KEY_REQUESTS_FORCE + " integer default 0, "
            + KEY_REQUESTS_DATE_ADDED + " real)";

    public static final String CREATE_TABLE_PUSH_REQUESTS = "create table if not exists "
            + TABLE_PUSH_REQUESTS + " ("
            + KEY_REQUESTS_ID + " integer primary key autoincrement, "
            + KEY_REQUESTS_MICROAPP + " text not null, "
            + KEY_REQUESTS_ENTITY + " text not null, "
            + KEY_REQUESTS_FORCE + " integer default 0, "
            + KEY_REQUESTS_DATE_ADDED + " real)";

    public static final String CREATE_TABLE_PUSH_ACK_REQUESTS = "create table if not exists "
            + TABLE_PUSH_ACK_REQUESTS + " ("
            + KEY_REQUESTS_ID + " integer primary key autoincrement, "
            + KEY_REQUESTS_MICROAPP + " text not null, "
            + KEY_REQUESTS_ENTITY + " text not null, "
            + KEY_REQUESTS_FORCE + " integer default 0, "
            + KEY_REQUESTS_DATE_ADDED + " real)";

    public static final String CREATE_TABLE_TAGS = "create table if not exists "
            + TABLE_TAGS + " ("
            + KEY_TAG + " text not null, "
            + KEY_TAG_LAST_ADD_DATE + " real, unique(" + KEY_TAG + ") on conflict replace)";

    public static final String CREATE_TABLE_CLIENT = "create table if not exists "
            + TABLE_CLIENT + " ("
            + KEY_CLIENT_ID + " text, "
            + KEY_CLIENT_SECRET + " text)";

    public static final String CREATE_TABLE_AUTH = "create table if not exists "
            + TABLE_AUTH + " ("
            + KEY_ACCESS_TOKEN + " text, "
            + KEY_REFRESH_TOKEN + " text)";

    public static final String CREATE_TABLE_TELEMATICS = "create table if not exists "
            + TABLE_TELEMATICS + " ("
            + KEY_TIMESTAMP + " text, "
            + KEY_ACCELERATION + " real, "
            + KEY_SPEED + " real)";

    // ===========================================================
    // Fields
    // ===========================================================

    private SQLiteDatabase mDb;
    private static WarplyDBHelper mDBHelperInstance;

    // ===========================================================
    // Constructors
    // ===========================================================

    public static synchronized WarplyDBHelper getInstance(Context context) {
        if (mDBHelperInstance == null) {
//            SQLiteDatabase.loadLibs(context); //old implementation
            SQLiteDatabase.loadLibs(context, libraries -> {
                for (String library : libraries) {
                    ReLinker.loadLibrary(context, library);
                }
            });
            mDBHelperInstance = new WarplyDBHelper(context);
        }
        return mDBHelperInstance;
    }

    private WarplyDBHelper(Context context) {
        super(context, DB_NAME, null, DB_VERSION);
        State tempDatabaseState = getDatabaseState(context, DB_NAME);
        if (tempDatabaseState.equals(State.UNENCRYPTED)) {
            encrypt(context, context.getDatabasePath(DB_NAME), KEY_CIPHER.getBytes());
        }
    }

    /**
     * If database connection already initialized, return the db. Else create a
     * new one
     */
    private SQLiteDatabase getDb() {
        if (mDb == null)
            mDb = getWritableDatabase(KEY_CIPHER);
        return mDb;
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        db.execSQL(CREATE_TABLE_REQUESTS);
        db.execSQL(CREATE_TABLE_TAGS);
        db.execSQL(CREATE_TABLE_PUSH_REQUESTS);
        db.execSQL(CREATE_TABLE_PUSH_ACK_REQUESTS);
        db.execSQL(CREATE_TABLE_CLIENT);
        db.execSQL(CREATE_TABLE_AUTH);
        db.execSQL(CREATE_TABLE_TELEMATICS);
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        /* Called when database scheme is changed */
        if (oldVersion < DB_VERSION) {
            db.execSQL("drop table if exists " + TABLE_REQUESTS);
            db.execSQL("drop table if exists " + TABLE_PUSH_REQUESTS);
            db.execSQL("drop table if exists " + TABLE_PUSH_ACK_REQUESTS);
            db.execSQL("drop table if exists " + TABLE_TAGS);
            db.execSQL("drop table if exists " + TABLE_CLIENT);
            db.execSQL("drop table if exists " + TABLE_AUTH);
            db.execSQL("drop table if exists " + TABLE_TELEMATICS);
            onCreate(db);
        }
    }

    @Override
    public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        db.execSQL("drop table if exists " + TABLE_REQUESTS);
        db.execSQL("drop table if exists " + TABLE_PUSH_REQUESTS);
        db.execSQL("drop table if exists " + TABLE_PUSH_ACK_REQUESTS);
        db.execSQL("drop table if exists " + TABLE_TAGS);
        db.execSQL("drop table if exists " + TABLE_CLIENT);
        db.execSQL("drop table if exists " + TABLE_AUTH);
        db.execSQL("drop table if exists " + TABLE_TELEMATICS);
        onCreate(db);
    }

    /**
     * Release database connection
     */
    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        if (mDb != null && mDb.isOpen()) {
            mDb.close();
            mDb = null;
        }
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public synchronized void clearTable(String tableName) {
        SQLiteDatabase db = getDb();
        db.delete(tableName, null, null);
    }

    public synchronized void insert(String tableName, ContentValues values) {
        SQLiteDatabase db = getDb();
        db.insert(tableName, null, values);
    }

    public synchronized void update(String tableName, ContentValues values) {
        SQLiteDatabase db = getDb();
        db.update(tableName, values, null, null);
    }

    public synchronized boolean isTableNotEmpty(String tableName) {
        Cursor cursor = getDb().rawQuery("SELECT COUNT(*) FROM " + tableName,
                null);
        if (cursor != null && cursor.moveToFirst()) {
            boolean isNotEmpty = cursor.getInt(0) > 0;
            cursor.close();
            return isNotEmpty;
        } else
            return false;
    }

    //------------------------------ Auth -----------------------------//
    public synchronized void saveClientAccess(String clientId, String clientSecret) {
        ContentValues values = new ContentValues();
        if (!TextUtils.isEmpty(clientId))
            values.put(KEY_CLIENT_ID, clientId);
        if (!TextUtils.isEmpty(clientSecret))
            values.put(KEY_CLIENT_SECRET, clientSecret);
        if (isTableNotEmpty(TABLE_CLIENT))
            update(TABLE_CLIENT, values);
        else
            insert(TABLE_CLIENT, values);
    }

    public synchronized void saveAuthAccess(String accessToken, String refreshToken) {
        ContentValues values = new ContentValues();
        if (!TextUtils.isEmpty(accessToken))
            values.put(KEY_ACCESS_TOKEN, accessToken);
        if (!TextUtils.isEmpty(refreshToken))
            values.put(KEY_REFRESH_TOKEN, refreshToken);
        if (isTableNotEmpty(TABLE_AUTH))
            update(TABLE_AUTH, values);
        else
            insert(TABLE_AUTH, values);
    }

    @Nullable
    public synchronized String getAuthValue(String columnName) {
        String columnValue = "";
        Cursor cursor = getDb().query(TABLE_AUTH, new String[]{columnName}, null, null, null, null, null);
        if (cursor != null && cursor.moveToFirst()) {
            cursor.moveToFirst();
            columnValue = cursor.getString(cursor.getColumnIndex(columnName));
            cursor.close();
        }
        return columnValue;
    }

    @Nullable
    public synchronized String getClientValue(String columnName) {
        String columnValue = "";
        Cursor cursor = getDb().query(TABLE_CLIENT, new String[]{columnName}, null, null, null, null, null);
        if (cursor != null && cursor.moveToFirst()) {
            cursor.moveToFirst();
            columnValue = cursor.getString(cursor.getColumnIndex(columnName));
            cursor.close();
        }
        return columnValue;
    }

    public synchronized void deleteClient() {
        clearTable(TABLE_CLIENT);
    }

    public synchronized void deleteAuth() {
        clearTable(TABLE_AUTH);
    }

    //------------------------------ Api requests -----------------------------//
    public synchronized Cursor getAllRequests() {

        return getDb().query(TABLE_REQUESTS,
                new String[]{KEY_REQUESTS_ID,
                        KEY_REQUESTS_MICROAPP, KEY_REQUESTS_ENTITY}, null, null, null, null,
                KEY_REQUESTS_DATE_ADDED + " asc", "20");
    }

    public synchronized Cursor getAllPushRequests() {

        return getDb().query(TABLE_PUSH_REQUESTS,
                new String[]{KEY_REQUESTS_ID,
                        KEY_REQUESTS_MICROAPP, KEY_REQUESTS_ENTITY}, null, null, null, null,
                KEY_REQUESTS_DATE_ADDED + " asc", "20");
    }

    public synchronized Cursor getAllPushAckRequests() {

        return getDb().query(TABLE_PUSH_ACK_REQUESTS,
                new String[]{KEY_REQUESTS_ID,
                        KEY_REQUESTS_MICROAPP, KEY_REQUESTS_ENTITY}, null, null, null, null,
                KEY_REQUESTS_DATE_ADDED + " asc", "20");
    }

    public synchronized void deleteAllRequests() {
        clearTable(TABLE_REQUESTS);
    }

    public synchronized void deleteAllPushRequests() {
        clearTable(TABLE_PUSH_REQUESTS);
    }

    public synchronized void deleteAllPushAckRequests() {
        clearTable(TABLE_PUSH_ACK_REQUESTS);
    }

    public synchronized long addRequest(String microapp, String entity, boolean force) {
        ContentValues values = new ContentValues();
        values.put(KEY_REQUESTS_MICROAPP, microapp);
        values.put(KEY_REQUESTS_ENTITY, entity);
        values.put(KEY_REQUESTS_FORCE, force ? 1 : 0);
        values.put(KEY_REQUESTS_DATE_ADDED, System.currentTimeMillis());
        insert(TABLE_REQUESTS, values);
        return getRequestsInQueueCount();
    }

    public synchronized long addPushRequest(String microapp, String entity, boolean force) {
        ContentValues values = new ContentValues();
        values.put(KEY_REQUESTS_MICROAPP, microapp);
        values.put(KEY_REQUESTS_ENTITY, entity);
        values.put(KEY_REQUESTS_FORCE, force ? 1 : 0);
        values.put(KEY_REQUESTS_DATE_ADDED, System.currentTimeMillis());
        insert(TABLE_PUSH_REQUESTS, values);
        return getPushRequestsInQueueCount();
    }

    public synchronized long addPushAckRequest(String microapp, String entity, boolean force) {
        ContentValues values = new ContentValues();
        values.put(KEY_REQUESTS_MICROAPP, microapp);
        values.put(KEY_REQUESTS_ENTITY, entity);
        values.put(KEY_REQUESTS_FORCE, force ? 1 : 0);
        values.put(KEY_REQUESTS_DATE_ADDED, System.currentTimeMillis());
        insert(TABLE_PUSH_ACK_REQUESTS, values);
        return getPushAckRequestsInQueueCount();
    }

    public synchronized long getRequestsInQueueCount() {
        return DatabaseUtils.queryNumEntries(getReadableDatabase(KEY_CIPHER), TABLE_REQUESTS);
    }

    public synchronized long getPushRequestsInQueueCount() {
        return DatabaseUtils.queryNumEntries(getReadableDatabase(KEY_CIPHER), TABLE_PUSH_REQUESTS);
    }

    public synchronized long getPushAckRequestsInQueueCount() {
        return DatabaseUtils.queryNumEntries(getReadableDatabase(KEY_CIPHER), TABLE_PUSH_ACK_REQUESTS);
    }

    public synchronized void deleteRequests(Long... ids) {

        StringBuilder strFilter = new StringBuilder();
        for (int i = 0; i < ids.length; i++) {
            if (i > 0) {
                strFilter.append(" OR ");
            }
            strFilter.append(KEY_REQUESTS_ID);
            strFilter.append("=");
            strFilter.append(ids[i]);
        }
        getDb().delete(TABLE_REQUESTS, strFilter.toString(), null);
    }

    public synchronized void deletePushRequests(Long... ids) {

        StringBuilder strFilter = new StringBuilder();
        for (int i = 0; i < ids.length; i++) {
            if (i > 0) {
                strFilter.append(" OR ");
            }
            strFilter.append(KEY_REQUESTS_ID);
            strFilter.append("=");
            strFilter.append(ids[i]);
        }
        getDb().delete(TABLE_PUSH_REQUESTS, strFilter.toString(), null);
    }

    public synchronized void deletePushAckRequests(Long... ids) {

        StringBuilder strFilter = new StringBuilder();
        for (int i = 0; i < ids.length; i++) {
            if (i > 0) {
                strFilter.append(" OR ");
            }
            strFilter.append(KEY_REQUESTS_ID);
            strFilter.append("=");
            strFilter.append(ids[i]);
        }
        getDb().delete(TABLE_PUSH_ACK_REQUESTS, strFilter.toString(), null);
    }

    public synchronized boolean isForceRequestsExist() {
        Cursor cursor = getDb().query(TABLE_REQUESTS, null, KEY_REQUESTS_FORCE + "=1",
                null, null, null, null);
        boolean result = false;
        if (cursor != null) {
            result = cursor.getCount() > 0;
            cursor.close();
        }
        return result;
    }

    public synchronized boolean isForcePushRequestsExist() {
        Cursor cursor = getDb().query(TABLE_PUSH_REQUESTS, null, KEY_REQUESTS_FORCE + "=1",
                null, null, null, null);
        boolean result = false;
        if (cursor != null) {
            result = cursor.getCount() > 0;
            cursor.close();
        }
        return result;
    }

    public synchronized boolean isForcePushAckRequestsExist() {
        Cursor cursor = getDb().query(TABLE_PUSH_ACK_REQUESTS, null, KEY_REQUESTS_FORCE + "=1",
                null, null, null, null);
        boolean result = false;
        if (cursor != null) {
            result = cursor.getCount() > 0;
            cursor.close();
        }
        return result;
    }

    //------------------------------ Tags -----------------------------//
    public synchronized void saveTags(String[] tags) {
        if (tags != null && tags.length > 0) {

            try {
                getDb().beginTransaction();
                ContentValues values = new ContentValues();
                for (String tag : tags) {
                    values.put(KEY_TAG, tag);
                    values.put(KEY_TAG_LAST_ADD_DATE, System.currentTimeMillis());
                    insert(TABLE_TAGS, values);
                }
                getDb().setTransactionSuccessful();

            } catch (SQLException e) {
                if (WarpConstants.DEBUG) {
                    e.printStackTrace();
                }
            } finally {
                getDb().endTransaction();
            }
        }
    }

    public synchronized void saveTelematics(JSONArray jsonArray) {
        if (jsonArray != null && jsonArray.length() > 0) {
            ContentValues values = new ContentValues();
            for (int i = 0; i < jsonArray.length(); i++) {
                JSONObject jsonobject = jsonArray.optJSONObject(i);
                if (jsonobject != null) {
                    String timestamp = jsonobject.keys().next();
                    values.put(KEY_TIMESTAMP, timestamp);
                    JSONObject jobjData = jsonobject.optJSONObject(timestamp);
                    if (jobjData != null) {
                        values.put(KEY_ACCELERATION, jobjData.optDouble(KEY_ACCELERATION));
                        values.put(KEY_SPEED, jobjData.optDouble(KEY_SPEED));
                    }
                    insert(TABLE_TELEMATICS, values);
                }
            }
        }
    }

    public synchronized void removeTags(String[] tags) {
        StringBuilder strFilter = new StringBuilder();
        for (int i = 0; i < tags.length; i++) {
            if (i > 0) {
                strFilter.append(" OR ");
            }
            strFilter.append(KEY_TAG);
            strFilter.append("=");
            strFilter.append("'");
            strFilter.append(tags[i]);
            strFilter.append("'");
        }
        getDb().delete(TABLE_TAGS, strFilter.toString(), null);
    }

    public synchronized void removeAllTags() {
        clearTable(TABLE_TAGS);
    }

    @Nullable
    public synchronized String[] getTags() {

        List<String> tags = null;
        Cursor cursor = getDb().query(TABLE_TAGS, null, null, null, null, null, null);
        if (cursor != null) {

            tags = new ArrayList<>(cursor.getCount());
            while (cursor.moveToNext()) {
                tags.add(cursor.getString(cursor
                        .getColumnIndex(KEY_TAG)));
            }
            cursor.close();
        }
        return tags != null ? tags.toArray(new String[tags.size()]) : null;
    }

    private State getDatabaseState(Context context, String dbName) {
        SQLiteDatabase.loadLibs(context);

        return (getDatabaseState(context.getDatabasePath(dbName)));
    }

    private static State getDatabaseState(File dbPath) {
        if (dbPath.exists()) {
            SQLiteDatabase db = null;
            try {
                db = SQLiteDatabase.openDatabase(dbPath.getAbsolutePath(), "", null, SQLiteDatabase.OPEN_READONLY);
                db.getVersion();

                return (State.UNENCRYPTED);
            } catch (Exception e) {
                return (State.ENCRYPTED);
            } finally {
                if (db != null) {
                    db.close();
                }
            }
        }

        return (State.DOES_NOT_EXIST);
    }

    private void encrypt(Context context, File originalFile, byte[] passphrase) {
        SQLiteDatabase.loadLibs(context);

        try {
            if (originalFile.exists()) {
                File newFile = File.createTempFile("sqlcipherutils", "tmp", context.getCacheDir());
                SQLiteDatabase db = SQLiteDatabase.openDatabase(originalFile.getAbsolutePath(),
                        "", null, SQLiteDatabase.OPEN_READWRITE);
                int version = db.getVersion();

                db.close();

                db = SQLiteDatabase.openDatabase(newFile.getAbsolutePath(), passphrase,
                        null, SQLiteDatabase.OPEN_READWRITE, null, null);

                final SQLiteStatement st = db.compileStatement("ATTACH DATABASE ? AS plaintext KEY ''");

                st.bindString(1, originalFile.getAbsolutePath());
                st.execute();

                db.rawExecSQL("SELECT sqlcipher_export('main', 'plaintext')");
                db.rawExecSQL("DETACH DATABASE plaintext");
                db.setVersion(version);
                st.close();
                db.close();

                originalFile.delete();
                newFile.renameTo(originalFile);
            } else {
                throw new FileNotFoundException(originalFile.getAbsolutePath() + " not found");
            }
        } catch (IOException ex) {
            Log.v("WarplyDB Exception: ", ex.getMessage());
        }
    }
}
