package ly.warp.sdk.fragments;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.StateListDrawable;
import android.os.Build;
import android.os.Bundle;
import androidx.fragment.app.Fragment;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.webkit.WebView;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.WarplyPreferences;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.views.WarpView;

public class WarpViewFragment extends Fragment {

    // ===========================================================
    // Constants
    // ===========================================================

    private static final String ARGS_KEY_SESSION_UUID = "args_key_session_uuid";
    private static final String ARGS_KEY_USE_PROGRESS = "args_key_use_progress";
    private static final String ARGS_KEY_SHOW_CLOSE_BTN = "args_key_show_close_btn";

    // ===========================================================
    // Fields
    // ===========================================================

    private WarpView mWarpView;

    private String mSessionUuid;
    private boolean mIsUseProgress, mIsShowCloseBtn;
    private ImageView mBtnClose;

    // ===========================================================
    // Constructors
    // ===========================================================

    public static WarpViewFragment newInstance() {
        return newInstance(null, false, true);
    }

    public static WarpViewFragment newInstance(String sessionUuid) {
        return newInstance(sessionUuid, false, true);
    }

    public static WarpViewFragment newInstance(String sessionUuid, boolean isUseProgress, boolean isShowCloseBtn) {

        Bundle args = new Bundle();
        args.putString(ARGS_KEY_SESSION_UUID, sessionUuid);
        args.putBoolean(ARGS_KEY_USE_PROGRESS, isUseProgress);
        args.putBoolean(ARGS_KEY_SHOW_CLOSE_BTN, isShowCloseBtn);

        WarpViewFragment fr = new WarpViewFragment();
        fr.setArguments(args);
        return fr;
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================


    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        if (getArguments() != null) {
            mSessionUuid = getArguments().getString(ARGS_KEY_SESSION_UUID);
            mIsUseProgress = getArguments().getBoolean(ARGS_KEY_USE_PROGRESS, false);
            mIsShowCloseBtn = getArguments().getBoolean(ARGS_KEY_SHOW_CLOSE_BTN, true);
        }
    }

    @Override
    public View onCreateView(LayoutInflater inflater, final ViewGroup container,
                             Bundle savedInstanceState) {

        // root
        final RelativeLayout root = new RelativeLayout(getContext());
        root.setBackgroundColor(Color.parseColor(WarpConstants.WARPLY_THEME_TRANSPARENT_COLOR));
        root.setClickable(true);

        // logo
        final FrameLayout ivLogoContainer = new FrameLayout(getContext());
        ivLogoContainer.setBackgroundColor(Color.WHITE);
        RelativeLayout.LayoutParams ivLogoContainerParams = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.WRAP_CONTENT,
                RelativeLayout.LayoutParams.WRAP_CONTENT);
        ivLogoContainerParams.addRule(RelativeLayout.CENTER_IN_PARENT);
        root.addView(ivLogoContainer, ivLogoContainerParams);
        final ImageView ivLogo = new ImageView(getContext());
        FrameLayout.LayoutParams ivLogoParams = new FrameLayout.LayoutParams(
                RelativeLayout.LayoutParams.WRAP_CONTENT,
                RelativeLayout.LayoutParams.WRAP_CONTENT);
        ivLogoParams.gravity = Gravity.CENTER;
        ivLogoContainer.addView(ivLogo, ivLogoParams);

        ivLogo.setImageDrawable(WarplyProperty.getProgressDrawable(getContext()));
        ivLogo.startAnimation(WarpUtils.getPulseAnimation());

        // web view
        mWarpView = new WarpView(getContext());
        mWarpView.setVisibility(View.INVISIBLE);
        final RelativeLayout.LayoutParams warpViewParams = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT,
                RelativeLayout.LayoutParams.MATCH_PARENT);
        root.addView(mWarpView, warpViewParams);

        if (mIsUseProgress) {
            int progressHeightDp = 7;
            final ProgressBar progressBar = new ProgressBar(getContext(), null, android.R.attr.progressBarStyleHorizontal);
            progressBar.setProgressDrawable(WarpUtils.getHorizontalProgressDrawable(WarplyProperty.getProgressColor(getContext())));
            RelativeLayout.LayoutParams progressBarParams = new RelativeLayout.LayoutParams(
                    RelativeLayout.LayoutParams.MATCH_PARENT, (int)
                    (progressHeightDp * getResources().getDisplayMetrics().density + 0.5f));
            mWarpView.setProgressChangeListener(new WarpView.ProgressChangeListener() {
                @Override
                public void onProgressChanged(WebView view, int newProgress) {

                    progressBar.setProgress(newProgress);
                    if (progressBar.getProgress() == 100) {
                        WarpUtils.animateVisibility(ivLogo, View.INVISIBLE);
                        WarpUtils.animateVisibility(progressBar, View.INVISIBLE);
                    } else {
                        WarpUtils.animateVisibility(ivLogo, View.VISIBLE);
                        WarpUtils.animateVisibility(progressBar, View.VISIBLE);
                    }
                }
            });
            root.addView(progressBar, progressBarParams);
        }

        // x button
        if (mIsShowCloseBtn) {
            mBtnClose = new ImageView(getContext());
            RelativeLayout.LayoutParams btnCloseParams = new RelativeLayout.LayoutParams(
                    RelativeLayout.LayoutParams.WRAP_CONTENT,
                    RelativeLayout.LayoutParams.WRAP_CONTENT);
            btnCloseParams.addRule(RelativeLayout.ALIGN_PARENT_TOP);
            btnCloseParams.addRule(RelativeLayout.ALIGN_PARENT_RIGHT);
            root.addView(mBtnClose, btnCloseParams);
            Bitmap bmpCloseBtnNormal = WarpUtils.getWarplyCloseBtn();
            Bitmap bmpCloseBtnPressed = Bitmap.createBitmap(
                    bmpCloseBtnNormal.getWidth(),
                    bmpCloseBtnNormal.getHeight(), Bitmap.Config.ARGB_8888);
            Canvas canvas = new Canvas(bmpCloseBtnPressed);
            Paint paint = new Paint();
            paint.setAlpha(127);
            canvas.drawBitmap(bmpCloseBtnNormal, 0, 0, paint);
            BitmapDrawable normalState = new BitmapDrawable(getResources(), bmpCloseBtnNormal);
            BitmapDrawable pressedState = new BitmapDrawable(getResources(), bmpCloseBtnPressed);
            pressedState.setAlpha(0);
            StateListDrawable states = new StateListDrawable();
            states.addState(new int[]{android.R.attr.state_pressed}, pressedState);
            states.addState(new int[]{}, normalState);
            mBtnClose.setImageDrawable(states);
            mBtnClose.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {

                    if (getActivity().getSupportFragmentManager()
                            .findFragmentById(getId()) != null) {

                        getActivity().getSupportFragmentManager()
                                .beginTransaction().remove(WarpViewFragment.this)
                                .commitAllowingStateLoss();
                    } else if (getParentFragment() != null) {

                        if (getParentFragment().getChildFragmentManager()
                                .findFragmentById(getId()) != null) {
                            getParentFragment().getChildFragmentManager().beginTransaction()
                                    .remove(WarpViewFragment.this)
                                    .commitAllowingStateLoss();
                        }
                    }
                }
            });
        }

        // setup runtime margins
        container.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {

                if (container.getWidth() > 0 && container.getHeight() > 0) {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                        container.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                    } else {
                        container.getViewTreeObserver().removeGlobalOnLayoutListener(this);
                    }

                    int margin = mBtnClose == null ? 0 : container.getWidth() / 8;
                    RelativeLayout.LayoutParams lp = (RelativeLayout.LayoutParams) mWarpView.getLayoutParams();
                    lp.setMargins(margin, margin, margin, margin);
                    mWarpView.setLayoutParams(lp);

                    lp = (RelativeLayout.LayoutParams) ivLogoContainer.getLayoutParams();
                    lp.width = RelativeLayout.LayoutParams.MATCH_PARENT;
                    lp.height = RelativeLayout.LayoutParams.MATCH_PARENT;
                    lp.setMargins(margin, margin, margin, margin);
                    ivLogoContainer.setLayoutParams(lp);

                    if (mBtnClose != null) {
                        mBtnClose.measure(View.MeasureSpec.UNSPECIFIED, View.MeasureSpec.UNSPECIFIED);
                        mBtnClose.setPadding(0, (margin / 2) - (mBtnClose.getMeasuredHeight() / 2),
                                (margin / 2) - (mBtnClose.getMeasuredWidth() / 2), 0);
                    }
                }
            }
        });

        return root;
    }

    @Override
    public void onViewCreated(View view, Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        updatePage();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    private void updatePage() {

        if (mWarpView != null && !TextUtils.isEmpty(mSessionUuid)) {

            // ensure update campaigns and count if was used cache
            if (Warply.INSTANCE.getLastReceivedCampaigns() != null &&
                    Warply.INSTANCE.getLastReceivedCampaigns().containsUuid(mSessionUuid)) {
                new WarplyPreferences(getActivity()).clearInboxLastCachedTimeStamps();
            }

            if (mWarpView.getVisibility() != View.VISIBLE) {
                mWarpView.setVisibility(View.VISIBLE);
            }

            mWarpView.loadWarpSessionUUID(mSessionUuid);
        }
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

}
