/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.callbacks;

import org.json.JSONException;
import org.json.JSONObject;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.io.volley.NoConnectionError;
import ly.warp.sdk.io.volley.ParseError;
import ly.warp.sdk.io.volley.Response.ErrorListener;
import ly.warp.sdk.io.volley.Response.Listener;
import ly.warp.sdk.io.volley.ServerError;
import ly.warp.sdk.io.volley.TimeoutError;
import ly.warp.sdk.io.volley.VolleyError;
import ly.warp.sdk.utils.WarpUtils;

import static ly.warp.sdk.Warply.getWarplyContext;

public class VolleyTransformer implements Listener<JSONObject>, ErrorListener {

    public VolleyTransformer(CallbackReceiver<JSONObject> listener) {
        mListener = listener;
    }

    private static final String WARPLY_STATUS = "status";
    private final CallbackReceiver<JSONObject> mListener;

    @Override
    public void onErrorResponse(VolleyError error) {
        if (mListener == null)
            return;
        int resultCode = WarpConstants.RESULT_CODE_UNKNOWN;
        if (NoConnectionError.class.isInstance(error)) {
            resultCode = WarpConstants.RESULT_CODE_NO_INTERNET;
        } else if (ParseError.class.isInstance(error)) {
            resultCode = WarpConstants.RESULT_CODE_PARSING_ERROR;
        } else if (ServerError.class.isInstance(error)) {
//                resultCode = WarpConstants.RESULT_CODE_SERVER_ERROR;
            try {
                resultCode = error.networkResponse.statusCode;
            } catch (NullPointerException e) {
                resultCode = WarpConstants.RESULT_CODE_SERVER_ERROR;
            }
        } else if (TimeoutError.class.isInstance(error)) {
            resultCode = WarpConstants.RESULT_CODE_TIMEOUT;
        } else if (error.networkResponse != null) {
            resultCode = error.networkResponse.statusCode;
        }

        WarpUtils.warn("[WARP Trace] Warply Connection Error: " + resultCode,
                error);
        mListener.onFailure(resultCode);
    }

    @Override
    public void onResponse(JSONObject response) {
        if (response != null) {
            if (response.optJSONObject("context") != null) {
                if (response.optJSONObject("context").optJSONObject("device_status") != null) {
                    WarpUtils.setIsWarped(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optBoolean("warped"));
//                    WarpUtils.setDeviceToken(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optString("device_token"));
//                    WarpUtils.setHasApplicationInfo(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optBoolean("application_info_saved"));
                    WarpUtils.setDeviceId(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optString("device_id"));
                    WarpUtils.setIsDeviceIdSaved(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optBoolean("device_id_saved"));
//                    WarpUtils.setIsDeviceInfoSaved(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optBoolean("device_info_saved"));
                    if (response.optJSONObject("context").optJSONObject("device_status").optJSONObject("application_info") != null) {
                        WarpUtils.setAppDataObject(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optJSONObject("application_info"));
                        WarpUtils.setSDKVersion(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optJSONObject("application_info").optString("sdk_version"));
                        WarpUtils.setAppVersion(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optJSONObject("application_info").optString("app_version"));
                    }
                    if (response.optJSONObject("context").optJSONObject("device_status").optJSONObject("device_info") != null) {
                        WarpUtils.setDeviceInfoObject(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optJSONObject("device_info"));
                        WarpUtils.setOSVersion(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optJSONObject("device_info").optString("os_version"));
                        WarpUtils.setDeviceBrand(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optJSONObject("device_info").optString("manufacturer"));
                        WarpUtils.setCarrierName(getWarplyContext(), response.optJSONObject("context").optJSONObject("device_status").optJSONObject("device_info").optString("carrier_name"));
                    }
                }
                if (response.optJSONObject("context").optJSONObject("application_data") != null) {
                    WarpUtils.setIsAPPDATAENABLED(getWarplyContext(), response.optJSONObject("context").optJSONObject("application_data").optBoolean("APPLICATION_DATA_ENABLED"));
                    WarpUtils.setIsDEVICEINFOENABLED(getWarplyContext(), response.optJSONObject("context").optJSONObject("application_data").optBoolean("DEVICE_INFO_ENABLED"));
                    WarpUtils.setIsLIFECYCLEANALYTICSENABLED(getWarplyContext(), response.optJSONObject("context").optJSONObject("application_data").optBoolean("LIFECYCLE_ANALYTICS_ENABLED", true));
                }
            }
        }

        if (mListener == null)
            return;
        WarpUtils.log("**************** WARPLY Response *****************");
        WarpUtils.verbose("[WARP Trace] HTTP Web Id: "
                + WarpUtils.getWebId(Warply.INSTANCE.mContext));
        WarpUtils.verbose("[WARP Trace] HTTP API Key: "
                + WarplyProperty.getAppUuid(Warply.INSTANCE.mContext));
        if (response != null) {
            try {
//                WarpUtils.log("[WARP Trace] Request: " + response.toString(2));

                WarpUtils.log("[WARP Trace] Response: ");
                if (response.toString().length() > 1000) {
                    for (int i = 0; i < response.toString().length(); i += 1000) {
                        if (i + 1000 < response.toString().length())
                            WarpUtils.log(response.toString().substring(i, i + 1000));
                        else
                            WarpUtils.log(response.toString().substring(i, response.toString().length()));
                    }
//                for (int i = 0, length = response.toString().length(); i < length; i++) {
//                    int newline = response.toString().indexOf('\n', i);
//                    newline = newline != -1 ? newline : length;
//                    do {
//                        int end = Math.min(newline, i + 4000);
//                        WarpUtils.log(response.toString().substring(i, end));
//                        i = end;
//                    } while (i < newline);
//                }
                } else
                    WarpUtils.log("[WARP Trace] Request: " + response.toString(2));
            } catch (/*JSONException*/ Exception e) {
                WarpUtils.warn(
                        "[WARP Trace] Failed conversting JSON to string", e);
            }

            WarpUtils.log("*************************************************");

            if (response.has("result")) {
                JSONObject obj = response.optJSONObject("result");
                if (obj != null) {
                    if ((response.has("status") && response.optInt("status") == 7) && (obj.has("status") && obj.optString("status").equals("rejected"))) {
                        mListener.onSuccess(response);
                        return;
                    }

                    if (obj.has("status") && obj.optString("status").equals("accepted")) {
                        mListener.onSuccess(response);
                        return;
                    }
                }
            }

            if (response.has("code")) {
                mListener.onSuccess(response);
                return;
            }

            if (response.has("token_type")) {
                mListener.onSuccess(response);
                return;
            }

            if (response.has("tokens")) {
                if (response.optInt("result") == 1) {
                    mListener.onSuccess(response);
                    return;
                }
                mListener.onFailure(response.optInt("result"));
                return;
            }

            // Check for server related error.
            int serverResult = response.optInt(WARPLY_STATUS);

            if (serverResult == WarpConstants.RESULT_CODE_ERROR || serverResult == WarpConstants.RESULT_CODE_NO_ACCESS
                    || serverResult == WarpConstants.RESULT_CODE_INVALID_CAMPAIGN) {
                mListener.onSuccess(response);
                return;
            }
            if (serverResult != WarpConstants.RESULT_CODE_SUCCESS) {
                if ((serverResult == WarpConstants.RESULT_CODE_INVALID_WEB_ID)
                        && (Warply.isInitialized()))
                    Warply.resetWarplyWebId();
                WarpUtils
                        .log("************* WARPLY Request Error ********************");
                WarpUtils.log("[WARP Trace] Warply Request Error: " + serverResult);
                WarpUtils
                        .log("*******************************************************");
                mListener.onFailure(serverResult);
                return;
            }
        }

        mListener.onSuccess(response);
    }
}
