/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 18-Jan-22.
 */

public class Address implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String ADDRESS_NAME = "address_name";
    private static final String ADDRESS_NUMBER = "address_number";
    private static final String CITY = "city";
    private static final String COUNTRY = "country";
    private static final String DOORBELL = "doorbel";
    private static final String FLOOR_NUMBER = "floor_number";
    private static final String FRIENDLY_NAME = "friendly_name";
    private static final String LATITUDE = "latitude";
    private static final String LONGITUDE = "longitude";
    private static final String NOTES = "notes";
    private static final String POSTAL_CODE = "postal_code";
    private static final String REGION = "region";
    private static final String UUID = "uuid";

    /* Member variables of the Campaign object */

    private String addressName = "";
    private int addressNumber = 0;
    private String city = "";
    private String country = "";
    private String doorbell = "";
    private String floorNumber = "";
    private String friendlyName = "";
    private double latitude = 0.0d;
    private double longitude = 0.0d;
    private String notes = "";
    private String postalCode = "";
    private String region = "";
    private String uuid = "";

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public Address(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Address
     */
    public Address(JSONObject json) {
        if (json != null) {
            this.addressName = json.optString(ADDRESS_NAME);
            this.addressNumber = json.optInt(ADDRESS_NUMBER);
            this.city = json.optString(CITY);
            this.country = json.optString(COUNTRY);
            this.doorbell = json.optString(DOORBELL);
            this.floorNumber = json.optString(FLOOR_NUMBER);
            this.friendlyName = json.optString(FRIENDLY_NAME);
            this.latitude = json.optDouble(LATITUDE);
            this.longitude = json.optDouble(LONGITUDE);
            this.notes = json.optString(NOTES);
            this.postalCode = json.optString(POSTAL_CODE);
            this.region = json.optString(REGION);
            this.uuid = json.optString(UUID);
        }
    }

    public Address(Parcel source) {
        this.addressName = source.readString();
        this.addressNumber = source.readInt();
        this.city = source.readString();
        this.country = source.readString();
        this.doorbell = source.readString();
        this.floorNumber = source.readString();
        this.friendlyName = source.readString();
        this.latitude = source.readDouble();
        this.longitude = source.readDouble();
        this.notes = source.readString();
        this.postalCode = source.readString();
        this.region = source.readString();
        this.uuid = source.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.addressName);
        dest.writeInt(this.addressNumber);
        dest.writeString(this.city);
        dest.writeString(this.country);
        dest.writeString(this.doorbell);
        dest.writeString(this.floorNumber);
        dest.writeString(this.friendlyName);
        dest.writeDouble(this.latitude);
        dest.writeDouble(this.longitude);
        dest.writeString(this.notes);
        dest.writeString(this.postalCode);
        dest.writeString(this.region);
        dest.writeString(this.uuid);
    }

    /**
     * Converts the Address into a JSON Object
     *
     * @return The JSON Object created from this Address
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(ADDRESS_NAME, this.addressName);
            jObj.putOpt(ADDRESS_NUMBER, this.addressNumber);
            jObj.putOpt(CITY, this.city);
            jObj.putOpt(COUNTRY, this.country);
            jObj.putOpt(DOORBELL, this.doorbell);
            jObj.putOpt(FLOOR_NUMBER, this.floorNumber);
            jObj.putOpt(FRIENDLY_NAME, this.friendlyName);
            jObj.putOpt(LATITUDE, this.latitude);
            jObj.putOpt(LONGITUDE, this.longitude);
            jObj.putOpt(NOTES, this.notes);
            jObj.putOpt(POSTAL_CODE, this.postalCode);
            jObj.putOpt(REGION, this.region);
            jObj.putOpt(UUID, this.uuid);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the Address, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Address, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Address JSON object to String", e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================


    public String getAddressName() {
        return addressName;
    }

    public int getAddressNumber() {
        return addressNumber;
    }

    public String getCity() {
        return city;
    }

    public String getCountry() {
        return country;
    }

    public String getDoorbell() {
        return doorbell;
    }

    public String getFloorNumber() {
        return floorNumber;
    }

    public String getFriendlyName() {
        return friendlyName;
    }

    public double getLatitude() {
        return latitude;
    }

    public double getLongitude() {
        return longitude;
    }

    public String getNotes() {
        return notes;
    }

    public String getPostalCode() {
        return postalCode;
    }

    public String getRegion() {
        return region;
    }

    public String getUuid() {
        return uuid;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Address> CREATOR = new Creator<Address>() {
        public Address createFromParcel(Parcel source) {
            return new Address(source);
        }

        public Address[] newArray(int size) {
            return new Address[size];
        }
    };
}
