/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarpJSONParser;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;

public class Campaign implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String INDEX_URL = "index_url";
    private static final String LOGO_URL = "logo_url";
    private static final String ACTION = "action";
    private static final String DELIVERED = "delivered";
    private static final String EXPIRES = "expires";
    private static final String MESSAGE = "message";
    private static final String OFFER_CATEGORY = "offer_category";
    private static final String OFFER_MESSAGE = "offer_message";
    private static final String SESSION_UUID = "session_uuid";
    private static final String STARTS = "starts";
    private static final String SUBTITLE = "subtitle";
    private static final String TITLE = "title";
    private static final String SORTING = "sorting";
    private static final String OPENED = "opened";
    private static final String IS_NEW = "is_new";
    private static final String EXTRA_FIELDS = "extra_fields";
    private static final String TYPE = "campaign_type";
    private static final String EXPIRED = "expired";
    private static final String SHOW = "show";
    private static final String DELIVERY_METHOD = "delivery_method";
    private static final String DISPLAY_TYPE = "display_type";
    private static final String ACTIONS = "actions";
    private static final String BUS_ID = "busId";
    private static final String CAMPAIGN_TYPE_SETTINGS = "campaign_type_settings";
    private static final String BANNER_IMAGE = "Banner_img";
    private static final String BANNER_TITLE = "Banner_title";

    /* Member variables of the Campaign object */

    private int action;
    private double delivered;
    private double expires;
    private String message;
    private String offerCategory;
    private String offerMessage;
    private String sessionUUID;
    private double starts;
    private String subtitle;
    private String title;
    private int sorting;
    private int opened;
    private boolean isNew;
    private String logoUrl;
    private String extraFields;
    private String type;
    private boolean expired;
    private boolean show;
    private String deliveryMethod;
    private String displayType;
    private String indexUrl;
    private String busId;
    private ArrayList<CampaignAction> actions = new ArrayList<>();
    private String campaignTypeSettings;
    private String bannerImage;
    private String bannerTitle;

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public Campaign(String json) throws JSONException {
        this(new JSONObject(json));
    }

    public Campaign() {
        this.action = 0;
        this.delivered = 0.0d;
        this.expires = 0.0d;
        this.message = "";
        this.offerCategory = "";
        this.offerMessage = "";
        this.sessionUUID = "";
        this.starts = 0.0d;
        this.subtitle = "";
        this.title = "";
        this.sorting = 0;
        this.opened = 0;
        this.isNew = true;
        this.logoUrl = "";
        this.extraFields = "";
        this.type = "";
        this.expired = false;
        this.show = true;
        this.deliveryMethod = "";
        this.displayType = "";
        this.indexUrl = "";
        this.busId = "";
        this.actions = new ArrayList<>();
        this.campaignTypeSettings = "";
        this.bannerImage = "";
        this.bannerTitle = "";
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Campaign
     */
    public Campaign(JSONObject json) {
        this.indexUrl = json.optString(INDEX_URL);
        this.logoUrl = json.optString(LOGO_URL);
        this.delivered = json.optDouble(DELIVERED);
        this.sessionUUID = json.optString(SESSION_UUID);
        this.subtitle = json.optString(SUBTITLE);
        this.starts = json.optDouble(STARTS);
        this.message = json.optString(MESSAGE);
        this.opened = json.optInt(OPENED);
        this.expires = json.optDouble(EXPIRES);
        this.offerCategory = json.optString(OFFER_CATEGORY);

        if (belongsInCategory(WarpConstants.MICROAPP_CAMPAIGNS_STANDARD_OFFER_NO_DATE)) {
            this.expires = 0;
            this.offerCategory = WarpConstants.MICROAPP_CAMPAIGNS_STANDARD_OFFER;
        }

        this.title = json.optString(TITLE);
        this.action = json.optInt(ACTION);
        this.offerMessage = json.optString(OFFER_MESSAGE);
        this.isNew = json.optBoolean(IS_NEW);
        this.sorting = json.optInt(SORTING);
        this.extraFields = json.optString(EXTRA_FIELDS);
        if (!TextUtils.isEmpty(this.extraFields)) {
            JSONObject extraFieldsResp = WarpJSONParser.getJSONFromString(this.extraFields);
            if (extraFieldsResp != null) {
                if (extraFieldsResp.has("Banner_title")) {
                    this.bannerTitle = extraFieldsResp.optString(BANNER_TITLE, "");
                }
                if (extraFieldsResp.has("Banner_img")) {
                    this.bannerImage = extraFieldsResp.optString(BANNER_IMAGE, "");
                }
            }
        } else {
            this.bannerImage = "";
            this.bannerTitle = "";
        }
        this.type = json.optString(TYPE);
        this.expired = json.optBoolean(EXPIRED);
        this.show = json.optBoolean(SHOW);
        this.deliveryMethod = json.optString(DELIVERY_METHOD);
        this.displayType = json.optString(DISPLAY_TYPE);
        this.busId = json.optString(BUS_ID);
        this.campaignTypeSettings = json.optString(CAMPAIGN_TYPE_SETTINGS);

        JSONArray actions = json.optJSONArray(ACTIONS);
        if (actions != null) {
            for (int i = 0; i < actions.length(); i++) {
                this.actions.add(new CampaignAction(actions.optJSONObject(i)));
            }
        } else {
            this.actions = null;
        }
    }

    public Campaign(Parcel source) {
        this.indexUrl = source.readString();
        this.action = source.readInt();
        this.delivered = source.readDouble();
        this.expires = source.readDouble();
        this.message = source.readString();
        this.offerCategory = source.readString();
        this.offerMessage = source.readString();
        this.sessionUUID = source.readString();
        this.starts = source.readDouble();
        this.subtitle = source.readString();
        this.title = source.readString();
        this.sorting = source.readInt();
        this.opened = source.readInt();
        this.isNew = source.readInt() > 0;
        this.logoUrl = source.readString();
        this.extraFields = source.readString();
        this.type = source.readString();
        this.expired = source.readInt() > 0;
        this.show = source.readInt() > 0;
        this.deliveryMethod = source.readString();
        this.displayType = source.readString();
        this.actions = source.readArrayList(CampaignAction.class.getClassLoader());
        this.busId = source.readString();
        this.campaignTypeSettings = source.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.indexUrl);
        dest.writeInt(this.action);
        dest.writeDouble(this.delivered);
        dest.writeDouble(this.expires);
        dest.writeString(this.message);
        dest.writeString(this.offerCategory);
        dest.writeString(this.offerMessage);
        dest.writeString(this.sessionUUID);
        dest.writeDouble(this.starts);
        dest.writeString(this.subtitle);
        dest.writeString(this.title);
        dest.writeInt(this.sorting);
        dest.writeInt(this.opened);
        dest.writeInt(this.isNew ? 1 : 0);
        dest.writeString(this.logoUrl);
        dest.writeString(this.extraFields);
        dest.writeString(this.type);
        dest.writeInt(this.expired ? 1 : 0);
        dest.writeInt(this.show ? 1 : 0);
        dest.writeString(this.deliveryMethod);
        dest.writeString(this.displayType);
        dest.writeList(this.actions);
        dest.writeString(this.busId);
        dest.writeString(this.campaignTypeSettings);
    }

    /**
     * Converts the Campaign into a JSON Object
     *
     * @return The JSON Object created from this campaign
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(INDEX_URL, this.indexUrl);
            jObj.putOpt(LOGO_URL, this.logoUrl);
            jObj.putOpt(ACTION, this.action);
            jObj.putOpt(DELIVERED, this.delivered);
            jObj.putOpt(EXPIRES, this.expires);
            jObj.putOpt(MESSAGE, this.message);
            jObj.putOpt(OFFER_CATEGORY, this.offerCategory);
            jObj.putOpt(OFFER_MESSAGE, this.offerMessage);
            jObj.putOpt(SESSION_UUID, this.sessionUUID);
            jObj.putOpt(STARTS, this.starts);
            jObj.putOpt(SUBTITLE, this.subtitle);
            jObj.putOpt(TITLE, this.title);
            jObj.putOpt(SORTING, this.sorting);
            jObj.putOpt(OPENED, this.opened);
            jObj.putOpt(IS_NEW, this.isNew);
            jObj.putOpt(SORTING, this.sorting);
            jObj.putOpt(EXTRA_FIELDS, this.extraFields);
            jObj.putOpt(TYPE, this.type);
            jObj.putOpt(EXPIRED, this.expired);
            jObj.putOpt(SHOW, this.show);
            jObj.putOpt(DELIVERY_METHOD, this.deliveryMethod);
            jObj.putOpt(DISPLAY_TYPE, this.displayType);
            jObj.put(BUS_ID, this.busId);
            jObj.put(CAMPAIGN_TYPE_SETTINGS, this.campaignTypeSettings);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * Method used to check if this Campaign belongs to the given category
     *
     * @param category The category to be checked
     * @return True if it belongs, false otherwise
     */
    public boolean belongsInCategory(String category) {
        return offerCategory != null && offerCategory.equalsIgnoreCase(category);
    }

    public boolean belongsInType(String type) {
        return type != null && type.equalsIgnoreCase(type);
    }

    /**
     * Method used to return if a Campaign belongs in a category, after
     * splitting the category using the given delimiter.
     *
     * @param category  The category to be checked
     * @param delimiter The delimiter to split the category string
     * @return True or false
     */
    public boolean belongsInCategory(String category, String delimiter) {
        if (offerCategory == null)
            return false;
        String[] categories = offerCategory.split(delimiter);
        for (String c : categories) {
            if (c.equalsIgnoreCase(category)) {
                return true;
            }
        }
        return false;
    }

    /**
     * String representation of the Campaign, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Campaign, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Campaign JSON object to String",
                    e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================

    /**
     * @return The loyalty action of the Campaign, used if custom actions are
     * implemented
     */
    public int getAction() {
        return action;
    }

    /**
     * @return The time that the campaign was delivered, measured in
     * milliseconds
     */
    public double getDelivered() {
        return delivered;
    }

    /**
     * @return The time that the campaign expires, measured in milliseconds
     */
    public double getExpires() {
        return expires;
    }

    /**
     * @return The message of the campaign
     */
    public String getMessage() {
        return message;
    }

    /**
     * @return The category that this campaign belongs, or multiple categories
     * separated by delimiters
     */
    public String getOfferCategory() {
        return offerCategory;
    }

    /**
     * @return The message shown when a push arrives
     */
    public String getOfferMessage() {
        return offerMessage;
    }

    /**
     * @return The sessionUUID, used to retrieve campaign's content from Warply
     * server
     */
    public String getSessionUUID() {
        return sessionUUID;
    }

    /**
     * @return The time that the campaign started, measured in milliseconds
     */
    public double getStarts() {
        return starts;
    }

    /**
     * @return The subtitle of the campaign
     */
    public String getSubtitle() {
        return subtitle;
    }

    /**
     * @return The title of the campaign
     */
    public String getTitle() {
        return title;
    }

    /**
     * @return An int that is used for sorting. This should be used as the
     * default sorting value
     */
    public int getSorting() {
        return sorting;
    }

    /**
     * @return The times this campaigns has been opened
     */
    public int getOpened() {
        return opened;
    }

    /**
     * Method used to get the url of the image of the campaign
     *
     * @return The absolute path to the URL as a String
     */
    public String getImageUrl() {
        return WarplyProperty.getBaseUrl(Warply.getWarplyContext()) + WarpConstants.BASE_REMOTE_PAGE_URL + "logo/" + sessionUUID;
    }

    /**
     * Method used to get if the campaign is a new one
     *
     * @return <code>true</code> if the campaign is new; <code>false</code>
     * otherwise
     */
    public boolean getIsNew() {
        return this.isNew;
    }

    public String getLogoUrl() {
        return logoUrl;
    }

    public String getIndexUrl() {
        return indexUrl;
    }

    public String getExtraFields() {
        return extraFields;
    }

    public String getType() {
        return type;
    }

    public boolean isExpired() {
        return expired;
    }

    public boolean isShow() {
        return show;
    }

    public String getDeliveryMethod() {
        return deliveryMethod;
    }

    public String getDisplayType() {
        return displayType;
    }

    public ArrayList<CampaignAction> getActions() {
        return actions;
    }

    public String getBusId() {
        return busId;
    }

    public void setBusId(String busId) {
        this.busId = busId;
    }

    public void setNew(boolean aNew) {
        isNew = aNew;
    }

    public String getCampaignTypeSettings() {
        return campaignTypeSettings;
    }

    public void setExpires(double expires) {
        this.expires = expires;
    }

    public void setOfferCategory(String offerCategory) {
        this.offerCategory = offerCategory;
    }

    public void setShow(boolean show) {
        this.show = show;
    }

    public void setDeliveryMethod(String deliveryMethod) {
        this.deliveryMethod = deliveryMethod;
    }

    public void setDisplayType(String displayType) {
        this.displayType = displayType;
    }

    public void setIndexUrl(String indexUrl) {
        this.indexUrl = indexUrl;
    }

    public void setActions(ArrayList<CampaignAction> actions) {
        this.actions = actions;
    }

    public void setAction(int action) {
        this.action = action;
    }

    public void setDelivered(double delivered) {
        this.delivered = delivered;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    public void setOfferMessage(String offerMessage) {
        this.offerMessage = offerMessage;
    }

    public void setSessionUUID(String sessionUUID) {
        this.sessionUUID = sessionUUID;
    }

    public void setStarts(double starts) {
        this.starts = starts;
    }

    public void setSubtitle(String subtitle) {
        this.subtitle = subtitle;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public void setSorting(int sorting) {
        this.sorting = sorting;
    }

    public void setOpened(int opened) {
        this.opened = opened;
    }

    public void setLogoUrl(String logoUrl) {
        this.logoUrl = logoUrl;
    }

    public void setExtraFields(String extraFields) {
        this.extraFields = extraFields;
    }

    public void setType(String type) {
        this.type = type;
    }

    public void setExpired(boolean expired) {
        this.expired = expired;
    }

    public void setCampaignTypeSettings(String campaignTypeSettings) {
        this.campaignTypeSettings = campaignTypeSettings;
    }

    public String getBannerImage() {
        return this.bannerImage;
    }

    public String getBannerTitle() {
        return this.bannerTitle;
    }

    public void setBannerImage(String bannerImage) {
        this.bannerImage = bannerImage;
    }

    public void setBannerTitle(String bannerTitle) {
        this.bannerTitle = bannerTitle;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Parcelable.Creator<Campaign> CREATOR = new Parcelable.Creator<Campaign>() {
        public Campaign createFromParcel(Parcel source) {
            return new Campaign(source);
        }

        public Campaign[] newArray(int size) {
            return new Campaign[size];
        }
    };
}
