/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import androidx.annotation.NonNull;
import android.text.TextUtils;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;

public class CampaignList extends ArrayList<Campaign> {

    /**
     * Generated for serialized class
     */
    private static final long serialVersionUID = -188843583823948267L;

    private static final String JSON_KEY_MAPP = "MAPP_OFFER";
    private static final String JSON_KEY_CONTEXT = "context";

    private String mRequestSignature = "";

    private CampaignList(String requestSignature) {
        this.mRequestSignature = requestSignature;
    }

    public CampaignList() {
        super();
    }

    /**
     * Constructor used to create the CampaignList from a JSON Object.
     *
     * @param campaignListJSONObject The JSON Object, used to create the CampaignList
     */
    public CampaignList(JSONObject campaignListJSONObject, String requestSignature) {
        this(requestSignature);

        campaignListJSONObject = campaignListJSONObject
                .optJSONObject(JSON_KEY_CONTEXT);
        if (campaignListJSONObject == null)
            return;
        JSONArray jArray = campaignListJSONObject
                .optJSONArray(JSON_KEY_MAPP);
        if (jArray != null) {
            for (int i = 0, lim = jArray.length(); i < lim; ++i) {
                add(new Campaign(jArray.optJSONObject(i)));
            }
        }
    }

    /**
     * Constructor to create the CampaignList from a String encoded in JSON
     * format
     *
     * @param stringJSON A String containing the Campaigns in JSON format
     */
    /*public CampaignList(String stringJSON) throws JSONException {
        this(new JSONObject(stringJSON));
    }*/

    /**
     * Method used to return a CampaignList, containing all the Campaigns,
     * belonging to at least one of the given categories
     *
     * @param categories The categories used for filtering
     * @return A new filtered CampaignList with the Campaigns belonging to at
     * lease one the given categories
     */
    public CampaignList filterCampaignsInCategories(String... categories) {

        CampaignList filtered = new CampaignList(mRequestSignature);
        for (Campaign campaign : this) {

            for (String category : categories) {
                if (campaign.belongsInCategory(category)) {
                    filtered.add(campaign);
                    break;
                }
            }
        }
        return filtered;
    }

    /**
     * Method used to return a CampaignList, filtering out all the Campaigns,
     * belonging to at least one of the given categories
     *
     * @param categories The categories used for filtering
     * @return A new filtered CampaignList with the Campaigns belonging to at
     * lease one the given categories
     */
    public CampaignList filterCampaignsOutCategories(String... categories) {

        CampaignList filtered = new CampaignList(mRequestSignature);
        boolean belongToCategory;
        for (Campaign campaign : this) {

            belongToCategory = false;
            for (String category : categories) {
                if (campaign.belongsInCategory(category)) {
                    belongToCategory = true;
                    break;
                }
            }
            if (!belongToCategory) {
                filtered.add(campaign);
            }
        }
        return filtered;
    }

    public int getUnreadCampaignsCount() {

        int count = 0;
        for (int i = 0; i < size(); i++) {
            if (get(i).getOpened() == 0) {
                count++;
            }
        }
        return count;
    }

    public boolean containsUuid(String campaignUuid) {

        if (!TextUtils.isEmpty(campaignUuid)) {
            for (int i = 0; i < size(); i++) {
                if (campaignUuid.equals(get(i).
                        getSessionUUID())) {
                    return true;
                }
            }
        }
        return false;
    }

    @NonNull
    public String getRequestSignature() {
        return mRequestSignature;
    }
}
