/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 11-Jan-22.
 */

public class Card implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String ACTIVE = "active";
    private static final String CARD_ISSUER = "card_issuer";
    private static final String CARD_NUMBER = "card_number";
    private static final String CARD_HOLDER = "cardholder_name";
    private static final String CARD_EXTRA_FIELDS = "extra_fields";
    private static final String CARD_LAST_USED = "last_used";
    private static final String CARD_TOKEN = "token";

    /* Member variables of the Campaign object */

    private boolean active = false;
    private String cardIssuer = "";
    private String cardNumber = "";
    private String cardHolder = "";
    private JSONObject cardExtraFields = new JSONObject();
    private String cardLastUsed = "";
    private String cardToken = "";

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public Card(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Card
     */
    public Card(JSONObject json) {
        if (json != null) {
            this.active = json.optBoolean(ACTIVE);
            this.cardIssuer = json.optString(CARD_ISSUER);
            this.cardNumber = json.optString(CARD_NUMBER);
            this.cardHolder = json.optString(CARD_HOLDER);
            this.cardExtraFields = json.optJSONObject(CARD_EXTRA_FIELDS);
            this.cardLastUsed = json.optString(CARD_LAST_USED);
            this.cardToken = json.optString(CARD_TOKEN);
        }
    }

    public Card(Parcel source) {
        this.active = source.readByte() != 0;
        this.cardIssuer = source.readString();
        this.cardNumber = source.readString();
        this.cardHolder = source.readString();
        this.cardLastUsed = source.readString();
        this.cardToken = source.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeByte((byte) (this.active ? 1 : 0));
        dest.writeString(this.cardIssuer);
        dest.writeString(this.cardNumber);
        dest.writeString(this.cardHolder);
        dest.writeString(this.cardLastUsed);
        dest.writeString(this.cardToken);
    }

    /**
     * Converts the Card into a JSON Object
     *
     * @return The JSON Object created from this Card
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(ACTIVE, this.active);
            jObj.putOpt(CARD_ISSUER, this.cardIssuer);
            jObj.putOpt(CARD_NUMBER, this.cardNumber);
            jObj.putOpt(CARD_HOLDER, this.cardHolder);
            jObj.putOpt(CARD_EXTRA_FIELDS, this.cardExtraFields);
            jObj.putOpt(CARD_LAST_USED, this.cardLastUsed);
            jObj.putOpt(CARD_TOKEN, this.cardToken);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the Card, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Card, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Card JSON object to String", e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================


    public boolean isActive() {
        return active;
    }

    public String getCardIssuer() {
        return cardIssuer;
    }

    public String getCardNumber() {
        return cardNumber;
    }

    public String getCardHolder() {
        return cardHolder;
    }

    public JSONObject getCardExtraFields() {
        return cardExtraFields;
    }

    public String getCardLastUsed() {
        return cardLastUsed;
    }

    public String getCardToken() {
        return cardToken;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Card> CREATOR = new Creator<Card>() {
        public Card createFromParcel(Parcel source) {
            return new Card(source);
        }

        public Card[] newArray(int size) {
            return new Card[size];
        }
    };
}
