/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 07-Dec-21.
 */

public class Content implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String NAME = "name";
    private static final String ID = "id";
    private static final String ID_ID = "id_id";
    private static final String DESCRIPTION = "description";
    private static final String SHORT_DESCRIPTION = "short_description";
    private static final String ACTIVE = "active";
    private static final String CREATED = "created";
    private static final String CATEGORY_ID = "category_id";
    private static final String CATEGORY_NAME = "category_name";
    private static final String CATEGORY_UUID = "category_uuid";
    private static final String MERCHANT_UUID = "merchant_uuid";
    private static final String TAGS = "tags";
    private static final String START_DATE = "start_date";
    private static final String END_DATE = "end_date";
    private static final String IMG_PREVIEW = "img_preview";
    private static final String IMG = "img";
    private static final String FAVOURITE = "favourite";
    private static final String EXTRA_FIELDS = "extra_fields";
    private static final String PARENT = "parent";
    private static final String SORTING = "sorting";
    private static final String CONSUMER_FULL_NAME = "consumer_full_name";
    private static final String CONSUMER_PHOTO = "consumer_photo";

    /* Member variables of the Campaign object */

    private String name = "";
    private String id = "";
    private int id_id = 0;
    private String description = "";
    private String short_description = "";
    private boolean active;
    private String created = "";
    private int category_id = 0;
    private String category_name = "";
    private String category_uuid = "";
    private String merchant_uuid = "";
    private JSONArray tags = new JSONArray();
    private String start_date = "";
    private String end_date = "";
    private String img_preview = "";
    private String img = "";
    private boolean favourite;
    private JSONObject extra_fields = new JSONObject();
    private String parent = "";
    private int sorting = 0;
    private String consumer_full_name = "";
    private String consumer_photo = "";

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public Content(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Content
     */
    public Content(JSONObject json) {
        if (json != null) {
            this.name = json.optString(NAME);
            this.id = json.optString(ID);
            this.id_id = json.optInt(ID_ID);
            this.description = json.optString(DESCRIPTION);
            this.short_description = json.optString(SHORT_DESCRIPTION);
            this.active = json.optBoolean(ACTIVE);
            this.created = json.optString(CREATED);
            this.category_id = json.optInt(CATEGORY_ID);
            this.category_name = json.optString(CATEGORY_NAME);
            this.category_uuid = json.optString(CATEGORY_UUID);
            this.merchant_uuid = json.optString(MERCHANT_UUID);
            this.tags = json.optJSONArray(TAGS);
            this.start_date = json.optString(START_DATE);
            this.end_date = json.optString(END_DATE);
            this.img_preview = json.optString(IMG_PREVIEW);
            this.img = json.optString(IMG);
            this.favourite = json.optBoolean(FAVOURITE);
            this.extra_fields = json.optJSONObject(EXTRA_FIELDS);
            this.parent = json.optString(PARENT);
            this.sorting = json.optInt(SORTING);
            this.consumer_full_name = json.optString(CONSUMER_FULL_NAME);
            this.consumer_photo = json.optString(CONSUMER_PHOTO);
        }
    }

    public Content(Parcel source) {
        this.name = source.readString();
        this.id = source.readString();
        this.id_id = source.readInt();
        this.description = source.readString();
        this.short_description = source.readString();
        this.active = source.readByte() != 0;
        this.created = source.readString();
        this.category_id = source.readInt();
        this.category_name = source.readString();
        this.category_uuid = source.readString();
        this.merchant_uuid = source.readString();
        this.start_date = source.readString();
        this.end_date = source.readString();
        this.img_preview = source.readString();
        this.img = source.readString();
        this.favourite = source.readByte() != 0;
        this.parent = source.readString();
        this.sorting = source.readInt();
        this.consumer_full_name = source.readString();
        this.consumer_photo = source.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.name);
        dest.writeString(this.id);
        dest.writeInt(this.id_id);
        dest.writeString(this.description);
        dest.writeString(this.short_description);
        dest.writeByte((byte) (this.active ? 1 : 0));
        dest.writeString(this.created);
        dest.writeInt(this.category_id);
        dest.writeString(this.category_name);
        dest.writeString(this.category_uuid);
        dest.writeString(this.merchant_uuid);
        dest.writeString(this.start_date);
        dest.writeString(this.end_date);
        dest.writeString(this.img_preview);
        dest.writeString(this.img);
        dest.writeByte((byte) (this.favourite ? 1 : 0));
        dest.writeString(this.parent);
        dest.writeInt(this.sorting);
        dest.writeString(this.consumer_full_name);
        dest.writeString(this.consumer_photo);
    }

    /**
     * Converts the Content into a JSON Object
     *
     * @return The JSON Object created from this Content
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(NAME, this.name);
            jObj.putOpt(ID, this.id);
            jObj.putOpt(ID_ID, this.id_id);
            jObj.putOpt(DESCRIPTION, this.description);
            jObj.putOpt(SHORT_DESCRIPTION, this.short_description);
            jObj.putOpt(ACTIVE, this.active);
            jObj.putOpt(CREATED, this.created);
            jObj.putOpt(CATEGORY_ID, this.category_id);
            jObj.putOpt(CATEGORY_NAME, this.category_name);
            jObj.putOpt(CATEGORY_UUID, this.category_uuid);
            jObj.putOpt(MERCHANT_UUID, this.merchant_uuid);
            jObj.putOpt(TAGS, this.tags);
            jObj.putOpt(START_DATE, this.start_date);
            jObj.putOpt(END_DATE, this.end_date);
            jObj.putOpt(IMG_PREVIEW, this.img_preview);
            jObj.putOpt(IMG, this.img);
            jObj.putOpt(FAVOURITE, this.favourite);
            jObj.putOpt(EXTRA_FIELDS, this.extra_fields);
            jObj.putOpt(PARENT, this.parent);
            jObj.putOpt(SORTING, this.sorting);
            jObj.putOpt(CONSUMER_FULL_NAME, this.consumer_full_name);
            jObj.putOpt(CONSUMER_PHOTO, this.consumer_photo);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the Content, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Content, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Content JSON object to String", e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================


    public String getName() {
        return name;
    }

    public String getId() {
        return id;
    }

    public int getIdId() {
        return id_id;
    }

    public String getDescription() {
        return description;
    }

    public String getShortDescription() {
        return short_description;
    }

    public boolean isActive() {
        return active;
    }

    public String getCreated() {
        return created;
    }

    public int getCategoryId() {
        return category_id;
    }

    public String getCategoryName() {
        return category_name;
    }

    public String getCategoryUuid() {
        return category_uuid;
    }

    public String getMerchantUuid() {
        return merchant_uuid;
    }

    public String getStartDate() {
        return start_date;
    }

    public String getEndDate() {
        return end_date;
    }

    public String getImgPreview() {
        return img_preview;
    }

    public String getImage() {
        return img;
    }

    public boolean isFavourite() {
        return favourite;
    }

    public JSONObject getExtraFields() {
        return extra_fields;
    }

    public String getParent() {
        return parent;
    }

    public int getSorting() {
        return sorting;
    }

    public String getConsumerFullName() {
        return consumer_full_name;
    }

    public String getConsumerPhoto() {
        return consumer_photo;
    }

    public JSONArray getTags() {
        return tags;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Content> CREATOR = new Creator<Content>() {
        public Content createFromParcel(Parcel source) {
            return new Content(source);
        }

        public Content[] newArray(int size) {
            return new Content[size];
        }
    };
}
