/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;
import android.view.View;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 14-Jan-22.
 */

public class Coupon implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String BARCODE = "barcode";
    private static final String CATEGORY = "category";
    private static final String COUPON = "coupon";
    private static final String CREATED = "created";
    private static final String DESCRIPTION = "description";
    private static final String DISCOUNT = "discount";
    private static final String EXPIRATION = "expiration";
    private static final String IMAGE = "image";
    private static final String NAME = "name";
    private static final String STATUS = "status";
    private static final String TRANSACTION_DATE = "transaction_date";
    private static final String TRANSACTION_UUID = "transaction_uuid";
    private static final String CHANGES_DATES = "changes_dates";
    private static final String COUPONSET_UUID = "couponset_uuid";
    private static final String MERCHANT_UUID = "merchant_uuid";
    private static final String INNER_TEXT = "inner_text";
    private static final String FINAL_PRICE = "final_price";
    private static final String DISCOUNT_TYPE = "discount_type";
    private static final String SHORT_DESCRIPTION = "short_description";
    private static final String TERMS = "terms";

    /* Member variables of the Campaign object */

    private String barcode = "";
    private String category = "";
    private String coupon = "";
    private String created = "";
    private String description = "";
    private String discount = "";
    private String expiration = "";
    private String image = "";
    private String name = "";
    private int status = 0;
    private String transactionDate = "";
    private String transactionUuid = "";
    private JSONObject changesDates = new JSONObject();
    private String couponsetUuid = "";
    private String merchantUuid = "";
    private String innerText = "";
    private Date expirationDate = new Date();
    private Date redeemDate = new Date();
    private String discount_type = "";
    private double final_price = 0.0d;
    private String short_description = "";
    private String terms = "";

    public Coupon() {
        this.barcode = "";
        this.category = "";
        this.coupon = "";
        this.created = "";
        this.description = "";
        this.discount = "";
        this.expiration = "";
        this.image = "";
        this.name = "";
        this.status = 0;
        this.transactionDate = "";
        this.transactionUuid = "";
        this.changesDates = new JSONObject();
        this.couponsetUuid = "";
        this.merchantUuid = "";
        this.innerText = "";
        this.expirationDate = new Date();
        this.redeemDate = new Date();
        this.discount_type = "";
        this.final_price = 0.0d;
        this.short_description = "";
        this.terms = "";
    }

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public Coupon(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Coupon
     */
    public Coupon(JSONObject json) {
        if (json != null) {
            this.barcode = json.optString(BARCODE);
            this.category = json.optString(CATEGORY);
            this.coupon = json.optString(COUPON);
            this.created = json.optString(CREATED);
            this.description = json.optString(DESCRIPTION);
            this.discount = json.optString(DISCOUNT);
            if (this.discount.contains(",")) {
                this.discount = this.discount.replace(",", ".");
            }
            this.expiration = json.optString(EXPIRATION);
            this.image = json.optString(IMAGE);
            this.name = json.optString(NAME);
            this.status = json.optInt(STATUS);
            this.transactionDate = json.optString(TRANSACTION_DATE);
            this.transactionUuid = json.optString(TRANSACTION_UUID);
            this.changesDates = json.optJSONObject(CHANGES_DATES);
            if (this.changesDates != null) {
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd hh:mm");
                Date newDate = new Date();
                String tempRedeemDate = this.changesDates.optString("redeemed");
                try {
                    newDate = simpleDateFormat.parse(tempRedeemDate);
                    this.redeemDate = newDate;
                } catch (ParseException e) {
                    e.printStackTrace();
                }
            }
            this.couponsetUuid = json.optString(COUPONSET_UUID);
            this.merchantUuid = json.optString(MERCHANT_UUID);
            this.innerText = json.optString(INNER_TEXT);
            this.discount_type = json.isNull(DISCOUNT_TYPE) ? "" : json.optString(DISCOUNT_TYPE);
            this.final_price = json.optDouble(FINAL_PRICE);
            this.short_description = json.optString(SHORT_DESCRIPTION);
            this.terms = json.optString(TERMS);
        }
    }

    public Coupon(Parcel source) {
        this.barcode = source.readString();
        this.category = source.readString();
        this.coupon = source.readString();
        this.created = source.readString();
        this.description = source.readString();
        this.discount = source.readString();
        this.expiration = source.readString();
        this.image = source.readString();
        this.name = source.readString();
        this.status = source.readInt();
        this.transactionDate = source.readString();
        this.transactionUuid = source.readString();
        this.couponsetUuid = source.readString();
        this.merchantUuid = source.readString();
        this.innerText = source.readString();
        this.discount_type = source.readString();
        this.final_price = source.readDouble();
        this.short_description = source.readString();
        this.terms = source.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.barcode);
        dest.writeString(this.category);
        dest.writeString(this.coupon);
        dest.writeString(this.created);
        dest.writeString(this.description);
        dest.writeString(this.discount);
        dest.writeString(this.expiration);
        dest.writeString(this.image);
        dest.writeString(this.name);
        dest.writeInt(this.status);
        dest.writeString(this.transactionDate);
        dest.writeString(this.transactionUuid);
        dest.writeString(this.couponsetUuid);
        dest.writeString(this.merchantUuid);
        dest.writeString(this.innerText);
        dest.writeString(this.discount_type);
        dest.writeDouble(this.final_price);
        dest.writeString(this.short_description);
        dest.writeString(this.terms);
    }

    /**
     * Converts the Coupon into a JSON Object
     *
     * @return The JSON Object created from this Coupon
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(BARCODE, this.barcode);
            jObj.putOpt(CATEGORY, this.category);
            jObj.putOpt(COUPON, this.coupon);
            jObj.putOpt(CREATED, this.created);
            jObj.putOpt(DESCRIPTION, this.description);
            jObj.putOpt(DISCOUNT, this.discount);
            jObj.putOpt(EXPIRATION, this.expiration);
            jObj.putOpt(IMAGE, this.image);
            jObj.putOpt(NAME, this.name);
            jObj.putOpt(STATUS, this.status);
            jObj.putOpt(TRANSACTION_DATE, this.transactionDate);
            jObj.putOpt(TRANSACTION_UUID, this.transactionUuid);
            jObj.putOpt(CHANGES_DATES, this.changesDates);
            jObj.putOpt(COUPONSET_UUID, this.couponsetUuid);
            jObj.putOpt(MERCHANT_UUID, this.merchantUuid);
            jObj.putOpt(INNER_TEXT, this.innerText);
            jObj.putOpt(DISCOUNT_TYPE, this.discount_type);
            jObj.putOpt(FINAL_PRICE, this.final_price);
            jObj.putOpt(SHORT_DESCRIPTION, this.short_description);
            jObj.putOpt(TERMS, this.terms);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the Coupon, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Coupon, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Coupon JSON object to String", e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================


    public String getBarcode() {
        return barcode;
    }

    public String getCategory() {
        return category;
    }

    public String getCoupon() {
        return coupon;
    }

    public String getCreated() {
        return created;
    }

    public String getDescription() {
        return description;
    }

    public String getDiscount() {
        return discount;
    }

    public String getExpiration() {
        return expiration;
    }

    public String getImage() {
        return image;
    }

    public String getName() {
        return name;
    }

    public int getStatus() {
        return status;
    }

    public String getTransactionDate() {
        return transactionDate;
    }

    public String getTransactionUuid() {
        return transactionUuid;
    }

    public JSONObject getChangesDates() {
        return changesDates;
    }

    public String getCouponsetUuid() {
        return couponsetUuid;
    }

    public void setBarcode(String barcode) {
        this.barcode = barcode;
    }

    public void setCategory(String category) {
        this.category = category;
    }

    public void setCoupon(String coupon) {
        this.coupon = coupon;
    }

    public void setCreated(String created) {
        this.created = created;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public void setDiscount(String discount) {
        this.discount = discount;
    }

    public void setExpiration(String expiration) {
        this.expiration = expiration;
    }

    public void setImage(String image) {
        this.image = image;
    }

    public void setName(String name) {
        this.name = name;
    }

    public void setStatus(int status) {
        this.status = status;
    }

    public void setTransactionDate(String transactionDate) {
        this.transactionDate = transactionDate;
    }

    public void setTransactionUuid(String transactionUuid) {
        this.transactionUuid = transactionUuid;
    }

    public void setChangesDates(JSONObject changesDates) {
        this.changesDates = changesDates;
    }

    public void setCouponsetUuid(String couponsetUuid) {
        this.couponsetUuid = couponsetUuid;
    }

    public String getMerchantUuid() {
        return merchantUuid;
    }

    public void setMerchantUuid(String merchantUuid) {
        this.merchantUuid = merchantUuid;
    }

    public String getInnerText() {
        return innerText;
    }

    public void setInnerText(String innerText) {
        this.innerText = innerText;
    }

    public Date getExpirationDate() {
        return expirationDate;
    }

    public void setExpirationDate(Date expirationDate) {
        this.expirationDate = expirationDate;
    }

    public Date getRedeemDate() {
        return redeemDate;
    }

    public void setRedeemDate(Date redeemDate) {
        this.redeemDate = redeemDate;
    }

    public String getDiscount_type() {
        return discount_type;
    }

    public void setDiscount_type(String discount_type) {
        this.discount_type = discount_type;
    }

    public double getFinal_price() {
        return final_price;
    }

    public void setFinal_price(double final_price) {
        this.final_price = final_price;
    }

    public String getShort_description() {
        return short_description;
    }

    public void setShort_description(String short_description) {
        this.short_description = short_description;
    }

    public String getTerms() {
        return terms;
    }

    public void setTerms(String terms) {
        this.terms = terms;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Coupon> CREATOR = new Creator<Coupon>() {
        public Coupon createFromParcel(Parcel source) {
            return new Coupon(source);
        }

        public Coupon[] newArray(int size) {
            return new Coupon[size];
        }
    };
}
