/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.Warply;

import org.json.JSONException;
import org.json.JSONObject;

public class InboxStats {

	/* Constants used to export the campaign in JSON formal and vice versa */

	private static final String TOTAL_COUNT = "count";
	private static final String NEW_COUNT = "new";
	private static final String UNREAD_COUNT = "unread";

	/* Member variables of the Campaign object */

	private int totalCount;
	private int newCount;
	private int unreadCount;

	/**
	 * Basic constructor using a String, representing a JSON Object
	 * 
	 * @param json
	 *            The String, representing the JSON Object
	 * @throws JSONException
	 *             Thrown if the String cannot be converted to JSON
	 */
	public InboxStats(String json) throws JSONException {
		this(new JSONObject(json));
	}

	/**
	 * Constructor used to create an IboxStats object from a given JSON object
	 * 
	 * @param jObj
	 *            JSON Object used to create the Campaign
	 */
	public InboxStats(JSONObject jObj) {
		JSONObject inboxStatus = null;
		try {
			inboxStatus = jObj.getJSONObject("context").getJSONObject("MAPP_OFFER");
		} catch (JSONException e) {
			WarpUtils.warn("Exception on parsing Inbox Status JSON", e);
		}
		totalCount = 0;
		newCount = 0;
		unreadCount = 0;
		if (inboxStatus != null) {
			if (inboxStatus.has(TOTAL_COUNT))
				totalCount = inboxStatus.optInt(TOTAL_COUNT);
			if (inboxStatus.has(NEW_COUNT))
				newCount = inboxStatus.optInt(NEW_COUNT);
			if (inboxStatus.has(UNREAD_COUNT))
				unreadCount = inboxStatus.optInt(UNREAD_COUNT);
		}
	}

	/**
	 * Converts IboxStats into a JSON Object
	 * 
	 * @return The JSON Object created
	 */
	public JSONObject toJSONObject() {
		JSONObject jObj = new JSONObject();
		try {
			jObj.putOpt(TOTAL_COUNT, totalCount);
			jObj.putOpt(NEW_COUNT, newCount);
			jObj.putOpt(UNREAD_COUNT, unreadCount);
		} catch (JSONException e) {
		}
		return jObj;
	}

	/**
	 * String representation of the IbnoxStats, as a JSON object
	 */
	@Override
	public String toString() {
		return toJSONObject().toString();
	}

	/**
	 * String representation of the IboxStats, as a human readable JSON object
	 * 
	 * @return A human readable String representation
	 */
	public String toHumanReadableString() {
		String s = null;
		try {
			s = toJSONObject().toString(2);
		} catch (JSONException e) {
		}
		return s;
	}

	/**
	 * The total number of active campaigns - This is the number of campaigns
	 * that the {@link Warply#getInbox(ly.warp.sdk.io.callbacks.CallbackReceiver)
	 * Warply.getInbox} call with return
	 */
	public int getTotalCount() {
		return totalCount;
	}

	/**
	 * The number of campaigns that have never been returned with a call to
	 * {@link Warply#getInbox(ly.warp.sdk.io.callbacks.CallbackReceiver) Warply.getInbox}
	 */
	public int getNewCount() {
		return newCount;
	}

	/**
	 * The number of campaigns that have never been opened
	 */
	public int getUnreadCount() {
		return unreadCount;
	}

}
