/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 07-Dec-21.
 */

public class MerchantCategory implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String UUID = "uuid";
    private static final String ADMIN_NAME = "admin_name";
    private static final String PARENT = "parent";
    private static final String FIELDS = "fields";
    private static final String CHILDREN = "children";
    private static final String COUNT = "count";
    private static final String NAME = "name";

    /* Member variables of the Campaign object */

    private String uuid = "";
    private String admin_name = "";
    private String parent = "";
    private JSONArray fields = new JSONArray();
    private JSONArray children = new JSONArray();
    private int count = 0;
    private String name = "";


    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public MerchantCategory(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the MerchantCategory
     */
    public MerchantCategory(JSONObject json) {
        if (json != null) {
            this.uuid = json.optString(UUID);
            this.admin_name = json.optString(ADMIN_NAME);
            this.parent = json.optString(PARENT);
            try {
                this.fields = new JSONArray(json.optString(FIELDS));
            } catch (JSONException e) {
                this.fields = new JSONArray();
            }
            this.children = json.optJSONArray(CHILDREN);
            this.count = json.optInt(COUNT);
            this.name = json.optString(NAME);
        }
    }

    public MerchantCategory(Parcel source) {
        this.uuid = source.readString();
        this.admin_name = source.readString();
        this.parent = source.readString();
        this.count = source.readInt();
        this.name = source.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.uuid);
        dest.writeString(this.admin_name);
        dest.writeString(this.parent);
        dest.writeInt(this.count);
        dest.writeString(this.name);
    }

    /**
     * Converts the MerchantCategory into a JSON Object
     *
     * @return The JSON Object created from this MerchantCategory
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(UUID, this.uuid);
            jObj.putOpt(ADMIN_NAME, this.admin_name);
            jObj.putOpt(PARENT, this.parent);
            jObj.putOpt(FIELDS, this.fields);
            jObj.putOpt(CHILDREN, this.children);
            jObj.putOpt(COUNT, this.count);
            jObj.putOpt(NAME, this.name);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the MerchantCategory, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the MerchantCategory, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting MerchantCategory JSON object to String", e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================


    public String getUuid() {
        return uuid;
    }

    public String getAdminName() {
        return admin_name;
    }

    public String getParent() {
        return parent;
    }

    public JSONArray getFields() {
        return fields;
    }

    public JSONArray getChildren() {
        return children;
    }

    public int getCount() {
        return count;
    }

    public String getName() {
        return name;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<MerchantCategory> CREATOR = new Creator<MerchantCategory>() {
        public MerchantCategory createFromParcel(Parcel source) {
            return new MerchantCategory(source);
        }

        public MerchantCategory[] newArray(int size) {
            return new MerchantCategory[size];
        }
    };
}
