/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 12-May-22.
 */

public class NewCampaign implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */
    private static final String SUBTITLE = "subtitle";
    private static final String TITLE = "title";
    private static final String COMMUNICATION_UUID = "communication_uuid";
    private static final String MESSAGE = "message";
    private static final String DESCRIPTION = "description";
    private static final String INDEX_URL = "index_url";
    private static final String LOGO_URL = "logo_url";
    private static final String START_DATE = "start_date";
    private static final String END_DATE = "end_date";
    private static final String SORTING = "sorting";
    private static final String CATEGORY = "category";
    private static final String COMMUNICATION_NAME = "communication_name";
    private static final String COMMUNICATION_CATEGORY = "communication_category";
    private static final String EXTRA_FIELDS = "extra_fields";
    private static final String WORKFLOW_SETTINGS = "workflow_settings";
    private static final String DISPLAY_TYPE = "display_type";
    private static final String DELIVERY_METHOD = "delivery_method";
    private static final String CAMPAIGN_TYPE = "campaign_type";
    private static final String SETTINGS = "settings";
    private static final String AUDIENCE = "audience";
    private static final String IS_NEW = "is_new";
    private static final String BANNER_IMAGE = "Banner_img";
    private static final String BANNER_TITLE = "Banner_title";

    /* Member variables of the Campaign object */
    private final String campaignType;
    private JSONObject settings = new JSONObject();
    private final String audience;
    private final String deliveryMethod;
    private final String displayType;
    private JSONObject extraFields = new JSONObject();
    private JSONObject workflowSettings = new JSONObject();
    private final String category;
    private final int sorting;
    private final String indexUrl;
    private final String message;
    private final String description;
    private final String subtitle;
    private final String title;
    private final String communicationUUID;
    private final String logoUrl;
    private final String startDate;
    private final String endDate;
    private final String communicationName;
    private final String communicationCategory;
    private boolean isNew = false;
    private String bannerImage = "";
    private String bannerTitle = "";

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public NewCampaign(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Campaign
     */
    public NewCampaign(JSONObject json) {
        this.indexUrl = json.optString(INDEX_URL);
        this.logoUrl = json.optString(LOGO_URL);
        this.communicationUUID = json.optString(COMMUNICATION_UUID);
        this.subtitle = json.isNull(SUBTITLE) ? null : json.optString(SUBTITLE);
        this.message = json.isNull(MESSAGE) ? null : json.optString(MESSAGE);
        this.category = json.optString(CATEGORY);
        this.title = json.optString(TITLE);
        this.sorting = json.optInt(SORTING);
        this.extraFields = json.optJSONObject(EXTRA_FIELDS);
        this.campaignType = json.optString(CAMPAIGN_TYPE);
        this.deliveryMethod = json.optString(DELIVERY_METHOD);
        this.displayType = json.optString(DISPLAY_TYPE);
        this.settings = json.optJSONObject(SETTINGS);
        this.audience = json.optString(AUDIENCE);
        this.workflowSettings = json.optJSONObject(WORKFLOW_SETTINGS);
        this.description = json.optString(DESCRIPTION);
        this.startDate = json.optString(START_DATE);
        this.endDate = json.optString(END_DATE);
        this.communicationName = json.optString(COMMUNICATION_NAME);
        this.communicationCategory = json.optString(COMMUNICATION_CATEGORY);
        this.isNew = json.optBoolean(IS_NEW);
        if (this.extraFields != null) {
            if (this.extraFields.has(BANNER_IMAGE)) {
                this.bannerImage = this.extraFields.optString(BANNER_IMAGE);
            }
            if (this.extraFields.has(BANNER_TITLE)) {
                this.bannerTitle = this.extraFields.optString(BANNER_TITLE);
            }
        }
    }

    public NewCampaign(Parcel source) {
        this.indexUrl = source.readString();
        this.message = source.readString();
        this.communicationUUID = source.readString();
        this.subtitle = source.readString();
        this.title = source.readString();
        this.sorting = source.readInt();
        this.isNew = source.readByte() != 0;
        this.logoUrl = source.readString();
//        this.extraFields = source.readString();
        this.deliveryMethod = source.readString();
        this.displayType = source.readString();
//        this.settings = source.readString();
        this.audience = source.readString();
//        this.workflowSettings = source.readString();
        this.description = source.readString();
        this.startDate = source.readString();
        this.endDate = source.readString();
        this.communicationName = source.readString();
        this.communicationCategory = source.readString();
        this.category = source.readString();
        this.campaignType = source.readString();
        this.bannerImage = source.readString();
        this.bannerTitle = source.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.indexUrl);
        dest.writeString(this.message);
        dest.writeString(this.subtitle);
        dest.writeString(this.title);
        dest.writeInt(this.sorting);
        dest.writeInt(this.isNew ? 1 : 0);
        dest.writeString(this.logoUrl);
//        dest.writeString(extraFields);
        dest.writeString(deliveryMethod);
        dest.writeString(displayType);
//        dest.writeString(this.settings);
        dest.writeString(this.audience);
//        dest.writeString(this.workflowSettings);
        dest.writeString(this.description);
        dest.writeString(this.startDate);
        dest.writeString(this.endDate);
        dest.writeString(this.communicationName);
        dest.writeString(this.communicationCategory);
        dest.writeString(this.category);
        dest.writeString(this.campaignType);
        dest.writeString(this.communicationUUID);
        dest.writeString(this.bannerImage);
        dest.writeString(this.bannerTitle);
    }

    /**
     * Converts the Campaign into a JSON Object
     *
     * @return The JSON Object created from this campaign
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(INDEX_URL, this.indexUrl);
            jObj.putOpt(LOGO_URL, this.logoUrl);
            jObj.putOpt(MESSAGE, this.message);
            jObj.putOpt(COMMUNICATION_UUID, this.communicationUUID);
            jObj.putOpt(SUBTITLE, this.subtitle);
            jObj.putOpt(TITLE, this.title);
            jObj.putOpt(SORTING, this.sorting);
            jObj.putOpt(IS_NEW, this.isNew);
            jObj.putOpt(EXTRA_FIELDS, this.extraFields);
            jObj.putOpt(DELIVERY_METHOD, this.deliveryMethod);
            jObj.putOpt(DISPLAY_TYPE, this.displayType);
            jObj.putOpt(SETTINGS, this.settings);
            jObj.putOpt(AUDIENCE, this.audience);
            jObj.putOpt(WORKFLOW_SETTINGS, this.workflowSettings);
            jObj.putOpt(DESCRIPTION, this.description);
            jObj.putOpt(START_DATE, this.startDate);
            jObj.putOpt(END_DATE, this.endDate);
            jObj.putOpt(COMMUNICATION_NAME, this.communicationName);
            jObj.putOpt(COMMUNICATION_CATEGORY, this.communicationCategory);
            jObj.putOpt(CATEGORY, this.category);
            jObj.putOpt(CAMPAIGN_TYPE, this.campaignType);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the Campaign, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Campaign, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Campaign JSON object to String",
                    e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================

    /**
     * @return The message of the campaign
     */
    public String getMessage() {
        return message;
    }

    /**
     * @return The subtitle of the campaign
     */
    public String getSubtitle() {
        return subtitle;
    }

    /**
     * @return The title of the campaign
     */
    public String getTitle() {
        return title;
    }

    /**
     * @return An int that is used for sorting. This should be used as the
     * default sorting value
     */
    public int getSorting() {
        return sorting;
    }

    /**
     * Method used to get if the campaign is a new one
     *
     * @return <code>true</code> if the campaign is new; <code>false</code>
     * otherwise
     */
    public boolean getIsNew() {
        return isNew;
    }

    public String getLogoUrl() {
        return logoUrl;
    }

    public String getIndexUrl() {
        return indexUrl;
    }

    public String getDeliveryMethod() {
        return deliveryMethod;
    }

    public String getDisplayType() {
        return displayType;
    }

    public String getCampaignType() {
        return campaignType;
    }

    public String getAudience() {
        return audience;
    }

    public String getCategory() {
        return category;
    }

    public String getDescription() {
        return description;
    }

    public String getCommunicationUUID() {
        return communicationUUID;
    }

    public String getStartDate() {
        return startDate;
    }

    public String getEndDate() {
        return endDate;
    }

    public String getCommunicationName() {
        return communicationName;
    }

    public String getCommunicationCategory() {
        return communicationCategory;
    }

    public boolean isNew() {
        return isNew;
    }

    public JSONObject getSettings() {
        return settings;
    }

    public JSONObject getExtraFields() {
        return extraFields;
    }

    public JSONObject getWorkflowSettings() {
        return workflowSettings;
    }

    public String getBannerImage() {
        return bannerImage;
    }

    public String getBannerTitle() {
        return bannerTitle;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<NewCampaign> CREATOR = new Creator<NewCampaign>() {
        public NewCampaign createFromParcel(Parcel source) {
            return new NewCampaign(source);
        }

        public NewCampaign[] newArray(int size) {
            return new NewCampaign[size];
        }
    };
}
