/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 24-June-22.
 */

public class PacingDetails {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */
    private static final String STEPS = "steps";
    private static final String METERS = "meters";
    private static final String RESULT = "result";
    private static final String ENABLED = "enabled";
    private static final String GOAL_REACHED = "goal_reached";
    private static final String MESSAGE = "msg";
    private static final String TITLE = "title";
    private static final String SHORTCUT_ENABLED = "shortcut_enabled";
    private static final String BUTTON = "button";

    /* Member variables of the Campaign object */
    private PacingInner steps = new PacingInner();
    private PacingInner meters = new PacingInner();
    private boolean enabled = false;
    private boolean goal_reached = false;
    private String msg = "";
    private String title = "";
    private boolean shortcut_enabled = false;
    private String buttonText = "";

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public PacingDetails(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Campaign
     */
    public PacingDetails(JSONObject json) {
        if (json != null) {
            JSONObject result = new JSONObject();
            result = json.optJSONObject(RESULT);
            if (result != null) {
                if (result.optJSONObject(STEPS) != null) {
                    this.steps = new PacingInner(result.optJSONObject(STEPS));
                }
                if (result.optJSONObject(METERS) != null) {
                    this.meters = new PacingInner(result.optJSONObject(METERS));
                }
                this.enabled = result.optBoolean(ENABLED);
                this.goal_reached = result.optBoolean(GOAL_REACHED);
                this.msg = result.optString(MESSAGE);
                this.shortcut_enabled = result.optBoolean(SHORTCUT_ENABLED);
                this.title = result.optString(TITLE);
                this.buttonText = result.optString(BUTTON);
            }
        }
    }

    /**
     * Converts the Campaign into a JSON Object
     *
     * @return The JSON Object created from this campaign
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(STEPS, this.steps);
            jObj.putOpt(METERS, this.meters);
            jObj.putOpt(ENABLED, this.enabled);
            jObj.putOpt(GOAL_REACHED, this.goal_reached);
            jObj.putOpt(MESSAGE, this.msg);
            jObj.putOpt(SHORTCUT_ENABLED, this.shortcut_enabled);
            jObj.putOpt(TITLE, this.title);
            jObj.putOpt(BUTTON, this.buttonText);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the Campaign, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Campaign, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Campaign JSON object to String",
                    e);
        }
        return humanReadableString;
    }

    public class PacingInner {
        private static final String TOTAL = "total";
        private static final String MONTH = "month";
        private static final String WEEK = "week";
        private static final String DAY = "day";

        private PacingDetailsInner total = new PacingDetailsInner();
        private PacingDetailsInner month = new PacingDetailsInner();
        private PacingDetailsInner week = new PacingDetailsInner();
        private PacingDetailsInner day = new PacingDetailsInner();

        public PacingInner() {
            this.total = new PacingDetailsInner();
            this.month = new PacingDetailsInner();
            this.week = new PacingDetailsInner();
            this.day = new PacingDetailsInner();
        }

        public PacingInner(JSONObject json) {
            if (json != null) {
                if (json.optJSONObject(TOTAL) != null) {
                    this.total = new PacingDetailsInner(json.optJSONObject(TOTAL));
                }
                if (json.optJSONObject(MONTH) != null) {
                    this.month = new PacingDetailsInner(json.optJSONObject(MONTH));
                }
                if (json.optJSONObject(WEEK) != null) {
                    this.week = new PacingDetailsInner(json.optJSONObject(WEEK));
                }
                if (json.optJSONObject(DAY) != null) {
                    this.day = new PacingDetailsInner(json.optJSONObject(DAY));
                }
            }
        }

        public PacingDetailsInner getTotal() {
            return total;
        }

        public void setTotal(PacingDetailsInner total) {
            this.total = total;
        }

        public PacingDetailsInner getMonth() {
            return month;
        }

        public void setMonth(PacingDetailsInner month) {
            this.month = month;
        }

        public PacingDetailsInner getWeek() {
            return week;
        }

        public void setWeek(PacingDetailsInner week) {
            this.week = week;
        }

        public PacingDetailsInner getDay() {
            return day;
        }

        public void setDay(PacingDetailsInner day) {
            this.day = day;
        }
    }

    public class PacingDetailsInner {
        private static final String GOAL = "goal";
        private static final String VALLUE = "value";
        private static final String PER_DAY = "per_day";

        private double goal = 0.0d;
        private double value = 0.0d;
        private ArrayList<PacingDetailsDay> per_day = new ArrayList<>();

        public PacingDetailsInner() {
            this.goal = 0.0f;
            this.value = 0.0f;
            this.per_day = new ArrayList<>();
        }

        public PacingDetailsInner(JSONObject json) {
            if (json != null) {
                this.goal = json.optDouble(GOAL, 0.0d);
                this.value = json.optDouble(VALLUE, 0.0d);
                if (json.optJSONArray(PER_DAY) != null) {
                    JSONArray perDay = new JSONArray();
                    perDay = json.optJSONArray(PER_DAY);
                    if (perDay != null) {
                        ArrayList<PacingDetailsDay> list = new ArrayList<>();
                        for (int i = 0; i < perDay.length(); i++) {
                            JSONObject object = perDay.optJSONObject(i);
                            if (object != null) {
                                list.add(new PacingDetailsDay(object));
                            }
                        }
                        this.per_day = list;
                    }
                }
            }
        }

        public double getGoal() {
            return goal;
        }

        public void setGoal(double goal) {
            this.goal = goal;
        }

        public double getValue() {
            return value;
        }

        public void setValue(double value) {
            this.value = value;
        }

        public ArrayList<PacingDetailsDay> getPer_day() {
            return per_day;
        }

        public void setPer_day(ArrayList<PacingDetailsDay> per_day) {
            this.per_day = per_day;
        }
    }

    public class PacingDetailsDay {
        private static final String DAY = "day";
        private static final String VALLUE = "value";

        private String day = "";
        private double value = 0.0d;

        public PacingDetailsDay() {
            this.day = "";
            this.value = 0.0f;
        }

        public PacingDetailsDay(JSONObject json) {
            if (json != null) {
                this.day = json.optString(DAY);
                this.value = json.optDouble(VALLUE, 0.0d);
            }
        }

        public String getDay() {
            return day;
        }

        public void setDay(String day) {
            this.day = day;
        }

        public double getValue() {
            return value;
        }

        public void setValue(double value) {
            this.value = value;
        }
    }

    // ================================================================================
    // Getters
    // ================================================================================


    public boolean isEnabled() {
        return enabled;
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    public PacingInner getSteps() {
        return steps;
    }

    public void setSteps(PacingInner steps) {
        this.steps = steps;
    }

    public PacingInner getMeters() {
        return meters;
    }

    public void setMeters(PacingInner meters) {
        this.meters = meters;
    }

    public boolean isGoal_reached() {
        return goal_reached;
    }

    public String getMsg() {
        return msg;
    }

    public void setGoal_reached(boolean goal_reached) {
        this.goal_reached = goal_reached;
    }

    public void setMsg(String msg) {
        this.msg = msg;
    }

    public boolean isShortcut_enabled() {
        return shortcut_enabled;
    }

    public void setShortcut_enabled(boolean shortcut_enabled) {
        this.shortcut_enabled = shortcut_enabled;
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public String getButtonText() {
        return buttonText;
    }

    public void setButtonText(String buttonText) {
        this.buttonText = buttonText;
    }
}
