/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 17-Jan-22.
 */

public class PointTransaction implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String CAUSE = "cause";
    private static final String CREATED = "created";
    private static final String EXPIRE_DATE = "expire_date";
    private static final String EXPIRED = "expired";
    private static final String MERCHANT_UUID = "merchant_uuid";
    private static final String TYPE = "type";
    private static final String UUID = "uuid";
    private static final String VALUE = "value";

    /* Member variables of the Campaign object */

    private String cause = "";
    private String created = "";
    private String expireDate = "";
    private boolean expired = false;
    private String merchantUuid = "";
    private String type = "";
    private String uuid = "";
    private double value = 0.0d;

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public PointTransaction(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the PointTransaction
     */
    public PointTransaction(JSONObject json) {
        if (json != null) {
            this.cause = json.optString(CAUSE);
            this.created = json.optString(CREATED);
            this.expireDate = json.optString(EXPIRE_DATE);
            this.expired = json.optBoolean(EXPIRED);
            this.merchantUuid = json.optString(MERCHANT_UUID);
            this.type = json.optString(TYPE);
            this.uuid = json.optString(UUID);
            this.value = json.optDouble(VALUE);
        }
    }

    public PointTransaction(Parcel source) {
        this.cause = source.readString();
        this.created = source.readString();
        this.expireDate = source.readString();
        this.expired = source.readByte() != 0;
        this.merchantUuid = source.readString();
        this.type = source.readString();
        this.uuid = source.readString();
        this.value = source.readDouble();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.cause);
        dest.writeString(this.created);
        dest.writeString(this.expireDate);
        dest.writeByte((byte) (this.expired ? 1 : 0));
        dest.writeString(this.merchantUuid);
        dest.writeString(this.type);
        dest.writeString(this.uuid);
        dest.writeDouble(this.value);
    }

    /**
     * Converts the PointTransaction into a JSON Object
     *
     * @return The JSON Object created from this PointTransaction
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(CAUSE, this.cause);
            jObj.putOpt(CREATED, this.created);
            jObj.putOpt(EXPIRE_DATE, this.expireDate);
            jObj.putOpt(EXPIRED, this.expired);
            jObj.putOpt(MERCHANT_UUID, this.merchantUuid);
            jObj.putOpt(TYPE, this.type);
            jObj.putOpt(UUID, this.uuid);
            jObj.putOpt(VALUE, this.value);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the PointTransaction, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the PointTransaction, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting PointTransaction JSON object to String", e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================


    public String getCause() {
        return cause;
    }

    public String getCreated() {
        return created;
    }

    public String getExpireDate() {
        return expireDate;
    }

    public boolean isExpired() {
        return expired;
    }

    public String getMerchantUuid() {
        return merchantUuid;
    }

    public String getType() {
        return type;
    }

    public String getUuid() {
        return uuid;
    }

    public double getValue() {
        return value;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<PointTransaction> CREATOR = new Creator<PointTransaction>() {
        public PointTransaction createFromParcel(Parcel source) {
            return new PointTransaction(source);
        }

        public PointTransaction[] newArray(int size) {
            return new PointTransaction[size];
        }
    };
}
