/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 07-Dec-21.
 */

public class Product implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String ADMIN_NAME = "admin_name";
    private static final String AVAILABILITY = "availability";
    private static final String BARCODE = "barcode";
    private static final String CATEGORY = "category";
    private static final String CATEGORY_NAME = "category_name";
    private static final String CREATED = "created";
    private static final String CURRENCY = "currency";
    private static final String DESCRIPTION = "description";
    private static final String END_DATE = "end_date";
    private static final String EXTRA_FIELDS = "extra_fields";
    private static final String ID = "id";
    private static final String INNER_PHOTO = "inner_photo";
    private static final String IS_ACTIVE = "is_active";
    private static final String MERCHANT = "merchant";
    private static final String NAME = "name";
    private static final String PHOTO = "photo";
    private static final String PRICE = "price";
    private static final String PRODUCT_METADATA = "product_metadata";
    private static final String SKU = "sku";
    private static final String SORTING = "sorting";
    private static final String START_DATE = "start_date";
    private static final String UPDATED = "updated";
    private static final String UUID = "uuid";
    private static final String VAT = "vat";

    /* Member variables of the Campaign object */

    private String admin_name = "";
    private int availability = 0;
    private int barcode = 0;
    private int category = 0;
    private String category_name = "";
    private String created = "";
    private String currency = "";
    private String description = "";
    private String end_date = "";
    private JSONObject extra_fields = new JSONObject();
    private int id = 0;
    private String inner_photo = "";
    private boolean is_active;
    private int merchant = 0;
    private String name = "";
    private String photo = "";
    private int price = 0;
    private String product_metadata = "";
    private String sku = "";
    private int sorting = 0;
    private String start_date = "";
    private String updated = "";
    private String uuid = "";
    private int vat = 0;

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public Product(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Product
     */
    public Product(JSONObject json) {
        if (json != null) {
            this.admin_name = json.optString(ADMIN_NAME);
            this.availability = json.optInt(AVAILABILITY);
            this.barcode = json.optInt(BARCODE);
            this.category = json.optInt(CATEGORY);
            this.category_name = json.optString(CATEGORY_NAME);
            this.created = json.optString(CREATED);
            this.currency = json.optString(CURRENCY);
            this.description = json.optString(DESCRIPTION);
            this.end_date = json.optString(END_DATE);
            this.extra_fields = json.optJSONObject(EXTRA_FIELDS);
            this.id = json.optInt(ID);
            this.inner_photo = json.optString(INNER_PHOTO);
            this.is_active = json.optBoolean(IS_ACTIVE);
            this.merchant = json.optInt(MERCHANT);
            this.name = json.optString(NAME);
            this.photo = json.optString(PHOTO);
            this.price = json.optInt(PRICE);
            this.product_metadata = json.optString(PRODUCT_METADATA);
            this.sku = json.optString(SKU);
            this.sorting = json.optInt(SORTING);
            this.start_date = json.optString(START_DATE);
            this.updated = json.optString(UPDATED);
            this.uuid = json.optString(UUID);
            this.vat = json.optInt(VAT);
        }
    }

    public Product(Parcel source) {
        this.admin_name = source.readString();
        this.availability = source.readInt();
        this.barcode = source.readInt();
        this.category = source.readInt();
        this.category_name = source.readString();
        this.created = source.readString();
        this.currency = source.readString();
        this.description = source.readString();
        this.end_date = source.readString();
        this.id = source.readInt();
        this.inner_photo = source.readString();
        this.is_active = source.readByte() != 0;
        this.merchant = source.readInt();
        this.name = source.readString();
        this.photo = source.readString();
        this.price = source.readInt();
        this.product_metadata = source.readString();
        this.sku = source.readString();
        this.sorting = source.readInt();
        this.start_date = source.readString();
        this.updated = source.readString();
        this.uuid = source.readString();
        this.vat = source.readInt();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.admin_name);
        dest.writeInt(this.availability);
        dest.writeInt(this.barcode);
        dest.writeInt(this.category);
        dest.writeString(this.category_name);
        dest.writeString(this.created);
        dest.writeString(this.currency);
        dest.writeString(this.description);
        dest.writeString(this.end_date);
        dest.writeInt(this.id);
        dest.writeString(this.inner_photo);
        dest.writeByte((byte) (this.is_active ? 1 : 0));
        dest.writeInt(this.merchant);
        dest.writeString(this.name);
        dest.writeString(this.photo);
        dest.writeInt(this.price);
        dest.writeString(this.product_metadata);
        dest.writeString(this.sku);
        dest.writeInt(this.sorting);
        dest.writeString(this.start_date);
        dest.writeString(this.updated);
        dest.writeString(this.uuid);
        dest.writeInt(this.vat);
    }

    /**
     * Converts the Product into a JSON Object
     *
     * @return The JSON Object created from this Product
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(ADMIN_NAME, this.admin_name);
            jObj.putOpt(AVAILABILITY, this.availability);
            jObj.putOpt(BARCODE, this.barcode);
            jObj.putOpt(CATEGORY, this.category);
            jObj.putOpt(CATEGORY_NAME, this.category_name);
            jObj.putOpt(CREATED, this.created);
            jObj.putOpt(CURRENCY, this.currency);
            jObj.putOpt(DESCRIPTION, this.description);
            jObj.putOpt(END_DATE, this.end_date);
            jObj.putOpt(EXTRA_FIELDS, this.extra_fields);
            jObj.putOpt(ID, this.id);
            jObj.putOpt(INNER_PHOTO, this.inner_photo);
            jObj.putOpt(IS_ACTIVE, this.is_active);
            jObj.putOpt(MERCHANT, this.merchant);
            jObj.putOpt(NAME, this.name);
            jObj.putOpt(PHOTO, this.photo);
            jObj.putOpt(PRICE, this.price);
            jObj.putOpt(PRODUCT_METADATA, this.product_metadata);
            jObj.putOpt(SKU, this.sku);
            jObj.putOpt(SORTING, this.sorting);
            jObj.putOpt(START_DATE, this.start_date);
            jObj.putOpt(UPDATED, this.updated);
            jObj.putOpt(UUID, this.uuid);
            jObj.putOpt(VAT, this.vat);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the Product, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Product, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Product JSON object to String", e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================

    public String getAdminName() {
        return admin_name;
    }

    public int getAvailability() {
        return availability;
    }

    public int getBarcode() {
        return barcode;
    }

    public int getCategory() {
        return category;
    }

    public String getCategoryName() {
        return category_name;
    }

    public String getCreated() {
        return created;
    }

    public String getCurrency() {
        return currency;
    }

    public String getDescription() {
        return description;
    }

    public String getEndDate() {
        return end_date;
    }

    public JSONObject getExtraFields() {
        return extra_fields;
    }

    public int getId() {
        return id;
    }

    public String getInnerPhoto() {
        return inner_photo;
    }

    public boolean isIsActive() {
        return is_active;
    }

    public int getMerchant() {
        return merchant;
    }

    public String getName() {
        return name;
    }

    public String getPhoto() {
        return photo;
    }

    public int getPrice() {
        return price;
    }

    public String getProductMetadata() {
        return product_metadata;
    }

    public String getSku() {
        return sku;
    }

    public int getSorting() {
        return sorting;
    }

    public String getStartDate() {
        return start_date;
    }

    public String getUpdated() {
        return updated;
    }

    public String getUuid() {
        return uuid;
    }

    public int getVat() {
        return vat;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Product> CREATOR = new Creator<Product>() {
        public Product createFromParcel(Parcel source) {
            return new Product(source);
        }

        public Product[] newArray(int size) {
            return new Product[size];
        }
    };
}
