/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 17-Jan-22.
 */

public class Transaction implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String BASKET = "basket";
    private static final String CONSUMER_METADATA = "consumer_metadata";
    private static final String CUSTOM_TRANSACTION_ID = "custom_transaction_id";
    private static final String CREATED = "created";
    private static final String DELIVERY_IFO = "delivery_info";
    private static final String MERCHANT = "merchant";
    private static final String ORDER_ID = "order_id";
    private static final String ORDER_NO = "order_no";
    private static final String PRODUCTS = "products";
    private static final String STATUS = "status";
    private static final String TRANSACTION_CAUSE = "transaction_cause";
    private static final String TRANSACTION_ID = "transaction_id";
    private static final String TRANSACTION_METADATA = "transaction_metadata";
    private static final String TRANSACTION_TYPE = "transaction_type";
    private static final String VALUE = "value";

    /* Member variables of the Campaign object */

    private JSONArray basket = new JSONArray();
    private JSONObject consumerMetadata = new JSONObject();
    private String customTransactionId = "";
    private String created = "";
    private JSONObject deliveryInfo = new JSONObject();
    private String merchant = "";
    private String orderId = "";
    private String orderNo = "";
    private JSONArray products = new JSONArray();
    private String  status = "";
    private String transactionCause = "";
    private String transactionId = "";
    private JSONObject transactionMetadata = new JSONObject();
    private String transactionType = "";
    private double value = 0.0d;

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public Transaction(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Transaction
     */
    public Transaction(JSONObject json) {
        if (json != null) {
            this.basket = json.optJSONArray(BASKET);
            this.consumerMetadata = json.optJSONObject(CONSUMER_METADATA);
            this.customTransactionId = json.optString(CUSTOM_TRANSACTION_ID);
            this.created = json.optString(CREATED);
            this.deliveryInfo = json.optJSONObject(DELIVERY_IFO);
            this.merchant = json.optString(MERCHANT);
            this.orderId = json.optString(ORDER_ID);
            this.orderNo = json.optString(ORDER_NO);
            this.products = json.optJSONArray(PRODUCTS);
            this.status = json.optString(STATUS);
            this.transactionCause = json.optString(TRANSACTION_CAUSE);
            this.transactionId = json.optString(TRANSACTION_ID);
            this.transactionMetadata = json.optJSONObject(TRANSACTION_METADATA);
            this.transactionType = json.optString(TRANSACTION_TYPE);
            this.value = json.optDouble(VALUE);
        }
    }

    public Transaction(Parcel source) {
        this.customTransactionId = source.readString();
        this.created = source.readString();
        this.merchant = source.readString();
        this.orderId = source.readString();
        this.orderNo = source.readString();
        this.status = source.readString();
        this.transactionCause = source.readString();
        this.transactionId = source.readString();
        this.transactionType = source.readString();
        this.value = source.readDouble();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.customTransactionId);
        dest.writeString(this.created);
        dest.writeString(this.merchant);
        dest.writeString(this.orderId);
        dest.writeString(this.orderNo);
        dest.writeString(this.transactionCause);
        dest.writeString(this.transactionId);
        dest.writeString(this.status);
        dest.writeString(this.transactionType);
        dest.writeDouble(this.value);
    }

    /**
     * Converts the Transaction into a JSON Object
     *
     * @return The JSON Object created from this Transaction
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(BASKET, this.basket);
            jObj.putOpt(CONSUMER_METADATA, this.consumerMetadata);
            jObj.putOpt(CUSTOM_TRANSACTION_ID, this.customTransactionId);
            jObj.putOpt(CREATED, this.created);
            jObj.putOpt(DELIVERY_IFO, this.deliveryInfo);
            jObj.putOpt(MERCHANT, this.merchant);
            jObj.putOpt(ORDER_ID, this.orderId);
            jObj.putOpt(ORDER_NO, this.orderNo);
            jObj.putOpt(PRODUCTS, this.products);
            jObj.putOpt(STATUS, this.status);
            jObj.putOpt(TRANSACTION_CAUSE, this.transactionCause);
            jObj.putOpt(TRANSACTION_ID, this.transactionId);
            jObj.putOpt(TRANSACTION_METADATA, this.transactionMetadata);
            jObj.putOpt(TRANSACTION_TYPE, this.transactionType);
            jObj.putOpt(VALUE, this.value);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the Transaction, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Transaction, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Transaction JSON object to String", e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================


    public JSONArray getBasket() {
        return basket;
    }

    public JSONObject getConsumerMetadata() {
        return consumerMetadata;
    }

    public String getCustomTransactionId() {
        return customTransactionId;
    }

    public String getCreated() {
        return created;
    }

    public JSONObject getDeliveryInfo() {
        return deliveryInfo;
    }

    public String getMerchant() {
        return merchant;
    }

    public String getOrderId() {
        return orderId;
    }

    public String getOrderNo() {
        return orderNo;
    }

    public JSONArray getProducts() {
        return products;
    }

    public String getStatus() {
        return status;
    }

    public String getTransactionCause() {
        return transactionCause;
    }

    public String getTransactionId() {
        return transactionId;
    }

    public JSONObject getTransactionMetadata() {
        return transactionMetadata;
    }

    public String getTransactionType() {
        return transactionType;
    }

    public double getValue() {
        return value;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<Transaction> CREATOR = new Creator<Transaction>() {
        public Transaction createFromParcel(Parcel source) {
            return new Transaction(source);
        }

        public Transaction[] newArray(int size) {
            return new Transaction[size];
        }
    };
}
