/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.models;

import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;

import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 04-Apr-23.
 */

public class UnifiedCoupon implements Parcelable, Serializable {

    private static final long serialVersionUID = -4754964462459705285L;

    /* Constants used to export the campaign in JSON formal and vice versa */

    private static final String BARCODE = "barcode";
    private static final String CODE = "code";
    private static final String CREATED = "created";
    private static final String COUPONS = "coupons";
    private static final String STATUS = "status";
    private static final String DESCRIPTION = "description";

    /* Member variables of the Campaign object */

    private String barcode = "";
    private String description = "";
    private String status = "";
    private int code = 0;
    private String created = "";
    private ArrayList<Coupon> coupons = new ArrayList();
    private Date expirationDate = new Date();

    public UnifiedCoupon() {
        this.barcode = "";
        this.description = "";
        this.status = "";
        this.code = 0;
        this.created = "";
        this.coupons = new ArrayList();
        this.expirationDate = new Date();
    }

    /**
     * Basic constructor used to create an object from a String, representing a
     * JSON Object
     *
     * @param json The String, representing the JSON Object
     * @throws JSONException Thrown if the String cannot be converted to JSON
     */
    public UnifiedCoupon(String json) throws JSONException {
        this(new JSONObject(json));
    }

    /**
     * Constructor used to create an Object from a given JSON Object
     *
     * @param json JSON Object used to create the Coupon
     */
    public UnifiedCoupon(JSONObject json) {
        if (json != null) {
            this.barcode = json.optString(BARCODE);
            this.description = json.optString(DESCRIPTION);
            this.status = json.optString(STATUS);
            this.code = json.optInt(CODE);
            this.created = json.optString(CREATED);
            JSONArray jArray = null;
            jArray = json.optJSONArray(COUPONS);
            if (jArray != null && jArray.length() > 0) {
                for (int i = 0; i < jArray.length(); i++) {
                    JSONObject jObj = jArray.optJSONObject(i);
                    this.coupons.add(new Coupon(jObj));
                }
            }
        }
    }

    public UnifiedCoupon(Parcel source) {
        this.barcode = source.readString();
        this.description = source.readString();
        this.status = source.readString();
        this.code = source.readInt();
        this.created = source.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.barcode);
        dest.writeString(this.description);
        dest.writeString(this.status);
        dest.writeInt(this.code);
        dest.writeString(this.created);
    }

    /**
     * Converts the Unified Coupon into a JSON Object
     *
     * @return The JSON Object created from this Unified Coupon
     */
    public JSONObject toJSONObject() {
        JSONObject jObj = new JSONObject();
        try {
            jObj.putOpt(BARCODE, this.barcode);
            jObj.putOpt(DESCRIPTION, this.description);
            jObj.putOpt(STATUS, this.status);
            jObj.putOpt(CODE, this.code);
            jObj.putOpt(CREATED, this.created);
            jObj.putOpt(COUPONS, this.coupons);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG) {
                e.printStackTrace();
            }
        }
        return jObj;
    }

    /**
     * String representation of the Unified Coupon, as a JSON object
     *
     * @return A String representation of JSON object
     */
    public String toString() {
        if (toJSONObject() != null)
            return toJSONObject().toString();
        return null;
    }

    /**
     * String representation of the Coupon, as a human readable JSON object
     *
     * @return A human readable String representation of JSON object
     */
    public String toHumanReadableString() {
        String humanReadableString = null;
        try {
            humanReadableString = toJSONObject().toString(2);
        } catch (JSONException e) {
            WarpUtils.warn("Failed converting Unified Coupon JSON object to String", e);
        }
        return humanReadableString;
    }

    // ================================================================================
    // Getters
    // ================================================================================


    public String getBarcode() {
        return barcode;
    }

    public void setBarcode(String barcode) {
        this.barcode = barcode;
    }

    public int getCode() {
        return code;
    }

    public void setCode(int code) {
        this.code = code;
    }

    public String getCreated() {
        return created;
    }

    public void setCreated(String created) {
        this.created = created;
    }

    public ArrayList<Coupon> getCoupons() {
        return coupons;
    }

    public void setCoupons(ArrayList<Coupon> coupons) {
        this.coupons = coupons;
    }

    public String getStatus() {
        return status;
    }

    public void setStatus(String status) {
        this.status = status;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public Date getExpirationDate() {
        return expirationDate;
    }

    public void setExpirationDate(Date expirationDate) {
        this.expirationDate = expirationDate;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<UnifiedCoupon> CREATOR = new Creator<UnifiedCoupon>() {
        public UnifiedCoupon createFromParcel(Parcel source) {
            return new UnifiedCoupon(source);
        }

        public UnifiedCoupon[] newArray(int size) {
            return new UnifiedCoupon[size];
        }
    };
}
