package ly.warp.sdk.io.request;

import android.util.Base64;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 02-Aug-22.
 */

public class PacingCalculateRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_MAPP = "consumer_data";
    private final String KEY_ACTION = "action";
    private final String KEY_ACTION_VALUE = "integration";
    private final String KEY_METHOD = "method";
    private final String KEY_METHOD_VALUE = "steps";
    private final String KEY_OPERATION = "operation";
    private final String KEY_OPERATION_VALUE = "add";
    private final String KEY_DATA = "data";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;
    private String mDate = "";
    private int mCounter = 0;

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyConsumerRequest, initializes an empty filters HashMap
     */
    public PacingCalculateRequest() {
        mFilters = new HashMap<>();
    }

    public PacingCalculateRequest(PacingCalculateRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof PacingCalculateRequest) {
            PacingCalculateRequest other = (PacingCalculateRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public PacingCalculateRequest setDate(String date) {
        this.mDate = date;
        return this;
    }

    public PacingCalculateRequest setCounter(int counter) {
        this.mCounter = counter;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return PacingCalculateRequest
     */
    public PacingCalculateRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return PacingCalculateRequest
     */
    public PacingCalculateRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            JSONObject extraJson = new JSONObject();
            extraJson.putOpt(KEY_ACTION, KEY_ACTION_VALUE);
            extraJson.putOpt(KEY_METHOD, KEY_METHOD_VALUE);
            extraJson.putOpt(KEY_OPERATION, KEY_OPERATION_VALUE);

            JSONObject data = new JSONObject();
            data.putOpt("event", "steps");
            data.putOpt("type", "counter");
            data.putOpt("counter", mCounter);
            data.putOpt("date", mDate);

            extraJson.putOpt(KEY_DATA, data);
            bodyJsonObject.putOpt(KEY_MAPP, extraJson);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_pacing_calculate_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
