package ly.warp.sdk.io.request;

import android.util.Base64;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 11-Jan-22.
 */

public class WarplyAddCardRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_ACTION = "action";
    private final String KEY_ACTION_VALUE = "add_card";
    private final String KEY_CARD_ISSUER = "card_issuer";
    private final String KEY_CARD_NUMBER = "card_number";
    private final String KEY_CARD_HOLDER = "cardholder";
    private final String KEY_CARD_EXP_MONTH = "expiration_month";
    private final String KEY_CARD_EXP_YEAR = "expiration_year";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;
    private String mCardIssuer = "";
    private String mCardNumber = "";
    private String mCardHolder = "";
    private String mCardExpMonth = "";
    private String mCardExpYear = "";

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyAddCardRequest, initializes an empty filters HashMap
     */
    public WarplyAddCardRequest() {
        mFilters = new HashMap<>();
    }

    public WarplyAddCardRequest(WarplyAddCardRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyAddCardRequest) {
            WarplyAddCardRequest other = (WarplyAddCardRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public WarplyAddCardRequest setCardIssuer(String cardIssuer) {
        mCardIssuer = cardIssuer;
        return this;
    }

    public WarplyAddCardRequest setCardNumber(String cardNumber) {
        mCardNumber = cardNumber;
        return this;
    }

    public WarplyAddCardRequest setCardHolder(String cardHolder) {
        mCardHolder = cardHolder;
        return this;
    }

    public WarplyAddCardRequest setCardExpMonth(String cardExpMonth) {
        mCardExpMonth = cardExpMonth;
        return this;
    }

    public WarplyAddCardRequest setCardExpYear(String cardExpYear) {
        mCardExpYear = cardExpYear;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyAddCardRequest
     */
    public WarplyAddCardRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyAddCardRequest
     */
    public WarplyAddCardRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            bodyJsonObject.putOpt(KEY_ACTION, KEY_ACTION_VALUE);
            bodyJsonObject.putOpt(KEY_CARD_ISSUER, mCardIssuer);
            bodyJsonObject.putOpt(KEY_CARD_NUMBER, mCardNumber);
            bodyJsonObject.putOpt(KEY_CARD_HOLDER, mCardHolder);
            bodyJsonObject.putOpt(KEY_CARD_EXP_MONTH, mCardExpMonth);
            bodyJsonObject.putOpt(KEY_CARD_EXP_YEAR, mCardExpYear);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_add_card_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
