package ly.warp.sdk.io.request;

import android.util.Base64;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 15-Dec-21.
 */

public class WarplyConsumerRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_MAPP = "consumer_data";
    private final String KEY_ACTION = "action";
    private final String KEY_ACTION_VALUE = "handle_user_details";
    private final String KEY_PROCESS = "process";
    private final String KEY_PROCESS_VALUE = "get";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyConsumerRequest, initializes an empty filters HashMap
     */
    public WarplyConsumerRequest() {
        mFilters = new HashMap<>();
    }

    public WarplyConsumerRequest(WarplyConsumerRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyConsumerRequest) {
            WarplyConsumerRequest other = (WarplyConsumerRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyConsumerRequest
     */
    public WarplyConsumerRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyConsumerRequest
     */
    public WarplyConsumerRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            JSONObject extraJson = new JSONObject();
            extraJson.putOpt(KEY_ACTION, KEY_ACTION_VALUE);
            extraJson.putOpt(KEY_PROCESS, KEY_PROCESS_VALUE);
            bodyJsonObject.putOpt(KEY_MAPP, extraJson);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_consumer_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
