package ly.warp.sdk.io.request;

import android.text.TextUtils;
import android.util.Base64;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 07-Dec-21.
 */

public class WarplyContactRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_ACTION = "action";
    private final String KEY_ACTION_VALUE = "addContact";
    private final String KEY_SUBJECT = "contact_type";
    private final String KEY_NAME = "name";
    private final String KEY_EMAIL = "email";
    private final String KEY_MSISDN = "msisdn";
    private final String KEY_MESSAGE = "message";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;
    private String mSubject = "";
    private String mName = "";
    private String mEmail = "";
    private String mMsisdn = "";
    private String mMessage = "";

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyContactRequest, initializes an empty filters HashMap
     */
    public WarplyContactRequest() {
        mFilters = new HashMap<>();
    }

    public WarplyContactRequest(WarplyContactRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyContactRequest) {
            WarplyContactRequest other = (WarplyContactRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public WarplyContactRequest setContactSubject(String subject) {
        mSubject = subject;
        return this;
    }

    public WarplyContactRequest setContactName(String name) {
        mName = name;
        return this;
    }

    public WarplyContactRequest setContactEmail(String email) {
        mEmail = email;
        return this;
    }

    public WarplyContactRequest setContactNumber(String msisdn) {
        mMsisdn = msisdn;
        return this;
    }

    public WarplyContactRequest setContactMessage(String message) {
        mMessage = message;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyContactRequest
     */
    public WarplyContactRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyContactRequest
     */
    public WarplyContactRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            bodyJsonObject.putOpt(KEY_ACTION, KEY_ACTION_VALUE);
            bodyJsonObject.putOpt(KEY_SUBJECT, TextUtils.isEmpty(mSubject) ? "" : mSubject);
            bodyJsonObject.putOpt(KEY_NAME, TextUtils.isEmpty(mName) ? "" : mName);
            bodyJsonObject.putOpt(KEY_EMAIL, TextUtils.isEmpty(mEmail) ? "" : mEmail);
            bodyJsonObject.putOpt(KEY_MSISDN, TextUtils.isEmpty(mMsisdn) ? "" : mMsisdn);
            bodyJsonObject.putOpt(KEY_MESSAGE, TextUtils.isEmpty(mMessage) ? "" : mMessage);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_contact_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
