package ly.warp.sdk.io.request;

import android.util.Base64;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 12-May-22.
 */

public class WarplyGetCampaignsRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_ACTION = "action";
    private final String KEY_ACTION_VALUE = "retrieve";
    private final String KEY_LANGUAGE = "language";
    private final String KEY_FILTERS = "filters";

    // ===========================================================
    // Fields
    // ===========================================================

    private JSONObject mFilters = new JSONObject();
    private long mCacheUpdateInterval = 0;
    private String mLanguage = WarplyProperty.getLanguage(Warply.getWarplyContext());

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyGetCampaignsRequest, initializes an empty filters HashMap
     */
    public WarplyGetCampaignsRequest() {
        mFilters = new JSONObject();
    }

    public WarplyGetCampaignsRequest(WarplyGetCampaignsRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyGetCampaignsRequest) {
            WarplyGetCampaignsRequest other = (WarplyGetCampaignsRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public WarplyGetCampaignsRequest setLanguage(String language) {
        mLanguage = language;
        return this;
    }

    public WarplyGetCampaignsRequest setFilters(JSONObject filters) {
        mFilters = filters;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyGetCampaignsRequest
     */
    public WarplyGetCampaignsRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyGetCampaignsRequest
     */
    public WarplyGetCampaignsRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the campaigns Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            bodyJsonObject.putOpt(KEY_ACTION, KEY_ACTION_VALUE);
            bodyJsonObject.putOpt(KEY_LANGUAGE, mLanguage);
            if (mFilters != null && mFilters.length() > 0)
                bodyJsonObject.putOpt(KEY_FILTERS, mFilters);
            else
                bodyJsonObject.putOpt(KEY_FILTERS, new JSONObject());
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.length() > 0 ? String.valueOf(mFilters.hashCode()) : "default_get_campaigns_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
