package ly.warp.sdk.io.request;

import android.text.TextUtils;
import android.util.Base64;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 04-Feb-22.
 */

public class WarplyGetCouponsetsRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_ACTION = "action";
    private final String KEY_ACTION_VALUE = "retrieve_multilingual";
    private final String KEY_ACTIVE = "active";
    private final String KEY_VISIBLE = "visible";
    private final String KEY_LANGUAGE = "language";
    private final String KEY_UUIDS = "uuids";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;
    private boolean mActive = true;
    private boolean mVisible = true;
    private ArrayList<String> mUuids = new ArrayList<>();
    private String mLanguage = WarplyProperty.getLanguage(Warply.getWarplyContext());

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyGetCouponsetsRequest, initializes an empty filters HashMap
     */
    public WarplyGetCouponsetsRequest() {
        mFilters = new HashMap<>();
    }

    public WarplyGetCouponsetsRequest(WarplyGetCouponsetsRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyGetCouponsetsRequest) {
            WarplyGetCouponsetsRequest other = (WarplyGetCouponsetsRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public WarplyGetCouponsetsRequest setActive(boolean active) {
        mActive = active;
        return this;
    }

    public WarplyGetCouponsetsRequest setVisible(boolean visible) {
        mVisible = visible;
        return this;
    }

    public WarplyGetCouponsetsRequest setUuids(ArrayList<String> uuids) {
        mUuids = uuids;
        return this;
    }

    public WarplyGetCouponsetsRequest setLanguage(String language) {
        mLanguage = language;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyGetCouponsetsRequest
     */
    public WarplyGetCouponsetsRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyGetCouponsetsRequest
     */
    public WarplyGetCouponsetsRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the couponsets Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            bodyJsonObject.putOpt(KEY_ACTION, KEY_ACTION_VALUE);
            bodyJsonObject.putOpt(KEY_ACTIVE, mActive);
            bodyJsonObject.putOpt(KEY_VISIBLE, mVisible);
            bodyJsonObject.putOpt(KEY_LANGUAGE, TextUtils.isEmpty(mLanguage) ? "el" : mLanguage);
            if (mUuids != null && mUuids.size() > 0)
                bodyJsonObject.putOpt(KEY_UUIDS, new JSONArray(mUuids));
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_get_couponsets_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
