package ly.warp.sdk.io.request;

import android.util.Base64;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;

import ly.warp.sdk.utils.constants.WarpConstants;

public class WarplyInboxRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_ACTION = "action";
    private final String KEY_FILTER = "filter";
    private final String KEY_FILTER_VALUE = "value";
    private final String ACTION_GET_INBOX = "get_inbox";

    private static final String FILTER_MESSAGE = "message";
    private static final String FILTER_UUID = "session_uuid";
    private static final String FILTER_OFFER_MESSAGE = "offer_message";
    private static final String FILTER_OFFER_CATEGORY = "offer_category";
    private static final String FILTER_CAMPAIGN_TYPE = "campaign_type";
    private static final String FILTER_SORTING = "sorting";
    private static final String FILTER_DISPLAY_TYPE = "display_type";
    private final static String FILTER_EXTRA_FIELDS = "extra_fields";
    private final static String FILTER_EXPIRED = "expired";
    private final static String FILTER_DELIVERY_METHOD = "delivery_method";
    private final static String VALUE_INAPP = "inapp";


    public enum DisplayType {

        LIST("list"),
        FEED("feed"),
        TILE("tile"),
        FULL_PAGE("full_page"),
        HALF_PAGE("half_page");

        private String value;

        /**
         * @param value Set display type value
         */
        DisplayType(String value) {
            this.value = value;
        }

        /**
         * @return Get display type value
         */
        public String getValue() {
            return value;
        }
    }

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of Warply Inbox request, initializes an empty filters HashMap
     */
    public WarplyInboxRequest() {
        mFilters = new HashMap<>();
    }

    public WarplyInboxRequest(WarplyInboxRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {

        if (object instanceof WarplyInboxRequest) {
            WarplyInboxRequest other = (WarplyInboxRequest) object;
            return other == this ||
                    (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    /**
     * Call this to set the message filter, on Inbox Request
     *
     * @param message The message that will be used.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addMessageFilter(String message) {
        mFilters.put(FILTER_MESSAGE, message);
        return this;
    }

    /**
     * Call this to set the session_uuid filter, on Inbox Request
     *
     * @param uuid The uuid that will be used.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addUuidFilter(String uuid) {
        mFilters.put(FILTER_UUID, uuid);
        return this;
    }

    /**
     * Call this to set the offer message filter, on Inbox Request
     *
     * @param offerMessage The offer message that will be used.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addOfferMessageFilter(String offerMessage) {
        mFilters.put(FILTER_OFFER_MESSAGE, offerMessage);
        return this;
    }

    /**
     * Call this to set the offer category filter, on Inbox Request
     *
     * @param offerCategory The offer category that will be used.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addOfferCategoryFilter(String offerCategory) {
        mFilters.put(FILTER_OFFER_CATEGORY, offerCategory);
        return this;
    }

    /**
     * Call this to set the campaign type filter, on Inbox Request
     *
     * @param campaignType The offer category that will be used.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addCampaignTypeFilter(String campaignType) {
        mFilters.put(FILTER_CAMPAIGN_TYPE, campaignType);
        return this;
    }

    /**
     * Call this to set the sorting type filter, on Inbox Request
     *
     * @param sorting The sorting method that will be used.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addSortingFilter(String sorting) {
        mFilters.put(FILTER_SORTING, sorting);
        return this;
    }

    /**
     * Call this to add the displayType that campaign will be.
     *
     * @param displayType The display type that will be used.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addDisplayTypeFilter(DisplayType displayType) {
        mFilters.put(FILTER_DISPLAY_TYPE, displayType.getValue());
        return this;
    }

    /**
     * Call this to add an extra field filter, on Inbox Request.
     *
     * @param extra The extra field that will be used.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addExtraFieldFilter(String extra) {
        mFilters.put(FILTER_EXTRA_FIELDS, extra);
        return this;
    }

    /**
     * Call this to add an extra field filter, on Inbox Request.
     *
     * @param expired The extra field that will be used for expired inbox.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addExpireTypeFilter(String expired) {
        mFilters.put(FILTER_EXPIRED, expired);
        return this;
    }

    /**
     * Call this to add an extra field filter, on Inbox Request.
     *
     * @param key   The key for filtering.
     * @param extra The extra field that will be used.
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest addCustomFilter(String key, String extra) {
        mFilters.put(key, extra);
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyInboxRequest
     */
    public WarplyInboxRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return offersJsonObject
     */
    public JSONObject toJson() {

        JSONObject offersJsonObject = new JSONObject();
        try {
            offersJsonObject.putOpt(KEY_ACTION, ACTION_GET_INBOX);
            if (mFilters != null && mFilters.size() > 0) {
                JSONObject filterJson;
                JSONArray filterJsonArray = new JSONArray();
                for (Map.Entry<String, String> entry : mFilters.entrySet()) {
                    if (entry.getValue().equals("true") || entry.getValue().equals("false")) {
                        offersJsonObject.put(FILTER_EXPIRED, Boolean.valueOf(entry.getValue()));
                    } else {
                        filterJson = new JSONObject();
                        filterJson.put(KEY_FILTER, entry.getKey());
                        filterJson.put(KEY_FILTER_VALUE, entry.getValue());
                        filterJsonArray.put(filterJson);
                    }
                }
                offersJsonObject.put(KEY_FILTER, filterJsonArray);
            }
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return offersJsonObject;
    }

    public JSONObject toJsonInApp() {

        JSONObject offersJsonObject = new JSONObject();
        try {
            offersJsonObject.putOpt(KEY_ACTION, ACTION_GET_INBOX);
            offersJsonObject.putOpt(KEY_FILTER, FILTER_DELIVERY_METHOD);
            offersJsonObject.putOpt(KEY_FILTER_VALUE, VALUE_INAPP);
            if (mFilters != null && mFilters.size() > 0) {
                JSONObject filterJson;
                JSONArray filterJsonArray = new JSONArray();
                for (Map.Entry<String, String> entry : mFilters.entrySet()) {
                    if (entry.getValue().equals("true") || entry.getValue().equals("false")) {
                        offersJsonObject.put(FILTER_EXPIRED, Boolean.valueOf(entry.getValue()));
                    } else {
                        filterJson = new JSONObject();
                        filterJson.put(KEY_FILTER, entry.getKey());
                        filterJson.put(KEY_FILTER_VALUE, entry.getValue());
                        filterJsonArray.put(filterJson);
                    }
                }
                offersJsonObject.put(KEY_FILTER, filterJsonArray);
            }
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return offersJsonObject;
    }

    /**
     * Call this to set the signature for specific offers
     *
     * @return signature
     */
    public String getSignature() {

        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_inbox_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
