package ly.warp.sdk.io.request;

import android.util.Base64;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;

import ly.warp.sdk.utils.WarplyManagerHelper;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 15-June-22.
 */

public class WarplyIntegrationRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_MAPP = "consumer_data";
    private final String KEY_ACTION = "action";
    private final String KEY_ACTION_VALUE = "integration";
    private final String KEY_DATA = "data";
    private final String KEY_METHOD = "method";
    private final String KEY_METHOD_VALUE = "consumer_extra_data";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;
    private boolean mIsNonTelco = false;
    private boolean mHasAcceptedConsent = false;
    private String mGuid = "";
    private ArrayList<String> mPhoneList = new ArrayList<>();

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyIntegrationRequest, initializes an empty filters HashMap
     */
    public WarplyIntegrationRequest() {
        this.mFilters = new HashMap<>();
        this.mIsNonTelco = false;
        this.mHasAcceptedConsent = false;
        this.mGuid = "";
        this.mPhoneList = new ArrayList<>();
    }

    public WarplyIntegrationRequest(WarplyIntegrationRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyIntegrationRequest) {
            WarplyIntegrationRequest other = (WarplyIntegrationRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public WarplyIntegrationRequest setIsNoTelco(boolean isNonTelco) {
        this.mIsNonTelco = isNonTelco;
        return this;
    }

    public WarplyIntegrationRequest setHasAcceptedConsent(boolean hasAcceptedConsent) {
        this.mHasAcceptedConsent = hasAcceptedConsent;
        return this;
    }

    public WarplyIntegrationRequest setMsisdnList(ArrayList<String> phoneList) {
        this.mPhoneList = phoneList;
        return this;
    }

    public WarplyIntegrationRequest setGuid(String guid) {
        this.mGuid = guid;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyIntegrationRequest
     */
    public WarplyIntegrationRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyIntegrationRequest
     */
    public WarplyIntegrationRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            JSONObject extraJson = new JSONObject();
            extraJson.putOpt(KEY_ACTION, KEY_ACTION_VALUE);
            extraJson.putOpt(KEY_METHOD, KEY_METHOD_VALUE);
            JSONObject data = new JSONObject();
            data.putOpt("nonTelco", mIsNonTelco);
            data.putOpt("acceptedConsent", mHasAcceptedConsent);
            data.putOpt("msisdnList", new JSONArray(mPhoneList));
            data.putOpt("guid", mGuid);

            extraJson.putOpt(KEY_DATA, data);
            bodyJsonObject.putOpt(KEY_MAPP, extraJson);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_integration_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
