/*
 * Copyright 2010-2013 Warply Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, without modification, are
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE WARPLY LTD ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL WARPLY LTD OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ly.warp.sdk.io.request;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import ly.warp.sdk.utils.WarplyDeviceInfoCollector;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.Warply;
import ly.warp.sdk.io.volley.AuthFailureError;
import ly.warp.sdk.io.volley.NetworkResponse;
import ly.warp.sdk.io.volley.ParseError;
import ly.warp.sdk.io.volley.Response;
import ly.warp.sdk.io.volley.Response.ErrorListener;
import ly.warp.sdk.io.volley.Response.Listener;
import ly.warp.sdk.io.volley.toolbox.HttpHeaderParser;
import ly.warp.sdk.io.volley.toolbox.JsonRequest;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import android.content.Context;
import android.text.format.DateFormat;

public class WarplyJsonArrayRequest extends JsonRequest<JSONObject> {

    /**
     * the HTTP method to use
     *
     * @param url           URL to fetch the JSON from
     * @param data          A {@link JSONObject} to post with the request. Null is allowed
     *                      and indicates no parameters will be posted along with request.
     * @param listener      Listener to receive the JSON response
     * @param errorListener Error listener, or null to ignore errors.
     */
    public WarplyJsonArrayRequest(int method, String url, JSONArray data,
                                  Listener<JSONObject> listener, ErrorListener errorListener) {
        super(method, url, (data == null) ? null : data.toString(), listener,
                errorListener);
    }

    @Override
    public Map<String, String> getHeaders() throws AuthFailureError {
        Map<String, String> headers = new HashMap<String, String>();
        Context context = Warply.getWarplyContext();
        String timeStamp = DateFormat.format("yyyy-MM-dd hh:mm:ss", System.currentTimeMillis()).toString();
        headers.put(WarpConstants.HEADER_LOYALTY_BUNDLE_ID, "android:" + context.getPackageName());
        headers.put(WarpConstants.HEADER_DATE, timeStamp);
        headers.put(WarpConstants.HEADER_UNIQUE_DEVICE_ID, new WarplyDeviceInfoCollector(context).getUniqueDeviceId());
//        headers.put(WarpConstants.HEADER_ADVERTISING_ID, new WarplyDeviceInfoCollector(context).getAdvertisementId());
        headers.put(WarpConstants.HEADER_CHANNEL, "mobile");
        if (WarpUtils.isRegisteredWarply(context)) {
            String apiKey = WarpUtils.getApiKey(context);
            String webId = WarpUtils.getWebId(context);
            headers.put(WarpConstants.HEADER_WEB_ID, webId);
            headers.put(WarpConstants.HEADER_SIGNATURE, WarpUtils.produceSignature(apiKey + timeStamp));
        }
        return headers;
    }

    @Override
    protected Response<JSONObject> parseNetworkResponse(NetworkResponse response) {
        try {
            String jsonString = new String(response.data,
                    HttpHeaderParser.parseCharset(response.headers));
            return Response.success(new JSONObject(jsonString),
                    HttpHeaderParser.parseCacheHeaders(response));
        } catch (UnsupportedEncodingException e) {
            return Response.error(new ParseError(e));
        } catch (JSONException je) {
            return Response.error(new ParseError(je));
        }
    }
}
