package ly.warp.sdk.io.request;

import android.text.TextUtils;
import android.util.Base64;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 15-Dec-21.
 */

public class WarplyLoginRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_ID = "id";
    private final String KEY_PASSWORD = "password";
    private final String KEY_WEB_ID = "web_id";
    private final String KEY_APP_UUID = "app_uuid";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;
    private String mId = "";
    private String mPassword = "";

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyLoginRequest, initializes an empty filters HashMap
     */
    public WarplyLoginRequest() {
        mFilters = new HashMap<>();
    }

    public WarplyLoginRequest(WarplyLoginRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyLoginRequest) {
            WarplyLoginRequest other = (WarplyLoginRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public WarplyLoginRequest setId(String id) {
        mId = id;
        return this;
    }

    public WarplyLoginRequest setPassword(String password) {
        mPassword = password;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyLoginRequest
     */
    public WarplyLoginRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyLoginRequest
     */
    public WarplyLoginRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            bodyJsonObject.putOpt(KEY_ID, TextUtils.isEmpty(mId) ? "" : mId);
            bodyJsonObject.putOpt(KEY_PASSWORD, TextUtils.isEmpty(mPassword) ? "" : mPassword);
            bodyJsonObject.putOpt(KEY_WEB_ID, WarpUtils.getWebId(Warply.getWarplyContext()));
            bodyJsonObject.putOpt(KEY_APP_UUID, WarplyProperty.getAppUuid(Warply.getWarplyContext()));
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_login_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
