package ly.warp.sdk.io.request;

import android.text.TextUtils;
import android.util.Base64;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 07-Dec-21.
 */

public class WarplyMerchantsRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_ACTION = "action";
    private final String KEY_ACTION_VALUE = "get_by_parent";
    private final String KEY_ACTION_VALUE_MULTILINGUAL = "retrieve_multilingual";
    private final String KEY_ACTIVE = "active";
    private final String KEY_DEFAULT_SHOWN = "default_shown";
    private final String KEY_LANGUAGE = "language";
    private final String KEY_LOCATION = "location";
    private final String KEY_CATEGORIES = "categories";
    private final String KEY_TAGS = "tags";
    private final String KEY_UUID = "uuid";
    private final String KEY_CENTER = "center";
    private final String KEY_DISTANCE = "distance";
    private final String KEY_PARENT_UUIDS = "parent_uuids";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;
    private boolean mIsMultilingual = false;
    private ArrayList<String> mCategories = new ArrayList<>();
    private boolean mDefaultShown = false;
    private ArrayList<String> mTags = new ArrayList<>();
    private String mUuid = "";
    private int mDistance = 0;
    private ArrayList<Double> mCenter = new ArrayList<>();
    private String mLanguage = WarplyProperty.getLanguage(Warply.getWarplyContext());
    private ArrayList<String> mParentUuids = new ArrayList<>();

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyMerchantsRequest, initializes an empty filters HashMap
     */
    public WarplyMerchantsRequest() {
        mFilters = new HashMap<>();
    }

    public WarplyMerchantsRequest(WarplyMerchantsRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyMerchantsRequest) {
            WarplyMerchantsRequest other = (WarplyMerchantsRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public WarplyMerchantsRequest setParentUuids(ArrayList<String> uuids) {
        mParentUuids = uuids;
        return this;
    }

    public WarplyMerchantsRequest setIsMultilingual(boolean multilingual) {
        mIsMultilingual = multilingual;
        return this;
    }

    public WarplyMerchantsRequest setCategories(ArrayList<String> categories) {
        mCategories = categories;
        return this;
    }

    public WarplyMerchantsRequest setDefault(boolean defaultShown) {
        mDefaultShown = defaultShown;
        return this;
    }

    public WarplyMerchantsRequest setCenter(ArrayList<Double> center) {
        mCenter = center;
        return this;
    }

    public WarplyMerchantsRequest setDistance(int distance) {
        mDistance = distance;
        return this;
    }

    public WarplyMerchantsRequest setTags(ArrayList<String> tags) {
        mTags = tags;
        return this;
    }

    public WarplyMerchantsRequest setMerchantUuid(String uuid) {
        mUuid = uuid;
        return this;
    }

    public WarplyMerchantsRequest setLanguage(String lan) {
        mLanguage = lan;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyMerchantsRequest
     */
    public WarplyMerchantsRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyMerchantsRequest
     */
    public WarplyMerchantsRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            if (mIsMultilingual) {
                bodyJsonObject.putOpt(KEY_ACTION, KEY_ACTION_VALUE_MULTILINGUAL);
                bodyJsonObject.putOpt(KEY_ACTIVE, true);
                bodyJsonObject.putOpt(KEY_CATEGORIES, new JSONArray(mCategories));
                bodyJsonObject.putOpt(KEY_DEFAULT_SHOWN, mDefaultShown ? mDefaultShown : JSONObject.NULL);
                bodyJsonObject.putOpt(KEY_LANGUAGE, mLanguage);
                if (mDistance != 0 && mCenter != null && mCenter.size() > 0) {
                    JSONObject locObj = new JSONObject();
                    locObj.putOpt(KEY_CENTER, mCenter);
                    locObj.putOpt(KEY_DISTANCE, mDistance);
                    bodyJsonObject.putOpt(KEY_LOCATION, locObj);
                    bodyJsonObject.putOpt(KEY_CENTER, mCenter);
                    bodyJsonObject.putOpt(KEY_DISTANCE, mDistance);
                } else
                    bodyJsonObject.putOpt(KEY_LOCATION, JSONObject.NULL);
                if (mParentUuids == null || mParentUuids.size() == 0) {
                    bodyJsonObject.putOpt(KEY_PARENT_UUIDS, JSONObject.NULL);
                } else {
                    bodyJsonObject.putOpt(KEY_PARENT_UUIDS, new JSONArray(mParentUuids));
                }
                bodyJsonObject.putOpt(KEY_TAGS, mTags != null && mTags.size() > 0 ? new JSONArray(mTags) : JSONObject.NULL);
                if (!TextUtils.isEmpty(mUuid))
                    bodyJsonObject.putOpt(KEY_UUID, mUuid);
            } else
                bodyJsonObject.putOpt(KEY_ACTION, KEY_ACTION_VALUE);
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_shops_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
