package ly.warp.sdk.io.request;

import android.text.TextUtils;
import android.util.Base64;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 20-Jan-22.
 */

public class WarplyResetPasswordRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_CHANNEL = "channel";
    private final String KEY_APP_UUID = "app_uuid";
    private final String KEY_CONF_TOKEN = "conf_token";
    private final String KEY_NEW_PASS = "new_password";
    private final String KEY_CONF_CODE = "conf_code";
    private final String KEY_OTP_UUID = "otp_uuid";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;
    private String mConfToken = "";
    private String mNewPassword = "";
    private String mOtp = "";
    private String mOtpUuid = "";

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyResetPasswordRequest, initializes an empty filters HashMap
     */
    public WarplyResetPasswordRequest() {
        mFilters = new HashMap<>();
    }

    public WarplyResetPasswordRequest(WarplyResetPasswordRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyResetPasswordRequest) {
            WarplyResetPasswordRequest other = (WarplyResetPasswordRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public WarplyResetPasswordRequest setPassword(String password) {
        mNewPassword = password;
        return this;
    }

    public WarplyResetPasswordRequest setConfToken(String token) {
        mConfToken = token;
        return this;
    }

    public WarplyResetPasswordRequest setOtp(String otp) {
        mOtp = otp;
        return this;
    }

    public WarplyResetPasswordRequest setOtpUuid(String otpuuid) {
        mOtpUuid = otpuuid;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyResetPasswordRequest
     */
    public WarplyResetPasswordRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyResetPasswordRequest
     */
    public WarplyResetPasswordRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            bodyJsonObject.putOpt(KEY_CHANNEL, "mobile");
            bodyJsonObject.putOpt(KEY_APP_UUID, WarplyProperty.getAppUuid(Warply.getWarplyContext()));
            if (!TextUtils.isEmpty(mConfToken))
                bodyJsonObject.putOpt(KEY_CONF_TOKEN, mConfToken);
            bodyJsonObject.putOpt(KEY_NEW_PASS, mNewPassword);
            if (!TextUtils.isEmpty(mOtp)) {
                bodyJsonObject.putOpt(KEY_CONF_CODE, mOtp);
                bodyJsonObject.putOpt(KEY_OTP_UUID, mOtpUuid);
            }
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_reset_password_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
