package ly.warp.sdk.io.request;

import android.util.Base64;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

import ly.warp.sdk.Warply;
import ly.warp.sdk.utils.WarplyProperty;
import ly.warp.sdk.utils.constants.WarpConstants;

/**
 * Created by Panagiotis Triantafyllou on 15-Dec-21.
 */

public class WarplyTokenAuthorizeRequest {

    // ===========================================================
    // Constants
    // ===========================================================

    private final String KEY_GRANT_TYPE = "grant_type";
    private final String KEY_CODE = "code";
    private final String KEY_CLIENT_ID = "client_id";
    private final String KEY_CLIENT_SECRET = "client_secret";
    private final String KEY_SCOPE = "scope";
    private final String KEY_REDIRECT_URI = "redirect_uri";

    // ===========================================================
    // Fields
    // ===========================================================

    private HashMap<String, String> mFilters;
    private long mCacheUpdateInterval = 0;
    private JSONObject mAuthorizeData = new JSONObject();
    private boolean mIsRegister = false;

    // ===========================================================
    // Constructor
    // ===========================================================

    /**
     * Default constructor of WarplyTokenAuthorizeRequest, initializes an empty filters HashMap
     */
    public WarplyTokenAuthorizeRequest() {
        mFilters = new HashMap<>();
    }

    public WarplyTokenAuthorizeRequest(WarplyTokenAuthorizeRequest copy) {
        if (copy != null) {
            this.mFilters = copy.mFilters;
            this.mCacheUpdateInterval = copy.mCacheUpdateInterval;
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public boolean equals(Object object) {
        if (object instanceof WarplyTokenAuthorizeRequest) {
            WarplyTokenAuthorizeRequest other = (WarplyTokenAuthorizeRequest) object;
            return other == this || (this.mFilters == other.mFilters || (this.mFilters != null && this.mFilters.equals(other.mFilters)));
        }
        return false;
    }

    @Override
    public int hashCode() {
        return mFilters.hashCode();
    }

    // ===========================================================
    // Methods
    // ===========================================================

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    public WarplyTokenAuthorizeRequest setAuthorizeData(JSONObject authorizeData) {
        mAuthorizeData = authorizeData;
        return this;
    }

    public WarplyTokenAuthorizeRequest setIsRegister(boolean isRegisterFlow) {
        mIsRegister = isRegisterFlow;
        return this;
    }

    /**
     * Call this to get how often the cached data will be updated.
     *
     * @return mCacheUpdateInterval
     */
    public long getCacheUpdateInterval() {
        return mCacheUpdateInterval;
    }

    /**
     * Call this to set how often the cached data will be updated.
     *
     * @param updateInterval The time that data will be cached
     * @return WarplyTokenAuthorizeRequest
     */
    public WarplyTokenAuthorizeRequest setCacheUpdateInterval(long updateInterval) {

        this.mCacheUpdateInterval = updateInterval;
        if (mCacheUpdateInterval < 0) {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to check if the Application uses Cache
     *
     * @return <p>true - the Application is using Cache</p>
     * <p>false - the Application is not using Cache</p>
     */
    public boolean isUseCache() {
        return mCacheUpdateInterval > 0;
    }

    /**
     * Call this to check whether the cached data need to be updated
     *
     * @param useCache <p>true - the Application is using Cache</p>
     *                 <p>false - the Application is not using Cache</p>
     * @return WarplyTokenAuthorizeRequest
     */
    public WarplyTokenAuthorizeRequest setUseCache(boolean useCache) {

        if (useCache) {
            mCacheUpdateInterval = mCacheUpdateInterval > 0 ? mCacheUpdateInterval
                    : WarpConstants.INBOX_UPDATE_INTERVAL;
        } else {
            mCacheUpdateInterval = 0;
        }
        return this;
    }

    /**
     * Call this to build the offers Json object
     *
     * @return bodyJsonObject
     */
    public JSONObject toJson() {
        JSONObject bodyJsonObject = new JSONObject();
        try {
            bodyJsonObject.putOpt(KEY_GRANT_TYPE, "authorization_code");
            bodyJsonObject.putOpt(KEY_CODE, mAuthorizeData != null ? mAuthorizeData.optString("code") : "");
            bodyJsonObject.putOpt(KEY_SCOPE, WarplyProperty.getLoginType(Warply.getWarplyContext()) + " app_id");
            bodyJsonObject.putOpt(KEY_REDIRECT_URI, WarplyProperty.getBaseUrl(Warply.getWarplyContext()) + WarpConstants.BASE_URL_AUTH + "authorized");
            if (mIsRegister) {
                JSONObject message = new JSONObject();
                message = mAuthorizeData.optJSONObject("message");
                bodyJsonObject.putOpt(KEY_CLIENT_ID, message != null ? message.optString("client_id") : "");
                bodyJsonObject.putOpt(KEY_CLIENT_SECRET, message != null ? message.optString("client_secret") : "");
            } else {
                bodyJsonObject.putOpt(KEY_CLIENT_ID, mAuthorizeData != null ? mAuthorizeData.optString("client_id") : "");
                bodyJsonObject.putOpt(KEY_CLIENT_SECRET, mAuthorizeData != null ? mAuthorizeData.optString("client_secret") : "");
            }
        } catch (JSONException e) {
            if (WarpConstants.DEBUG)
                e.printStackTrace();
        }
        return bodyJsonObject;
    }

    public String getSignature() {
        String signature = mFilters != null && mFilters.size() > 0 ? String.valueOf(mFilters.hashCode()) : "default_login_request";
        try {
            byte[] hash = MessageDigest.getInstance("SHA-256").digest(signature.getBytes("UTF-8"));
            signature = Base64.encodeToString(hash, Base64.NO_WRAP);
        } catch (NullPointerException | NoSuchAlgorithmException
                | UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return signature;
    }
}
