/*
 * Copyright (C) 2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ly.warp.sdk.io.volley.toolbox;

import android.database.Cursor;

import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.ClientConnectionManager;
import org.apache.http.conn.scheme.PlainSocketFactory;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.conn.scheme.SocketFactory;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.tsccm.ThreadSafeClientConnManager;
import org.apache.http.message.BasicHeader;
import org.apache.http.params.HttpConnectionParams;
import org.apache.http.params.HttpParams;
import org.apache.http.protocol.HTTP;

import java.io.IOException;
import java.net.Socket;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import ly.warp.sdk.Warply;
import ly.warp.sdk.db.WarplyDBHelper;
import ly.warp.sdk.io.volley.AuthFailureError;
import ly.warp.sdk.io.volley.Request;
import ly.warp.sdk.io.volley.Request.Method;
import ly.warp.sdk.utils.constants.WarpConstants;

public class HttpClientStack implements HttpStack {

    // ===========================================================
    // Constants
    // ===========================================================

    private final int CONNECTION_TIMEOUT = 30000;
    private final int SOCKET_TIMEOUT = 30000;

    private final static String SCHEME_NAME_HTTPS = "https";
    private final static int SCHEME_PORT_HTTPS = 443;

    private final static String SCHEME_NAME_HTTP = "http";
    private final static int SCHEME_PORT_HTTP = 80;

    // ===========================================================
    // Fields
    // ===========================================================

    private static HttpClient mClient;

    // ===========================================================
    // Constructors
    // ===========================================================

    public HttpClientStack() {
        if (mClient == null) {
            mClient = getHttpClient();
        }
    }

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public HttpResponse performRequest(Request<?> request,
                                       Map<String, String> additionalHeaders)
            throws IOException, AuthFailureError {

        if (mClient == null) {
            mClient = getHttpClient();
        }
        return mClient.execute(
                createHttpRequest(request, additionalHeaders));
    }

    private HttpUriRequest createHttpRequest(Request<?> request,
                                             Map<String, String> additionalHeaders) throws AuthFailureError {

        List<Header> headers = new ArrayList<Header>();
        Map<String, String> basicHeaders = request.getHeaders();
        if (basicHeaders != null) {
            for (Map.Entry<String, String> header : basicHeaders.entrySet()) {
                headers.add(new BasicHeader(header.getKey(), header.getValue()));
            }
        }

        if (additionalHeaders != null) {
            for (Map.Entry<String, String> header : additionalHeaders.entrySet()) {
                headers.add(new BasicHeader(header.getKey(), header.getValue()));
            }
        }

        switch (request.getMethod()) {
            case Method.GET:

                HttpGet httpGet = new HttpGet(request.getUrl());
                if (headers.size() > 0) {
                    httpGet.setHeaders(headers.toArray(new Header[headers.size()]));
                }
                return httpGet;

            case Method.DELETE:

                HttpDelete httpDelete = new HttpDelete(request.getUrl());
                if (headers.size() > 0) {
                    httpDelete.setHeaders(headers.toArray(new Header[headers.size()]));
                }
                return httpDelete;

            case Method.POST:

                HttpPost httpPost = new HttpPost(request.getUrl());
                headers.add(new BasicHeader(HTTP.CONTENT_TYPE, request.getBodyContentType()));
                if (request.getTag() != null && request.getTag().equals("true")) {
                    headers.add(new BasicHeader(WarpConstants.HEADER_AUTHORIZATION,
                            "Bearer " + WarplyDBHelper.getInstance(Warply.getWarplyContext()).getAuthValue("access_token")));
                } else if (request.getTag() != null && request.getTag().equals("cosuser")) {
                    // prod MVBQNFhCQzhFYTJBaUdCNkJWZGFGUERlTTNLQ3kzMjU6YzViMzAyZDY5N2FiNGY3NzhiNThhMTg0YzBkZWRmNGU=
                    // dev MWlTM0EyNjcxT2Q0a1B5QkIydEs1ZU5uRENhR0NWQjQ6MjI4MjA4ZTliMTQzNGQ2MmIxNGI3ZDAzYjM2ZjUwMzg=
                    headers.add(new BasicHeader(WarpConstants.HEADER_AUTHORIZATION,
                            "Basic MVBQNFhCQzhFYTJBaUdCNkJWZGFGUERlTTNLQ3kzMjU6YzViMzAyZDY5N2FiNGY3NzhiNThhMTg0YzBkZWRmNGU="));
                }
                httpPost.setHeaders(headers.toArray(new Header[headers.size()]));
                byte[] body = request.getBody();
                if (body != null) {
                    ByteArrayEntity entity = new ByteArrayEntity(body);
                    entity.setContentType(new BasicHeader(HTTP.CONTENT_TYPE,
                            request.getBodyContentType()));
                    httpPost.setEntity(entity);
                }
                return httpPost;

            case Method.PUT:

                HttpPut httpPut = new HttpPut(request.getUrl());
                headers.add(new BasicHeader(HTTP.CONTENT_TYPE,
                        request.getBodyContentType()));
                httpPut.setHeaders(headers.toArray(new Header[headers.size()]));
                body = request.getBody();
                if (body != null) {
                    ByteArrayEntity entity = new ByteArrayEntity(body);
                    entity.setContentType(new BasicHeader(HTTP.CONTENT_TYPE,
                            request.getBodyContentType()));
                    httpPut.setEntity(entity);
                }
                return httpPut;

            default:
                throw new IllegalStateException("Unknown request method.");
        }
    }

    // ===========================================================
    // Methods
    // ===========================================================

    public HttpClient getHttpClient() {

        return new DefaultHttpClient() {
            @Override
            protected ClientConnectionManager createClientConnectionManager() {

                SchemeRegistry registry = new SchemeRegistry();
                registry.register(new Scheme(SCHEME_NAME_HTTPS, getHttpsSocketFactory(),
                        SCHEME_PORT_HTTPS));
                registry.register(new Scheme(SCHEME_NAME_HTTP, PlainSocketFactory
                        .getSocketFactory(), SCHEME_PORT_HTTP));
                HttpParams params = getParams();
                HttpConnectionParams.setConnectionTimeout(params,
                        CONNECTION_TIMEOUT);
                HttpConnectionParams.setSoTimeout(params, SOCKET_TIMEOUT);
                return new ThreadSafeClientConnManager(params, registry);
            }

            /**
             * Gets an HTTPS socket factory with SSL Session Caching if such
             * support is available, otherwise falls back to a non-caching
             * factory
             *
             * @return SocketFactory
             */
            protected SocketFactory getHttpsSocketFactory() {

                try {

                    SSLSocketFactory sf = new TLSSocketFactory();
                    sf.setHostnameVerifier(SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
                    return sf;

                } catch (Exception e) {
                    if (WarpConstants.DEBUG)
                        e.printStackTrace();
                    return SSLSocketFactory.getSocketFactory();
                }
            }
        };
    }

// ===========================================================
// Inner and Anonymous Classes
// ===========================================================

    private class TLSSocketFactory extends SSLSocketFactory {

        SSLContext sslContext = SSLContext.getInstance("TLS");

        public TLSSocketFactory() throws NoSuchAlgorithmException, KeyManagementException, KeyStoreException, UnrecoverableKeyException {
            super(null);

            TrustManager trustManager = new X509TrustManager() {
                public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                }

                public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
                }

                public X509Certificate[] getAcceptedIssuers() {
                    return null;
                }
            };

            sslContext.init(null, new TrustManager[]{trustManager}, null);
        }

        @Override
        public Socket createSocket(Socket socket, String host, int port, boolean autoClose) throws IOException {
            return sslContext.getSocketFactory().createSocket(socket, host, port, autoClose);
        }

        @Override
        public Socket createSocket() throws IOException {
            return sslContext.getSocketFactory().createSocket();
        }
    }

}
