package ly.warp.sdk.receivers;

import android.app.Application;
import android.content.ComponentName;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import androidx.annotation.Nullable;

import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.Region;
import org.altbeacon.beacon.powersave.BackgroundPowerSaver;
import org.altbeacon.beacon.startup.BootstrapNotifier;
import org.altbeacon.beacon.startup.RegionBootstrap;

import java.util.List;

import ly.warp.sdk.Warply;
import ly.warp.sdk.services.WarplyBeaconsRangingService;
import ly.warp.sdk.utils.WarpUtils;
import ly.warp.sdk.utils.WarplyPreferences;
import ly.warp.sdk.utils.constants.WarpConstants;
import ly.warp.sdk.utils.managers.WarplyServerPreferencesManager;

/**
 * extends for adding beacon functionality to app
 */
public class WarplyBeaconsApplication extends Application implements BootstrapNotifier {

    // ===========================================================
    // Constants
    // ===========================================================

    public static final String TAG = "WarplyBeacons: ";

    private static final String[] DEFAULT_BEACON_LAYOUTS = {
            "m:2-3=aabb,i:4-19,i:20-21,i:22-23,p:24-24,d:25-25",
            "m:0-3=0215,i:4-19,i:20-21,i:22-23,p:24-24",
            "m:2-3=0215,i:4-19,i:20-21,i:22-23,p:24-24,d:25-25"
    };
    public static final int DEFAULT_BEACON_DISTANCE_TO_SEND = 20;
    public static final long DEFAULT_BEACON_TIME_INTERVAL_TO_RESEND = 30 * 1000;
    public static final long DEFAULT_SCAN_PERIOD = 5 * 1000;

    private static WarplyPreferences mWarplyPref;

    // ===========================================================
    // Fields
    // ===========================================================

    private static WarplyBeaconsApplication mInstance = null;

    private RegionBootstrap mBootstrap;
    private Region mRegion;
    private BackgroundPowerSaver mBackgroundPowerSaver;
    private WarplyBeaconsRangingService mRangingService;

    private boolean isEnabled = false;

    // ===========================================================
    // Constructors
    // ===========================================================

    // ===========================================================
    // Methods for/from SuperClass/Interfaces
    // ===========================================================

    @Override
    public void onCreate() {
        super.onCreate();

        mInstance = this;
        mWarplyPref = new WarplyPreferences(mInstance);

        Warply.getInitializer(getInstance()).init();
        if (getInstance() != null) {
            getInstance().enable();
        }

        WarpUtils.log(TAG + "onCreate WarplyBeaconsApplication");
    }

    @Override
    public void didEnterRegion(Region region) {

        if (isEnabled) {
            startBindRangingService();
        }
        WarpUtils.log(TAG + "didEnterRegion");
    }

    @Override
    public void didExitRegion(Region region) {

        if (isEnabled) {
            stopUnBindRangingService();
        }

        WarpUtils.log(TAG + "didExitRegion");
    }

    @Override
    public void didDetermineStateForRegion(int i, Region region) {
        WarpUtils.log(TAG + "didDetermineStateForRegion");
    }

    // ===========================================================
    // Methods
    // ===========================================================

    private void initBeaconLayouts(String[] layouts) {

        if (layouts != null && layouts.length > 0) {
            BeaconManager beaconManager = BeaconManager.getInstanceForApplication(getInstance());
            if (beaconManager.isAnyConsumerBound()) {
                if (mBootstrap != null && getInstance() != null) {
                    mBootstrap.disable();
                    if (mRangingService != null) {
                        mRangingService.stopBeaconListener();
                    }

                    if (!beaconManager.isAnyConsumerBound()) {
                        List<BeaconParser> parsers = beaconManager.getBeaconParsers();
                        for (String layout : layouts) {
                            parsers.add(new BeaconParser().setBeaconLayout(layout));
                        }
                    }

                    mBootstrap = new RegionBootstrap(getInstance(), mRegion);
                    if (mRangingService != null) {
                        mRangingService.startBeaconListener();
                    }
                }
            } else {
                List<BeaconParser> parsers = beaconManager.getBeaconParsers();
                for (String layout : layouts) {
                    parsers.add(new BeaconParser().setBeaconLayout(layout));
                }
            }
        }
    }

    private boolean checkAvailability() {
        try {
            return BeaconManager.getInstanceForApplication(this).
                    checkAvailability();
        } catch (RuntimeException e) {
            return false;
        }
    }

    private void startBindRangingService() {
//        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
//            startForegroundService(new Intent(this, WarplyBeaconsRangingService.class));
//        } else {
            startService(new Intent(this, WarplyBeaconsRangingService.class));
//        }
        bindService(new Intent(this, WarplyBeaconsRangingService.class),
                mBeaconsRangingServiceConnection, BIND_AUTO_CREATE);
    }

    private void stopUnBindRangingService() {
        if (mRangingService != null) {
            unbindService(mBeaconsRangingServiceConnection);
        }
        stopService(new Intent(this, WarplyBeaconsRangingService.class));
    }

    // ===========================================================
    // Getter & Setter
    // ===========================================================

    /**
     * @return the {@link WarplyBeaconsApplication} instance
     */
    @Nullable
    public static synchronized WarplyBeaconsApplication getInstance() {
        return mInstance;
    }

    /**
     * Checks if Bluetooth is available from the device AND if beacons are enabled for this app
     * from Warply and enables the app to launch as soon as a beacon is seen
     */
    public void enable() {

        if (checkAvailability() && !isEnabled && isMicroAppEnabled() && getInstance() != null) {

            // By default the AndroidBeaconLibrary will only find AltBeacons.  If you wish to make it
            // find a different type of beacon, you must specify the byte layout for that beacon's
            // advertisement with a line like below.  The example shows how to find a beacon with the
            // same byte layout as AltBeacon but with a beaconTypeCode of 0xaabb.  To find the proper
            // layout expression for other beacon types, do a web search for "setBeaconLayout"
            // including the quotes.
            //
            // beaconManager.getBeaconParsers().add(new BeaconParser().
            //        setBeaconLayout("m:2-3=aabb,i:4-19,i:20-21,i:22-23,p:24-24,d:25-25"));
            //
            // add default layouts
            initBeaconLayouts(getBeaconLayouts());

            // wake up the app when a beacon is seen
            mRegion = new Region("BootStrapRegion", null, null, null);
            mBootstrap = new RegionBootstrap(getInstance(), mRegion);

            // simply constructing this class and holding a reference to it in your custom Application
            // class will automatically cause the BeaconLibrary to save battery whenever the application
            // is not visible.  This reduces bluetooth power usage by about 60%
            mBackgroundPowerSaver = new BackgroundPowerSaver(getInstance());

            isEnabled = !isEnabled;

            WarpUtils.log(TAG + "setting up beacons monitoring");
        }
    }

    /**
     * disables the app to launch and listen to transmitting beacons
     */
    public void disable() {

        if (isEnabled) {
            if (mBootstrap != null) {
                mBootstrap.disable();
            }

            stopUnBindRangingService();
            isEnabled = !isEnabled;
        }
    }

    /**
     * we can add layouts only one time per app start
     * while any consumer still not connected to BeaconManager
     *
     * @param layouts the layouts used to catch beacons' advertisements
     */
    public void addBeaconLayouts(String[] layouts) {
        if (isEnabled) {
            initBeaconLayouts(layouts);
        }
    }

    public boolean isEnabled() {
        return isEnabled;
    }

    /**
     * @return true if beacons feature is enabled for this app from Warply
     * false otherwise
     */
    public static boolean isMicroAppEnabled() {
        return Warply.isInitialized() && WarplyServerPreferencesManager.
                isMicroAppActive(WarpConstants.MicroApp.BEACONS);
    }

    /**
     * @return the layouts used to catch beacons' advertisements
     */
    public static String[] getBeaconLayouts() {
        return mWarplyPref == null ? DEFAULT_BEACON_LAYOUTS : mWarplyPref.
                getStringArray(WarpConstants.KEY_APP_VAR_BEACON_LAYOUTS, DEFAULT_BEACON_LAYOUTS);
    }

    /**
     * @return the minimum distance at which the app will notify Warply that a beacon was found
     */
    public static int getBeaconDistanceToSendWarply() {
        return mWarplyPref == null ? DEFAULT_BEACON_DISTANCE_TO_SEND : mWarplyPref.
                getInt(WarpConstants.KEY_APP_VAR_BEACON_MAX_DISTANCE, DEFAULT_BEACON_DISTANCE_TO_SEND);
    }

    /**
     * @return the minimum time interval at which the app will notify Warply that a beacon was found
     */
    public static long getBeaconTimeIntervalToSendWarply() {
        return mWarplyPref == null ? DEFAULT_BEACON_TIME_INTERVAL_TO_RESEND : mWarplyPref.
                getLong(WarpConstants.KEY_APP_VAR_BEACON_TIME_INTERVAL_TO_RESEND, DEFAULT_BEACON_TIME_INTERVAL_TO_RESEND);
    }

    /**
     * @return the time period in which the app scans for beacons
     */
    public static long getBeaconScanPeriod() {
        return mWarplyPref == null ? DEFAULT_SCAN_PERIOD : mWarplyPref.
                getLong(WarpConstants.KEY_APP_VAR_BEACON_TIME_INTERVAL, DEFAULT_SCAN_PERIOD);
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    private ServiceConnection mBeaconsRangingServiceConnection = new ServiceConnection() {

        @Override
        public void onServiceConnected(ComponentName className, IBinder service) {
            mRangingService = ((WarplyBeaconsRangingService.BeaconsRangingServiceBinder) service).getService();
        }

        @Override
        public void onServiceDisconnected(ComponentName arg0) {
            mRangingService = null;
        }
    };

}
